from __future__ import absolute_import

import base64
import os
import struct
import sys
import time
import urlparse

import subprocess as subproc
import Queue
import msgpack
import threading

from api.srvmngr import getRoot
import api.copier
import api.copier.errors

from . import BaseTransport
from ...common import determine_app_path
from ...rbtorrent import logger as logging
from ...rpc.client import RPCClient

from kernel.util.functional import singleton


TRANSPORT_MAGIC = 'rbtorrent'


@singleton
def log():
    return logging.initialize_client()


@singleton
def _slave_mode():
    root = getRoot()
    ST_RDONLY = 1  # noqa
    if os.path.exists(root):
        if os.statvfs(root).f_flag & ST_RDONLY:
            return True


def _get_default_subproc(mode):
    from library.config import query
    try:
        return getattr(query('skynet.services.copier').opts, mode).subproc
    except AttributeError:
        return False


def _get_deduplicate_nocheck():
    return False

    from library.config import query
    try:
        return query('skynet.services.copier').deduplicate_nocheck
    except AttributeError:
        return False


class CreateResult(object):
    def __init__(self, resid):
        if isinstance(resid, dict):
            self._resid = '%s:%s' % (TRANSPORT_MAGIC, resid['uid'])
            self._sign_key = resid.get('sign_key', None)
            self._sign_key_nid = resid.get('sign_key_nid', None)
        else:
            self._resid = '%s:%s' % (TRANSPORT_MAGIC, resid)
            self._sign_key = None
            self._sign_key_nid = None

    def resid(self):
        return self._resid

    def key(self):
        if not self._sign_key or not self._sign_key_nid:
            return

        return '%s:%s' % (self._sign_key_nid, base64.encodestring(self._sign_key).replace('\n', ''))


class Handle(object):
    def __init__(self, transport, resid):
        self.transport = transport
        self.resid = resid

    def get(
        self, dest, user,
        priority, network, deduplicate,
        max_dl_speed, max_ul_speed,
        **extra
    ):
        return self.transport.get(
            self.resid, dest, user,
            priority, network, deduplicate,
            max_dl_speed, max_ul_speed,
            **extra
        )

    def list(self, priority, network, **extra):
        return self.transport.list(self.resid, priority, network, **extra)


class StateV1(object):
    class Stage(object):
        stage = 'unknown'

        def __init__(self, *args):
            for idx, arg in enumerate(args):
                setattr(self, self.__slots__[idx], arg)

        def __str__(self):
            return '<Stage %s: %r>' % (
                self.stage,
                dict([
                    (n, getattr(self, n)) for n in self.__slots__
                ])
            )

    class Connecting(Stage):
        stage = 'connecting'
        __slots__ = ()

    class GetResource(Stage):
        stage = 'get_resource'
        __slots__ = ('total_bytes', 'done_bytes', 'ul_bytes', 'extra')

    class GetResourceHead(Stage):
        stage = 'get_resource_head'
        __slots__ = ('done', )

    class LockPaths(Stage):
        stage = 'lock_paths'
        __slots__ = ('total_count', 'locked_count')

    class Hashing(Stage):
        stage = 'hashing'
        __slots__ = ('total_bytes', 'hashed_bytes')

    class SpawnDownloader(Stage):
        stage = 'spawn_downloader'
        __slots__ = ('done', )

    class SubprocStarted(Stage):
        stage = 'subproc_started'
        __slots__ = ()

    class SubprocConnectMaster(Stage):
        stage = 'subproc_connect_master'
        __slots__ = ('done', )

    def __call__(self, state):
        try:
            if state['stage'] == 'connecting':
                state = self.Connecting()
            elif state['stage'] == 'get_resource':
                state = self.GetResource(
                    state['total_bytes'],
                    state['done_bytes'],
                    state['ul_bytes'],
                    state['state']
                )
            elif state['stage'] == 'get_resource_head':
                state = self.GetResourceHead(
                    state['done'],
                )
            elif state['stage'] == 'lock_paths':
                state = self.LockPaths(
                    state['total_count'], state['locked_count']
                )
            elif state['stage'] == 'spawn_downloader':
                state = self.SpawnDownloader(
                    state['done']
                )
            elif state['stage'] == 'hashing':
                state = self.Hashing(
                    state['total_bytes'],
                    state['hashed_bytes']
                )
            elif state['stage'] == 'subproc_started':
                state = self.SubprocStarted()
            elif state['stage'] == 'subproc_connect_master':
                state = self.SubprocConnectMaster(state['done'])
            else:
                return
        except Exception as ex:
            log().error('Unable to convert progress: %s (progress was %r)', ex, state)
            return

        return state


class SubprocJob(object):
    def __init__(self):
        self.msgqueue = Queue.Queue(maxsize=4096)
        self.proc = None

        self.__result_ready = False
        self.__result = None

    def _priority_to_int(self, priority):
        return {
            'Idle': -3,
            'Low': -2,
            'BelowNormal': -1,
            'Normal': 0,
            'AboveNormal': 1,
            'High': 2,
            'RealTime': 3
        }.get(priority, 0)

    def _msgqueue_worker(self, stream):
        reason = None

        try:
            while True:
                magic = stream.read(1)
                if not magic:
                    break

                if magic == '\xf2':
                    size_packed = stream.read(4)
                    if len(size_packed) < 4:
                        break

                    size = struct.unpack('!I', size_packed)[0]

                    buf = []
                    read = 0

                    while read < size:
                        data = stream.read(size - read)
                        if not data:
                            break

                        buf.append(data)
                        read += len(data)

                    if read != size:
                        break

                    unpacked = msgpack.loads(''.join(buf))

                    if self.msgqueue.qsize() >= 4096:
                        # throw out oldest message
                        self.msgqueue.get()

                    self.msgqueue.put(unpacked)
                else:
                    self.msgqueue.put(('stream', magic + stream.readline()))

        except Exception as ex:
            reason = ex

        finally:
            ret = -1

            try:
                # Attempt to kill process if we died somehow here
                ret = self.proc.poll()
                if ret is None:
                    self.proc.kill()
                    self.proc.wait()
                    ret = self.proc.returncode

                    try:
                        stream.read()
                    except Exception:
                        pass
            finally:
                result = {'code': ret, 'reason': None}

                if reason:
                    result['reason'] = reason

                self.msgqueue.put(('eof', result))

    def _runproc_via_master(self, api_sock, cmd, stdin, *args):
        rpc = RPCClient(api_sock, None)

        tries = 10

        for i in range(tries):
            try:
                try:
                    rpc.connect()
                except:
                    if i == tries - 1:
                        raise api.copier.errors.ApiConnectionError('Unable to connect copier service')

                job = rpc.call('runjob', str(cmd), stdin, *args)

                stdout_buff = ''
                parsed_wait = 0  # num of bytes after receiving magic

                for typ, kind, data in job:
                    assert typ == 'stream'
                    if kind == 'stderr':
                        sys.stderr.write(data)
                        continue
                    elif kind == 'stdout':
                        stdout_buff += data

                    wait_more = False

                    while not wait_more:
                        if parsed_wait == 0:
                            if len(stdout_buff) <= 5:
                                # wait more
                                wait_more = True
                                continue

                            if stdout_buff[0] == '\xf2':
                                size_packed = stdout_buff[1:5]
                                size = struct.unpack('!I', size_packed)[0]
                                parsed_wait = size
                                stdout_buff = stdout_buff[5:]

                        if parsed_wait > 0:
                            if len(stdout_buff) >= parsed_wait:
                                data = stdout_buff[:parsed_wait]
                                stdout_buff = stdout_buff[parsed_wait:]
                                parsed_wait = 0

                                unpacked = msgpack.loads(data)

                                if self.msgqueue.qsize() >= 4096:
                                    # throw out oldest message
                                    self.msgqueue.get()

                                self.msgqueue.put(unpacked)
                            else:
                                wait_more = True

                try:
                    ret = job.wait()
                finally:
                    result = {'code': ret, 'reason': None}
                    self.msgqueue.put(('eof', result))
                    return

            except api.copier.errors.CopierError:
                self.msgqueue.put(('eof', {'code': 1, 'reason': 'Unable to connect copier service'}))

            except KeyboardInterrupt:
                raise

            except:
                # 90 sec total sleep
                time.sleep(i * 2)
                continue

            break

        self.msgqueue.put(('eof', {'code': 1, 'reason': '10 tries'}))

    def _runproc(self, cmd, stdin, mimic_subproc, *args):
        if mimic_subproc:
            # Weird hack mode
            # We ask copier daemon to run process for us and forward stdin/out/err and return code
            assert 'api_sock' in mimic_subproc

            thr = threading.Thread(
                target=self._runproc_via_master,
                args=(
                    mimic_subproc['api_sock'], cmd, stdin
                ) + args
            )
            thr.daemon = True
            thr.start()
            return

        self.proc = subproc.Popen([str(cmd)] + list(args), stdout=subproc.PIPE, stdin=subproc.PIPE, close_fds=True)
        if stdin:
            self.proc.stdin.write(stdin)
        self.proc.stdin.close()

        thr = threading.Thread(target=self._msgqueue_worker, args=(self.proc.stdout, ))
        thr.daemon = True
        thr.start()

        return self.proc

    def _prociter(self, deadline=None):
        if self.__result_ready:
            return

        timedout = False

        if deadline:
            timeout_seconds = deadline - time.time()
        else:
            timeout_seconds = None

        while True:
            try:
                if deadline:
                    timeout = deadline - time.time()
                    if timeout <= 0:
                        raise Queue.Empty()
                else:
                    timeout = None

                # If we have no timeout -- cycle thru 1 sec timeouts to avoid thread
                # blocking all signals
                if timeout is None:
                    while True:
                        try:
                            msg = self.msgqueue.get(timeout=1)
                        except Queue.Empty:
                            continue
                        else:
                            break
                else:
                    msg = self.msgqueue.get(timeout=timeout)

                if msg[0] == 'eof':
                    if timedout:
                        raise api.copier.errors.Timeout(
                            '%s: process timed out (%.2f seconds)' % (self.procname.basename, timeout_seconds, )
                        )

                    if msg[1]['reason']:
                        raise api.copier.errors.ApiError(
                            '%s: process exited with error %r' % (self.procname.basename, msg[1]['reason'])
                        )

                    if msg[1]['code']:
                        raise api.copier.errors.ApiError(
                            '%s: process exited with code %r' % (self.procname.basename, msg[1]['code'])
                        )

                    if self.__result is None:
                        raise api.copier.errors.ApiError(
                            '%s: process exited without result (msg: %r)' % (self.procname.basename, msg)
                        )

                    self.__result_ready = True
                    return

                if msg[0] == 'result':
                    self.__result = msg[1]

                if msg[0] == 'error':
                    if isinstance(msg[1], (list, tuple)):
                        try:
                            ex = getattr(__import__(msg[1][0], fromlist=['']), msg[1][1])
                            exargs = msg[1][2]
                        except:
                            ex = api.copier.errors.CopierError
                            exargs = str(msg[1])

                        raise ex(exargs)
                    else:
                        raise api.copier.errors.CopierError('%s' % (msg[1], ))

                yield msg
            except Queue.Empty:
                # Timeout occurred
                deadline = None
                timedout = True
                self.stop()
                assert timedout
                raise api.copier.errors.Timeout(
                    '%s: process timed out (%.2f seconds)' % (self.procname.basename, timeout_seconds, )
                )

    def stop(self):
        if self.proc:
            self.proc.kill()
            self.proc.wait()
            self.proc = None

    def ready(self):
        return self.__result_ready

    def iter(self, timeout=None, state_version=1):
        if state_version == 1:
            state_converter = StateV1()
        else:
            state_converter = None

        if timeout is not None:
            deadline = time.time() + timeout
        else:
            deadline = None

        for procmsg in self._prociter(deadline=deadline):
            if procmsg[0] == 'progress':
                if state_converter:
                    state = state_converter(procmsg[1])
                else:
                    state = procmsg[1]

                if state is not None:
                    yield state


class CreateJob(SubprocJob):
    def __init__(self, master_uds, files, **extra):
        super(CreateJob, self).__init__()

        import yaml

        mimic_subproc = not extra.get('subproc', False)

        if mimic_subproc:
            mimic_subproc = {'api_sock': master_uds}

        sharer = determine_app_path().join('bin', 'skybone-sh')
        self.procname = sharer
        self.proc = self._runproc(
            self.procname,
            msgpack.dumps({'files': files}),
            mimic_subproc,
            '--child',
            '--master-uds', master_uds,
            '--extra', yaml.dump(extra)
        )

    def wait(self, timeout=None):
        if not self.ready():
            for _ in self.iter(timeout=timeout):
                pass

        assert self.ready()
        return CreateResult(self._SubprocJob__result)


class ListJob(SubprocJob):
    def __init__(
        self, master_uds, resid, priority, network, **extra
    ):
        super(ListJob, self).__init__()

        import yaml

        getter = determine_app_path().join('bin', 'skybone-dl')
        self.procname = getter

        mimic_subproc = not extra.get('subproc', False)

        if mimic_subproc:
            mimic_subproc = {'api_sock': master_uds}

        self.proc = self._runproc(
            self.procname, None, mimic_subproc,
            '--child',
            '--master-uds', master_uds,
            '--head',
            '--priority', str(self._priority_to_int(priority)),
            '--priority-hint', priority,
            '--network', network,
            '--deduplicate', 'No',
            '--max-dl-speed', '0',
            '--max-ul-speed', '0',
            '--tries', '0',
            '--extra', yaml.dump(extra),
            resid, '/'
        )

    def wait(self, timeout=None):
        if not self.ready():
            for _ in self.iter(timeout=timeout):
                pass

        assert self.ready()
        result = self._SubprocJob__result
        assert isinstance(result, dict), 'Invalid resource type %r' % (result, )

        files_list = []

        for name, info in sorted(result.iteritems()):
            info['name'] = name
            if 'md5sum' in info:
                info['md5sum'] = info['md5sum'].encode('hex')

            for key in info.keys():
                if key not in ('executable', 'name', 'md5sum', 'type', 'size'):
                    info.pop(key, None)

            files_list.append(info)

        return files_list


class GetJob(SubprocJob):
    def __init__(
        self, master_uds, resid, dest,
        priority, network, deduplicate,
        max_dl_speed, max_ul_speed,
        **extra
    ):
        super(GetJob, self).__init__()

        import yaml

        getter = determine_app_path().join('bin', 'skybone-dl')
        self.procname = getter

        # If we have no direct subproc, mimic it
        mimic_subproc = not extra.get('subproc', False)

        if mimic_subproc:
            mimic_subproc = {'api_sock': master_uds}

        self.proc = self._runproc(
            self.procname, None, mimic_subproc,
            '--child',
            '--master-uds', master_uds,
            '--priority', str(self._priority_to_int(priority)),
            '--priority-hint', priority,
            '--network', network,
            '--deduplicate', deduplicate,
            '--max-dl-speed', str(max_dl_speed) if max_dl_speed else '-1',
            '--max-ul-speed', str(max_ul_speed) if max_ul_speed else '-1',
            '--tries', '0',
            '--extra', yaml.dump(extra),
            resid, dest,
        )

    def wait(self, timeout=None):
        if not self.ready():
            for _ in self.iter(timeout=timeout):
                pass

        assert self.ready()
        result = self._SubprocJob__result
        assert isinstance(result, (list, tuple)), 'Invalid resource type %r' % (result, )

        for info in result:
            for key in info.keys():
                if key not in ('executable', 'name', 'path', 'md5sum', 'type', 'size'):
                    info.pop(key, None)

        return result


class TransportRBTorrent(BaseTransport):
    """
    RBTorrent transport. Robust resource downloading over torrent network.
    As <resid> acccepts rbtorrent hash which can be obtained via `sky share` command.
    """
    def handle(self, resid):
        return Handle(self, resid)

    @staticmethod
    def _parse_res_url(res_url):
        path = None
        query = None
        params = {}

        if res_url.startswith("//"):
            (_, _, path, _, query, _) = urlparse.urlparse(res_url[2:])
        elif '?' in res_url:
            (_, _, path, _, query, _) = urlparse.urlparse(res_url)

        if query:
            params = urlparse.parse_qs(query, True)

        if not path:
            path = res_url

        if len(path) != 40:
            raise api.copier.errors.ApiError('Invalid resource uid %r: %d bytes instead of 40' % (
                path, len(path)
            ))

        if params:
            for key, value in params.iteritems():
                if len(value) == 1:
                    value = value[0]
                    try:
                        value = int(value)
                    except:
                        pass
                    params[key] = value
                else:
                    for idx, subvalue in enumerate(value):
                        try:
                            value[idx] = int(subvalue)
                        except:
                            pass

        return path, params

    def detect_real_subproc(self, extra, typ):
        if 'subproc' not in extra:
            if _slave_mode():
                # Always enable real subproc mode in subcontainers
                extra['subproc'] = True
            else:
                # Look in config for default subproc value
                extra['subproc'] = _get_default_subproc(typ)

    def create(self, files, **extra):
        if not files:
            raise api.copier.errors.ApiError('You cant make resource with no contents (no files nor directories)')

        assert 'api_sock' in extra, 'Unable to find skybone api socket path'

        self.detect_real_subproc(extra, 'share')

        job = CreateJob(extra['api_sock'], files, **extra)
        return job

    def get(self, resid, dest, user, priority, network, deduplicate, max_dl_speed, max_ul_speed, **extra):
        infohash, params = self._parse_res_url(resid)
        assert 'api_sock' in extra, 'Unable to find skybone api socket path'

        if params:
            params.update(extra)
            extra = params

        self.detect_real_subproc(extra, 'download')

        if deduplicate == 'Hardlink':
            if _get_deduplicate_nocheck():
                deduplicate = 'HardlinkNocheck'

        job = GetJob(
            extra['api_sock'], infohash, dest,
            priority=priority,
            network=network,
            deduplicate=deduplicate,
            max_dl_speed=max_dl_speed,
            max_ul_speed=max_ul_speed,
            **extra
        )
        return job

    def list(self, resid, priority, network, **extra):
        infohash, params = self._parse_res_url(resid)
        assert 'api_sock' in extra, 'Unable to find skybone api socket path'

        if params:
            params.update(extra)
            extra = params

        self.detect_real_subproc(extra, 'list')

        job = ListJob(
            extra['api_sock'],
            infohash,
            priority=priority,
            network=network,
            head=True,
            **extra
        )
        return job


class TransportFactoryRBTorrent(object):
    """
    Factory for torrent (rb_libtorrent) based transports
    """
    def getMagic(self):  # noqa
        return [TRANSPORT_MAGIC]

    def checkMagic(self, magic):  # noqa
        return magic in self.getMagic()

    def create(self):
        return TransportRBTorrent()

    def ops(self):
        return 'api_sock', 'priority', 'network', 'deduplicate', 'max_dl_speed', 'max_ul_speed', 'kw'
