# coding: utf-8
from __future__ import absolute_import

import subprocess
import time
import os
import StringIO

from api.copier.errors import ResourceError

from . import BaseTransport


class TransportRsyncError(ResourceError):
    '''
    Rsync transport errors.
    '''
    pass


class RsyncBase(object):
    '''
    XXX: This transport doesn't support async download.
    wait must always be called to correct finalization.
    '''
    def __init__(self, resid, network, flags, args):
        try:
            from ya.skynet.services.copier.client.utils import fastbonizeURL
        except ImportError:
            from ya.skynet.services.skybone.client.utils import fastbonizeURL

        self._resid = resid
        self._result = None
        self._result_ready = False

        if network in ('Auto', 'Fastbone'):
            url = fastbonizeURL(resid)
        else:
            url = resid

        self.cmd = ['rsync'] + flags + [url] + args
        try:
            self.downloader = subprocess.Popen(
                self.cmd, stdout=subprocess.PIPE, stderr=subprocess.PIPE, close_fds=True)
        except Exception as e:
            raise TransportRsyncError("Error while running rsync: %s" % e)

    def resid(self):
        return self._resid

    def iter(self, timeout=None, state_version=1):
        # this transport doesn't work in async mode
        if timeout == 0:
            self.downloader.terminate()
            raise TransportRsyncError("Async download is not supported")
        if timeout:
            start = time.time()
            while self.downloader.poll() is None:
                if time.time() - start > timeout:
                    self.downloader.terminate()
                    raise TransportRsyncError("Timeout %s exceeded" % timeout)
                time.sleep(0.1)
            out = self.downloader.stdout.read()
            err = self.downloader.stderr.read()
        else:
            try:
                out, err = self.downloader.communicate()
            except Exception as e:
                raise TransportRsyncError("Error while running rsync: %s" % e)

        if self.downloader.returncode:
            raise TransportRsyncError("Rsync transport error:\n%s" % err)

        self._result = out
        self._result_ready = True
        return
        yield  # force this meth to be a generator

    def wait(self, timeout=None):
        if self._result_ready:
            return self._result

        for _ in self.iter(timeout=timeout):
            pass

        assert self._result_ready
        return self._result


class RsyncGet(RsyncBase):
    def __init__(self, resid, dest, user, network):
        try:
            rshelp = subprocess.check_output(['rsync', '--help'], stderr=subprocess.STDOUT)
        except (subprocess.CalledProcessError, OSError):
            rshelp = ''

        options = []
        if '--append-verify' in rshelp:
            options.append('--append-verify')
        if '--contimeout' in rshelp:
            options.append('--contimeout=10')

        super(RsyncGet, self).__init__(
            resid, network,
            [
                '--append',
                '--checksum',
                '--inplace',
                '--copy-links',
                '--timeout=60',
                '--temp-dir=/var/tmp',
                '--quiet',
                '--recursive',
                '--no-group',
                '--no-perms',
                '--executability',
                '--chmod=Da+rwx,Fa+rwX',
            ] + options,
            [dest]
        )
        self.__dest = dest
        self.__umask = None if user else os.umask(0)

    def __del__(self):
        if self.__umask is not None:
            os.umask(self.__umask)


class RsyncList(RsyncBase):
    def __init__(self, resid, network):
        super(RsyncList, self).__init__(resid, network, ['--list-only', '--timeout=60', '--recursive'], [])

    def wait(self, timeout=None):
        out = super(RsyncList, self).wait(timeout)
        items = []
        for line in StringIO.StringIO(out):
            try:
                mode, size, date, _, relative_path = line.split()
                object_type = 'file'
                if mode.startswith('d'):
                    object_type = 'dir'
                executable = False
                if mode[3] == 'x' or mode[6] == 'x' or mode[9] == 'x':
                    executable = True
                items.append({
                    'name': relative_path,
                    'type': object_type,
                    'size': int(size),
                    'executable': executable,
                })
            except:
                continue
        return sorted(items, key=lambda x: x['name'])


class RsyncHandle(object):
    def __init__(self, resid):
        self.__resid = resid

    def resid(self):
        return self.__resid

    def list(self, network='Auto'):
        return RsyncList(self.__resid, network)

    def get(self, dest, user=False, network='Auto'):
        return RsyncGet(self.__resid, dest, user, network)


class TransportRsync(BaseTransport):
    '''
    Rsync transport.
    As <resid> accepts rsync url.
    Don't copy access flags except executable.
    Note: this transport works only in synchronous mode.
    '''

    @staticmethod
    def _check_and_update_resid(resid):
        if resid.startswith(('//', ':')):
            return 'rsync:' + resid
        elif not resid.startswith(('rsync://', 'rsync::')):
            raise TransportRsyncError("Error: '%s' is not a valid resid" % resid)
        return resid

    def handle(self, resid):
        return RsyncHandle(self._check_and_update_resid(resid))

    def create(self, *args, **kw):
        raise TransportRsyncError("Rsync transport doesn't supports resource creation.")


class TransportFactoryRsync(object):
    """
    Factory for rsync transports
    """
    def getMagic(self):  # noqa
        return ['rsync']

    def checkMagic(self, magic):  # noqa
        return magic in self.getMagic()

    def create(self):
        return TransportRsync()

    def ops(self):
        return 'network',
