# coding: utf-8
from __future__ import absolute_import

import os
import sys

import api.copier
from api.copier.errors import ResourceError

from kernel.util.functional import singleton
from kernel.util.errors import formatException

from ...common import sandbox
from ...rbtorrent import logger as logging
from . import generic
from . import BaseTransport


class TransportSandBoxError(ResourceError):
    """
    Sandbox transport error
    """
    pass


@singleton
def logger():
    return logging.initialize_client().getChild('sandbox')


@singleton
def _sandbox_client():
    return sandbox.Client('https://sandbox.yandex-team.ru/api/v1.0', logger=logger(), user_agent='skynet')


class TransportBase(BaseTransport):
    @staticmethod
    def _validate_resid(resid):
        try:
            int(resid)
        except ValueError:
            sys.stderr.write(formatException())
            raise TransportSandBoxError("Bad resource id: '%s'" % resid)

    @classmethod
    def _handle_maker(cls, resid):
        raise NotImplemented()

    def handle(self, resid):
        self._validate_resid(resid)
        return self._handle_maker(resid)

    def create(self, *args, **kw):
        raise TransportSandBoxError('SandboxTask transport does not supports creation')


# Sandbox Task Transport
class SandBoxTaskBase(generic.HandlerBase):
    # A list of resource types to be ignored on fetching all of sandbox task resources
    IGNORE_RESOURCE_TYPES = ['TASK_LOGS']

    def __init__(self, resid, dest, user, priority, network, deduplicate, max_dl_speed, max_ul_speed):
        super(SandBoxTaskBase, self).__init__(resid)

        # Get a list of resources in the task specified first
        resid = int(resid)
        srv = _sandbox_client()
        response = srv.request('GET', '/resource', params={'task_id': resid, 'state': 'READY', 'limit': 1000})
        task_resources = response.get('items', [])

        if not task_resources:
            raise TransportSandBoxError('Task %s not found' % resid)

        resource_ids = [
            'sandbox-resource:%d' % (r.get('id'), )
            for r in task_resources
            if r.get('type_name') not in SandBoxTaskBase.IGNORE_RESOURCE_TYPES
        ]

        if not resource_ids:
            raise TransportSandBoxError('No resources found in sandbox task #%d' % resid)

        resources = 'all:' + ','.join(resource_ids)
        logger().getChild('task').debug('resource %r compiled for task #%d', resources, resid)
        self._downloader = self._downloader_maker(
            api.copier.Copier(),
            resources, dest, user,
            priority, network, deduplicate,
            max_dl_speed, max_ul_speed,
        )


class SandBoxTaskHandle(generic.HandlerBase):
    def list(self, priority, network):
        return type('SandBoxTaskList', (SandBoxTaskBase, generic.Lister), {})(
            self._resid, None, None, priority, network, None, None, None
        )

    def get(self, dest, user, priority, network, deduplicate, max_dl_speed, max_ul_speed):
        return type('SandBoxTaskGet', (SandBoxTaskBase, generic.Getter), {})(
            self._resid, dest, user, priority, network, deduplicate, max_dl_speed, max_ul_speed
        )


class TransportSandBoxTask(TransportBase):
    """
    Sandbox Task transport. Download task's resources from Sandbox.
    As <resid> accepts Sandbox's task_id. Will download all resources which are
    in READY state.
    The download process happens using rbtorrent:, rsync: or http: transports.
    """
    @classmethod
    def _handle_maker(cls, resid):
        return SandBoxTaskHandle(resid)


class TransportFactorySandBoxTask(generic.TransportFactoryBase):
    """
    Factory for sandbox task transport
    """
    def __init__(self):
        super(TransportFactorySandBoxTask, self).__init__(['sandbox', 'sandbox-task', 'sbt'], TransportSandBoxTask)

    def ops(self):
        return 'priority', 'network', 'deduplicate', 'max_dl_speed', 'max_ul_speed'


# Sandbox Resource Transport
class SandBoxResourceBase(generic.HandlerBase):
    def __init__(self, resid, dest, user, priority, network, deduplicate, max_dl_speed, max_ul_speed):
        super(SandBoxResourceBase, self).__init__(resid)

        srv = _sandbox_client()
        try:
            resource = srv.request('GET', '/resource/{}'.format(resid), headers={'X-Links': 'true'})
        except srv.HTTPError as exc:
            if exc.response.status_code == 404:
                raise TransportSandBoxError('Resource %s not found' % resid)
            raise

        # Prepare the `any` fetch string in a strict order - first try to get the resource with
        # `rbtorrent` transport, after try to it with `rsync` and at the last try `http` transport.
        urls = []
        torrent_url = resource.get('skynet_id')
        if torrent_url:
            urls.append(torrent_url)
        urls.extend(resource['rsync']['links'])
        if resource.get('md5'):
            # Directory resources has no MD5 checksum,
            # so we will not add directories to fetch with HTTP transport.
            urls.extend(resource['http']['links'])

        resources = 'any:' + ','.join(urls)
        logger().getChild('resource').debug('Resource %r compiled for resource #%s', resources, resid)
        self._downloader = self._downloader_maker(
            api.copier.Copier(),
            resources, dest, user,
            priority, network, deduplicate,
            max_dl_speed, max_ul_speed,
        )


class SandBoxResourceHandle(generic.HandlerBase):
    def list(self, priority, network):
        return type('SandBoxResourceList', (SandBoxResourceBase, generic.Lister), {})(
            self._resid, None, None, priority, network, None, None, None
        )

    def get(self, dest, user, priority, network, deduplicate, max_dl_speed, max_ul_speed):
        return type('SandBoxResourceGet', (SandBoxResourceBase, generic.Getter), {})(
            self._resid, dest, user, priority, network, deduplicate, max_dl_speed, max_ul_speed
        )


class TransportSandBoxResource(TransportBase):
    """
    Sandbox Resource transport. Download resource from Sandbox.
    As <resid> accepts Sandbox's resource_id.
    The download process happens using rbtorrent:, rsync: or http: transports.
    """
    @classmethod
    def _handle_maker(cls, resid):
        return SandBoxResourceHandle(resid)


class TransportFactorySandBoxResource(generic.TransportFactoryBase):
    """
    Factory for sandbox resource transport
    """
    def __init__(self):
        super(TransportFactorySandBoxResource, self).__init__(['sandbox-resource', 'sbr'], TransportSandBoxResource)

    def ops(self):
        return 'priority', 'network', 'deduplicate', 'max_dl_speed', 'max_ul_speed'
