from __future__ import absolute_import

import tarfile
import base64
from cStringIO import StringIO
import os

from . import DummyIter


class TgzCreate(DummyIter):
    def __init__(self, files):
        self._files = files
        self._result = None
        self._result_ready = False

    def iter(self, timeout=None, state_version=1):
        if self._result is None:
            output = StringIO()
            f = tarfile.open(None, "w:gz", output)
            for apath, path in self._files:
                if os.path.isfile(apath) or os.path.isdir(apath):
                    f.add(apath, arcname=path)
            f.close()
            self._result = TgzHandler(
                resid='%s:%s' % (
                    TransportFactoryTgz().get_magic()[0],
                    base64.b64encode(output.getvalue())
                )
            )

        self._result_ready = True
        return
        yield  # force this function to be a generator

    def wait(self, timeout=None):
        if self._result_ready:
            return self._result

        for _ in self.iter(timeout=timeout):
            pass

        assert self._result_ready
        return self._result


class TgzGet(object):
    def __init__(self, tar, dest, user):
        self._tar, self._dest, self._result = tar, dest, None
        self._result_ready = False

    def iter(self, timeout=None, state_version=1):
        if self._result is None:
            try:
                umask = os.umask(0)
                self._tar.extractall(self._dest)
                self._result = True
            finally:
                os.umask(umask)

        self._result_ready = True
        return
        yield  # force this function to be a generator

    def wait(self, timeout=None):
        if self._result_ready:
            return self._result

        for _ in self.iter(timeout=timeout):
            pass

        assert self._result_ready
        return self._result


class TgzList(object):
    def __init__(self, tar):
        self._tar = tar
        self._result = None

    def wait(self, timeout=None):
        if self._result is None:
            result = []
            for item in self._tar:
                assert item.isdir() or item.isfile()
                result.append({
                    'name': item.name,
                    'size': item.size,
                    'type': 'file' if item.isfile() else 'dir',
                    'executable': any(
                        item.mode & bit != 0
                        for bit in (0100, 010, 01)
                    )
                })
            self._result = result
        return self._result


class TgzHandler(object):
    def __init__(self, resid=None):
        self._resid = resid
        self._tar = None

    def _get_tar(self):
        if self._tar is None:
            sio = StringIO(base64.b64decode(self._resid))
            self._tar = tarfile.open(None, 'r:*', sio)
        return self._tar

    def resid(self):
        return self._resid

    def get(self, dest, user=False):
        return TgzGet(self._get_tar(), dest, user)

    def list(self):
        return TgzList(self._get_tar())


class TransportTgz(object):
    """
    Tar.[bz2|gz] transport. Unpacking .tar.[bz2|gz] archives.
    As <resid> accepts base64 encoded archive of .tar.gz or .tar.bz2 formats.
    """
    def create(self, files):
        return TgzCreate(files)

    def handle(self, resid):
        return TgzHandler(resid=resid)


class TransportFactoryTgz(object):
    """
    Factory for tgz-based transport
    """
    def get_magic(self):
        return ['tgz', 'tbz2', 'tar']

    def check_magic(self, magic):
        return magic in self.get_magic()

    def create(self):
        return TransportTgz()
