import os
import socket
import urlparse
import contextlib

# Hostname default domain name, which will be additionally searched.
SEARCH_DOMAIN = '.yandex.ru'
# Default NS record, which is returned by the reverse DNS requests
# for the IP address without real NS record assigned.
DEFAULT_NS_RECORD = 'any' + SEARCH_DOMAIN


def normalizeHost(host=None):
    """
    Normalizes given host name by lowering case, cutting off ending dot.
    :param host: Host name to be normalized. Get local host name in case of not specified.
    :return:     Normalized host name.
    """
    if not host:
        return socket.getfqdn().lower()

    # Normalize host name given a bit
    return host[:-1].lower() if host.endswith('.') else host.lower()


def looksLikeFastbonized(host=None):
    """
    Checks whatever the host name specified is suitable for fastbone network or not.
    :param host: Host name to check.
    :return:     True in case of host name is looks like fastbonized.
    """
    host = normalizeHost(host)

    if host.startswith('fb-'):
        return True
    elif host.endswith('.fb' + SEARCH_DOMAIN):
        return True
    else:
        return False


def hasReverseRecord(host=None):
    """
    :param host: Host name to check.
    :return:     True in case the host name given has reverse NS record.
    """
    host = normalizeHost(host)
    try:
        ip = [i for i in socket.getaddrinfo(host, 0) if i[0] == socket.AF_INET6][0][4][0]
        return socket.gethostbyaddr(ip)[0] != DEFAULT_NS_RECORD
    except:
        return False


def fastbonizeHostName(host=None):
    """
    :param host: Host name to check.
    :return:     Modified, full-qualified host name, which is suitable for fastbone network.
                 Otherwise, `None` will be returned.
    """
    host = normalizeHost(host)
    if looksLikeFastbonized(host) and hasReverseRecord(host):
        return host

    # Try to add '.fb' suffix if host name ends with "yandex.ru".
    # Otherwise try to add 'fb-' prefix now
    fb_host = (
        host[:-len(SEARCH_DOMAIN)] + '.fb' + SEARCH_DOMAIN
        if host.endswith(SEARCH_DOMAIN) else
        'fb-' + host
    )

    return fb_host if hasReverseRecord(fb_host) else None


def fastbonizeURL(url, checkLocalHost=True):
    """
    The method will detect if the remote and local hosts are belongs to fastbone network,
    and, if they are, replace the host name in the URL given with the fastbone-specific host name.
    :param url: URL to process.
    :return:    Updated URL in case of both remote and local hosts are belongs to fastbone network
                or original URL otherwise.
    """

    # First of all, check the host itself is in fastbone network
    # KORUM: Can we make it some kind of static?
    if not checkLocalHost or fastbonizeHostName():
        # Make sure the URL specified is suitable for fastbone network
        parsed_url = urlparse.urlparse(url)
        host = parsed_url.netloc.rsplit(':', 1)[0]
        fb_host = fastbonizeHostName(host)
        if fb_host:
            return url.replace(host, fastbonizeHostName(host), 1)

    return url


@contextlib.contextmanager
def umask(new_mask=0000):
    """
    Scoped umask setter.
    :param new_mask:    A mask to be set within the scope. In case of `None` passed,
                        the mask will left unchanged.
    """
    if new_mask is None:
        yield None
        return

    old_mask = os.umask(new_mask)
    try:
        yield old_mask
    finally:
        os.umask(old_mask)
