import errno
import getpass
import httplib
import logging
import re
import select
import socket
import time
import urlparse
import uuid

import requests

urllib3 = requests.packages.urllib3


class Client(object):
    """
    This is a backported and stripped-down version of `sandbox.common.rest.Client`
    designed to work under the obsolete virtual environment of Skynet.
    """
    TIMEOUT = 60

    TOO_MANY_REQUESTS = 429
    RETRIABLE_CODES = (
        TOO_MANY_REQUESTS,
        httplib.REQUEST_TIMEOUT,
        httplib.BAD_GATEWAY,
        httplib.SERVICE_UNAVAILABLE,
        httplib.GATEWAY_TIMEOUT
    )

    class TimeoutExceeded(Exception):
        """ A error class to be raised in case of maximum amount of wait time exceeded """

    class HTTPError(requests.HTTPError):
        @classmethod
        def create(cls, exc):
            obj = cls()
            obj.args = exc.args
            obj.response = exc.response
            return obj

    def __init__(self, base_url, logger=None, max_wait=600, user_agent=None):
        self.__parsed_base_url = urlparse.urlparse(base_url)
        self.__max_wait = max_wait

        host_id = '{}@{}'.format(getpass.getuser(), socket.getfqdn())

        if user_agent:
            self.__user_agent = '{}::{}'.format(user_agent, host_id)
        else:
            self.__user_agent = host_id

        self.logger = logger or logging.getLogger(__name__)

    def request(self, method, path, *args, **kwargs):
        request_id = uuid.uuid4().hex

        headers = kwargs.pop('headers', {}).copy()
        headers['X-Request-Id'] = request_id
        headers['User-Agent'] = self.__user_agent
        kwargs.pop('timeout', None)

        url = self._prepare_url(path)

        waiter = ProgressiveWaiter(start_tick=5, max_tick=180, max_wait=self.__max_wait)
        for _ in waiter:
            try:
                timeout = min(self.TIMEOUT, waiter.remained)
                self.logger.debug(
                    "[%s] %s %s, timeout %.1fs, query %s", request_id, method, url, timeout, kwargs.get("params") or {}
                )

                response = requests.request(method, url, *args, headers=headers, timeout=timeout, **kwargs)
                response.raise_for_status()

                self.logger.debug(
                    "[%s] finished at %r after %.3fs (HTTP %d)",
                    request_id, response.headers.get('X-Backend-Server'), waiter.spent, response.status_code
                )

                return self._decode_response(response)

            except select.error as ex:
                if ex.args[0] == errno.EINTR:
                    self.logger.warning("REST request [%s]: error: '%s'", request_id, ex)
                else:
                    raise

            except requests.ConnectionError as ex:
                self.logger.warning("[%s]: connection error after %.3fs: '%s'", request_id, waiter.spent, ex)

                try:
                    if isinstance(ex.args[0], urllib3.exceptions.MaxRetryError):
                        # Do not sleep on connection timeout
                        waiter.do_not_wait()
                        continue

                    err = ex.args[0].args[1]
                    errnos = (errno.ECONNREFUSED, errno.EHOSTDOWN, errno.ENETUNREACH)
                    if isinstance(err, socket.error) and err.errno in errnos:
                        waiter.do_not_wait()
                        continue

                except IndexError:
                    pass

            except requests.HTTPError as ex:
                status = ex.response.status_code
                self.logger.warning("[%s]: got HTTP %d after %.3fs: '%s'", request_id, status, waiter.spent, ex)

                if status == httplib.GONE or status not in self.RETRIABLE_CODES:
                    raise self.HTTPError.create(ex)

                if ex.response.status_code == httplib.BAD_GATEWAY:
                    waiter.do_not_wait()
                    continue

        raise self.TimeoutExceeded(
            "Failed to request '{} {}' - no response given for {:.3f}s."
            .format(method, path, waiter.spent)
        )

    def _prepare_url(self, path):
        full_path = '{}{}'.format(self.__parsed_base_url.path, re.sub(r'/+', '/', path))
        full_parsed_url = self.__parsed_base_url._replace(path=full_path)
        return urlparse.urlunparse(full_parsed_url)

    @staticmethod
    def _decode_response(response):
        try:
            return response.json()
        except Exception:
            return response.text


class ProgressiveWaiter(object):

    def __init__(self, start_tick, max_tick, max_wait, factor=55. / 34.):
        self.start_tick = float(start_tick) or 0.001
        self.max_tick = float(max_tick)
        self.max_wait = float(max_wait)
        self.factor = float(factor)

        self.__reset_flag = False
        self.__started = None
        self.__current_tick = None
        self.__attempt = None

    def do_not_wait(self, tick_reset=True):
        self.__reset_flag = True
        if tick_reset:
            self.__current_tick = self.start_tick

    @property
    def spent(self):
        if not self.__started:
            raise ValueError('Waiter has not started yet')
        return time.time() - self.__started

    @property
    def remained(self):
        if not self.__started:
            raise ValueError('Waiter has not started yet')
        return max(self.max_wait - self.spent, 0.)

    @property
    def tick(self):
        if not self.__current_tick:
            raise ValueError('Waiter has not started yet')
        return min(self.__current_tick, self.remained)

    def __iter__(self):
        self.__current_tick = self.start_tick
        self.__started = time.time()
        self.__attempt = 0

        yield self.__make_attempt()

        while self.remained > 0.:

            if self.__reset_flag:
                self.__reset_flag = False
                yield self.__make_attempt()

            sleep_for = self.tick
            time.sleep(sleep_for)
            self.__current_tick = min(self.max_tick, self.__current_tick * self.factor)
            self.__reset_flag = False

            yield self.__make_attempt()

    def __make_attempt(self):
        t = self.__attempt
        self.__attempt += 1
        return t
