"""
The purpose of this module is to perform administrative (maintenance) tasks on running copier instance.
"""

from __future__ import absolute_import, print_function, division

import argparse
import locale
import sys
import time

from ...kernel_util.functional import singleton
from ...rbtorrent import logger as logging
from ...rbtorrent.utils import human_size_sep, human_time
from ...rpc.errors import CallError, CallFail


@singleton
def log():
    return logging.initialize_client()


def _server():
    from api.skycore import ServiceManager
    return ServiceManager().get_service_python_api('skynet', 'skybone-mds', 'python_rpc')


def retry_call(name, *args, **kwargs):
    tries = 0
    while True:
        tries += 1
        try:
            result = _server().call(name, *args, **kwargs).wait()
        except Exception as ex:
            if isinstance(ex, (CallError, CallFail)):
                raise

            if tries >= 10:
                raise

            log().debug('admin (%s): server.call: error: %s, tries: %d', name, ex, tries)
            time.sleep(min(10, tries * 2))
        else:
            return result


def _formatter(data, fmt):
    formatters = {
        'text': lambda x: repr(x),
        'yaml': lambda x: __import__('yaml').dump(x),
        'json': lambda x: __import__('json').dumps(x),
        'pprint': lambda x: __import__('pprint').pformat(x),
        'msgpack': lambda x: __import__('msgpack').dumps(x),
    }
    print(formatters[fmt](data))
    return 0


def status(args):
    status = retry_call('status', dbwait=args.wait)
    if args.format != 'text':
        return _formatter(status, args.format)

    from collections import defaultdict

    class NotFound(object):
        def __str__(self):
            return '<???>'

        def __getitem__(self, key):
            return NotFound()

        __repr__ = __str__

    pstatus = defaultdict(NotFound)

    def grab_value(path, default=NotFound(), fltr=str):
        if isinstance(path, (list, tuple)):
            return [grab_value(onepath, default, fltr=fltr) for onepath in path]

        parts, key = path.rsplit('.', 1)
        parts = parts.split('.')

        # (root, key, item)
        stack = [(None, None, status)]
        value = default

        for part in parts:
            parent = stack[-1][2]
            current = parent.get(part, None)

            if not isinstance(current, dict):
                break

            stack.append((parent, part, current))
        else:
            value = stack[-1][2].pop(key, value)
            if value != default:
                value = fltr(value)

        for (parent, key, item) in reversed(stack):
            if not item and parent is not None:
                del parent[key]
            else:
                break

        return value

    # Daemon section
    stopping, active = grab_value((
        'daemon.stopping', 'daemon.active'
    ), False)

    if stopping:
        pstatus['status'] = 'stopping'
    elif active:
        pstatus['status'] = 'active'
    else:
        pstatus['status'] = 'unknown'

    daemon_uptime = grab_value('daemon.uptime', fltr=int)
    pstatus['uptime'] = human_time(daemon_uptime)
    pstatus['uid'] = grab_value('db.sqlite.maintainance.uid')

    if daemon_uptime > 0:
        cpu_self = sum(grab_value((
            'daemon.rusage.self.user',
            'daemon.rusage.self.system',
        ), fltr=float)) / float(daemon_uptime)

        cpu_children = sum(grab_value((
            'daemon.rusage.children.user',
            'daemon.rusage.children.system',
        ), fltr=float)) / float(daemon_uptime)
    else:
        status['daemon'].pop('rusage')
        cpu_self = cpu_children = 0

    pstatus['cpu_percent_self'] = cpu_self
    pstatus['cpu_percent_children'] = cpu_children

    pstatus['greenlets_count'] = grab_value('daemon.greenlets')

    # File cache
    pstatus['file_cache_resources_count'] = grab_value('file_cache.resources_count')

    # Jobs section
    line_job_stat = ''
    job_stat_str = 'RPC:\n'
    job_stat_str += '  Jobs (in progress, finished, error, failed):'
    all_job_types = sorted(set(
        status['rpc']['active']['jobs'].keys() +
        status['rpc']['counters']['completed'].keys() +
        status['rpc']['counters']['errors'].keys()
    ))

    for job in all_job_types:
        job_active = status['rpc']['active']['jobs'].get(job, 0)
        job_successes = status['rpc']['counters']['completed'].get(job, 0)
        job_errors = status['rpc']['counters']['errors'].get(job, {})

        job_error = job_fatal = 0

        for err, count in job_errors.iteritems():
            if err in (
                'ApiError',
            ):
                job_error += count
            else:
                job_fatal += count

        job_stat_str += '\n    %-20s: %s/%s/%s/%s' % (
            job,
            job_active,
            job_successes,
            job_error,
            job_fatal
        )

        if job == 'resource_create':
            short_job = 'cre'
        elif job == 'resource_download':
            short_job = 'dl '
        elif job == 'resource_info':
            short_job = 'inf'
        else:
            short_job = None

        if short_job:
            line_job_stat += '| %s %5s %5s %5s %5s ' % (
                short_job,
                job_active, job_successes,
                job_error, job_fatal
            )

    job_stat_str += '\n  Messages (total sessions: %d):' % (status['rpc']['counters']['sessions'], )

    for msg, count in status['rpc']['counters']['messages'].iteritems():
        job_stat_str += '\n    %-20s: %s' % (msg, count)

    status.pop('rpc')
    pstatus['job_stat'] = job_stat_str
    pstatus['line_job_stat'] = line_job_stat

    pstatus['jobs_in_progress'] = grab_value('jobs.pending')
    pstatus['jobs_success_count'] = grab_value('jobs.success')
    pstatus['jobs_failed_count'] = grab_value('jobs.failed')

    # Announcer section
    pstatus['announcer_pending_count'] = grab_value('announcer.pending')
    pstatus['announcer_packets_sent'] = grab_value('announcer.stats.packets.sent')
    pstatus['announcer_packets_recv'] = grab_value('announcer.stats.packets.recv')
    pstatus['announcer_packets_lost'] = grab_value('announcer.stats.packets.lost')

    # Database section
    locked = grab_value('db.sqlite.locked', fltr=bool)
    if locked:
        lock_reason = grab_value('db.sqlite.lock_reason')
        if lock_reason == 'None':
            lock_reason = 'unknown'
        pstatus['sqlite_locked_str'] = ', locked (reason: %s)' % (lock_reason, )
    else:
        pstatus['sqlite_locked_str'] = ''

    pstatus['sqlite_job_waiters'] = grab_value('db.sqlite.job_waiters', fltr=int)
    pstatus['sqlite_mem_generic'] = grab_value('db.sqlite.memory.generic', fltr=human_size_sep)
    pstatus['sqlite_mem_total'] = grab_value('db.sqlite.memory.total', fltr=human_size_sep)
    pstatus['sqlite_mem_cache'] = grab_value('db.sqlite.memory.cache', fltr=human_size_sep)
    pstatus['sqlite_cache_hit'] = grab_value('db.sqlite.cache.hit', fltr=int)
    pstatus['sqlite_cache_miss'] = grab_value('db.sqlite.cache.miss', fltr=int)
    pstatus['sqlite_cache_hit_miss_maxlen'] = max(
        len(str(_)) for _ in (pstatus['sqlite_cache_hit'], pstatus['sqlite_cache_miss'])
    )
    pstatus['sqlite_cache_writes'] = grab_value('db.sqlite.cache.writes', fltr=human_size_sep)
    pstatus['sqlite_prep_stmt_used'] = grab_value('db.sqlite.memory.prep_stmt', fltr=human_size_sep)

    data_used, data_free = grab_value((
        'db.sqlite.data.used_bytes', 'db.sqlite.data.free_bytes',
    ), fltr=int)

    if not isinstance(data_free, NotFound) and not isinstance(data_used, NotFound):
        data_free_percent = data_free / data_used
        pstatus['sqlite_data_used_bytes'] = human_size_sep(data_used)
        pstatus['sqlite_data_free_bytes'] = human_size_sep(data_free)
        pstatus['sqlite_data_free_percent'] = data_free_percent
    else:
        data_free_percent = NotFound()
        pstatus['sqlite_data_used_bytes'] = data_used
        pstatus['sqlite_data_free_bytes'] = data_free
        pstatus['sqlite_data_free_percent'] = 0

    try:
        total_cache_tries = pstatus['sqlite_cache_hit'] + pstatus['sqlite_cache_miss']
        pstatus['sqlite_cache_hit_percent'] = pstatus['sqlite_cache_hit'] / float(total_cache_tries)
        pstatus['sqlite_cache_miss_percent'] = pstatus['sqlite_cache_miss'] / float(total_cache_tries)
    except:
        pstatus['sqlite_cache_hit_percent'] = pstatus['sqlite_cache_miss_percent'] = 0

    last_vacuum = grab_value('db.sqlite.maintainance.last_vacuum', fltr=int)
    if not isinstance(last_vacuum, NotFound):
        pstatus['sqlite_maintainance_vacuum'] = human_time(time.time() - last_vacuum)

    pstatus['sqlite_maintainance_vacuum_duration'] = grab_value(
        'db.sqlite.maintainance.last_vacuum_duration', fltr=human_time
    )

    last_analyze = grab_value('db.sqlite.maintainance.last_analyze', fltr=int)
    if not isinstance(last_analyze, NotFound):
        pstatus['sqlite_maintainance_analyze'] = human_time(time.time() - last_analyze)

    pstatus['sqlite_maintainance_analyze_duration'] = grab_value(
        'db.sqlite.maintainance.last_analyze_duration', fltr=human_time
    )

    pstatus['sqlite_active_job'] = grab_value('db.sqlite.job', fltr=tuple)

    import string

    if not args.line:
        print(string.Formatter().vformat('''
Copier v2.0  (uid: {uid})

Status: {status}  |  uptime {uptime}  |  CPU {cpu_percent_self:2.0%} (+{cpu_percent_children:2.0%})

{job_stat}

Gevent
  Greenlets    : {greenlets_count}

File cache
  Resources cnt: {file_cache_resources_count}

Announcer
  Pending      : {announcer_pending_count}
  Packets      : {announcer_packets_sent} / {announcer_packets_recv} / {announcer_packets_lost}    (send/recv/lost)

Database (sqlite{sqlite_locked_str}):
  Memory info
    total      : {sqlite_mem_total[0]:>6} {sqlite_mem_total[1]}
    generic    : {sqlite_mem_generic[0]:>6} {sqlite_mem_generic[1]}
    cache      : {sqlite_mem_cache[0]:>6} {sqlite_mem_cache[1]}
    prep_stmt  : {sqlite_prep_stmt_used[0]:>6} {sqlite_prep_stmt_used[1]}
  Cache
    written    : {sqlite_cache_writes[0]:>6} {sqlite_cache_writes[1]}
    hits       : [{sqlite_cache_hit_percent:3.0%}] {sqlite_cache_hit:>{sqlite_cache_hit_miss_maxlen}}
    miss       : [{sqlite_cache_miss_percent:3.0%}] {sqlite_cache_miss:>{sqlite_cache_hit_miss_maxlen}}
  Data
    size       : {sqlite_data_used_bytes[0]:>6} {sqlite_data_used_bytes[1]}
    free       : {sqlite_data_free_bytes[0]:>6} {sqlite_data_free_bytes[1]} [{sqlite_data_free_percent:3.0%}]
  Maintanance
    vacuum     : {sqlite_maintainance_vacuum} ago (took {sqlite_maintainance_vacuum_duration})
    analyze    : {sqlite_maintainance_analyze} ago (took {sqlite_maintainance_analyze_duration})
  Locks
    Waiting    : {sqlite_job_waiters}
    Active
      meth     : {sqlite_active_job[0]}
      args     : {sqlite_active_job[1]}
      kwargs   : {sqlite_active_job[2]}
''', (), pstatus).strip())
    else:
        print(string.Formatter().vformat('''\
{status} | \
{uptime:>12} | \
{cpu_percent_self:>4.0%} | {cpu_percent_children:4.0%} \
|{line_job_stat}|| \
skbt {dlrs_skbt_ok:>5} {dlrs_skbt_err:>5} || \
look {proxy_infohash_time_spent:>8}
''', (), pstatus).strip())

    if status:
        print()
        print('Extra status items: %r' % (status, ))


def query(args):
    result = retry_call('query', args.sql)
    if result is None:
        print('result: NONE')
    else:
        if len(result) == 0:
            print('0 rows returned')
        else:
            cols = []
            resultset = []

            for row in result[:1000]:
                nrow = []
                for idx, col in enumerate(row):
                    value = str(col)
                    if len(cols) == idx:
                        cols.append(len(value))
                    else:
                        cols[idx] = max(cols[idx], len(value))

                    nrow.append(value)
                resultset.append(nrow)

            def header():
                return '+-' + '-+-'.join(['-' * cl for cl in cols]) + '-+'

            print(header())
            for row in resultset:
                frow = []
                for idx, col in enumerate(row):
                    collen = cols[idx]
                    frow.append('{0:<{len}}'.format(col, len=collen))

                print('| ' + ' | '.join(frow) + ' |')
            print(header())

            if len(result) > 1000:
                print('displayed first 1000 entries (%d total)' % (len(result), ))


def dbbackup(args):
    retry_call('dbbackup')


def evaluate(args):
    print(retry_call('evaluate', args.code))


def resource_add(args):
    data = sys.stdin.read()
    if args.json:
        import simplejson
        print(simplejson.loads(data))
    else:
        import msgpack
        data = msgpack.loads(data)

    retry_call('add_resource', data['uid'], data['head'], data['info'])


def resource_list(args):
    for ret in retry_call('resource_list'):
        print('rbtorrent:' + ret)


def resource_remove(args):
    retry_call('resource_remove', args.id)


def main():
    # First of all, set locale to user's preferred.
    locale.setlocale(locale.LC_ALL, '')

    # Create the top-level parser.
    parser = argparse.ArgumentParser(
        formatter_class=lambda *args, **kwargs: argparse.ArgumentDefaultsHelpFormatter(*args, width=120, **kwargs),
        description='Copier maintenance tool.'
    )
    parser.add_argument(
        '-f', '--format',
        choices=['text', 'yaml', 'json', 'pprint', 'msgpack'],
        default='text',
        help='output format'
    )
    subparsers = parser.add_subparsers(help='operational mode')

    # Create 'status' sub-parser.
    subparser = subparsers.add_parser('status', help='show copier daemon status')
    subparser.add_argument('--line', action='store_true')
    subparser.add_argument('--wait', type=int, default=3)
    subparser.set_defaults(func=status)

    # Create 'query' sub-parser.
    subparser = subparsers.add_parser('query', help='query db')
    subparser.add_argument('sql', metavar='SQL', help='sql to execute')
    subparser.set_defaults(func=query)

    # Create 'dbbackup' sub-parser
    subparser = subparsers.add_parser('dbbackup', help='backup db')
    subparser.set_defaults(func=dbbackup)

    # Create 'eval' sub-parser
    subparser = subparsers.add_parser('eval', help='evaluate code')
    subparser.add_argument('code', metavar='CODE', help='code to evaluate')
    subparser.set_defaults(func=evaluate)

    # Create 'resource_add' sub-parser
    subparser = subparsers.add_parser('resource_add', help='resource add')
    subparser.add_argument('--json', action='store_true', help='stdin in json format (by default we use msgpack)')
    subparser.set_defaults(func=resource_add)

    # Create 'resource_list' sub-parser
    subparser = subparsers.add_parser('resource_list', help='resources list')
    subparser.set_defaults(func=resource_list)

    # Create 'resource_remove' sub-parser
    subparser = subparsers.add_parser('resource_remove', help='remove resource by id')
    subparser.add_argument('id', metavar='ID', help='resource id')
    subparser.set_defaults(func=resource_remove)

    # Parse the args and call whatever function was selected.
    args = parser.parse_args()
    return args.func(args)


if __name__ == '__main__':
    raise SystemExit(main())
