import json
import msgpack
import random
import time

from api.copier import errors

from ..rbtorrent.bencode import bdecode
from ..rbtorrent.component import Component
from ..rbtorrent.resource.announcer import Announcer
from ..rbtorrent.skbn.world import World

from ..rbtorrent.utils import gevent_urlopen


class Lookup(Component):
    def __init__(self, db, dfs_mode, dfs_link_pattern, parent=None):
        self.db = db

        self.dfs_mode = dfs_mode
        self.dfs_link_pattern = dfs_link_pattern
        self.info_cache = {}  # uid => (ttl, info), with negative cache
        self.link_cache = {}  # md5 => (ttl, links)

        super(Lookup, self).__init__(logname='lookup', parent=parent)

    def look_head(self, uid):
        resource_info = self.db.query_one(
            'SELECT data, info FROM resource WHERE id = ?',
            [uid]
        )

        if resource_info:
            head = str(resource_info[0])
            info = msgpack.loads(resource_info[1])  # noqa

            if 'id_map' in info:
                extra = {}
                extra.setdefault('md5_links', {})

                for md5, md5_link_id in info['id_map'].iteritems():
                    links = extra['md5_links'].setdefault(md5, [])
                    links.append('http://localhost/%s.json' % (md5_link_id, ))
            else:
                extra = None

            return bdecode(head), extra

        return None, None

    def get_link(self, uid, md5, head=None):
        links = None

        ttl, links = self.link_cache.get(md5, (None, None))

        if links is None or ttl <= time.time():
            self.link_cache.pop(md5, None)
            links = ttl = None
        else:
            ttl = None

        if links is None:
            info_ttl, info = self.info_cache.get(uid, (None, None))

            if info is None or info_ttl <= time.time():
                info = self.db.query_one_col('SELECT info FROM resource WHERE id = ?', [uid])
                info_ttl = time.time() + 60

                if info:
                    info = msgpack.loads(info)

                if not info:
                    info = False  # set to False as negative caching mark

                self.info_cache[uid] = (info_ttl, info)

            ttl = None

            if info and 'http' in info:
                http_links = info['http']
                if isinstance(http_links, dict):
                    links = http_links.get(md5, [])
                info = None  # reset info, so we do not try other methods
                ttl = time.time() + 60

            if info:
                if self.dfs_mode in ('idmap', 'idmap_mds'):
                    if 'id_map' not in info or not isinstance(info['id_map'], dict):
                        info = None
                elif self.dfs_mode == 'yt':
                    if 'yt_lookup_uri' not in info or not isinstance(info['yt_lookup_uri'], basestring):
                        info = None

            if info:
                if self.dfs_mode == 'yt':
                    query_link = info['yt_lookup_uri']
                    req = gevent_urlopen(query_link)
                    data = req.read()
                    if data:
                        data = json.loads(data)
                        data_by_fn = {}

                        links = {}

                        for info in data['parts']:
                            data_by_fn.setdefault(info['filename'], []).append(info)

                        assert head, 'head was not set'

                        for fn, fninfo in head['structure'].iteritems():
                            ranges = links[fninfo['md5sum'].encode('hex')] = []

                            for range_info in sorted(data_by_fn[fn], key=lambda x: x['range'][0], reverse=0):
                                range_start, range_end = range_info['range']
                                if ranges:
                                    assert range_start == (ranges[-1][0] + ranges[-1][1]), 'ranges overlap!'

                                locations = list(range_info['locations'])
                                random.shuffle(locations)
                                ranges.append((range_start, range_end - range_start, tuple(locations)))

                        ttl = time.time() + 60

                elif self.dfs_mode in ('idmap', 'idmap_mds'):
                    file_id = info['id_map'].get(md5, None)
                    if file_id:
                        query_link = self.dfs_link_pattern % (file_id, )
                        req = gevent_urlopen(query_link)
                        data = req.read()
                        if data:
                            data = json.loads(data)
                            if self.dfs_mode == 'idmap':
                                if data and isinstance(data, (list, tuple)):
                                    links = []
                                    for link in data:
                                        links.append(str(link))
                                    ttl = time.time() + 60
                            elif self.dfs_mode == 'idmap_mds':
                                if data and isinstance(data, dict):
                                    links = []
                                    link = 'http://%s:8181%s?ts=%s&sign=%s' % (
                                        data['host'], data['path'],
                                        data['ts'], data['s']
                                    )
                                    links.append(str(link))
                                    ttl = time.time() + 60

        # Return format
        # [(start, size, (link1, linkN)), (start, size, (link1, linkN))]

        if links and ttl:
            if isinstance(links, dict):
                # Store resulting links
                resulting_links = links[md5]

                # Store all final links by md5 checksum in cache
                for md5, md5links in links.iteritems():
                    self.link_cache[md5] = (ttl, md5links)

                # Switch final links
                links = resulting_links
            else:
                self.link_cache[md5] = (ttl, links)

        if links:
            return links

        return None

    @Component.green_loop
    def cleancaches(self):
        now = time.time()

        for uid, cache in self.info_cache.items():
            if cache[0] < now:
                self.info_cache.pop(uid)

        for md5, cache in self.link_cache.items():
            if cache[0] < now:
                self.link_cache.pop(md5)

        return 3


class ResourceManager(Component):
    def __init__(
        self, uid, desc, workdir, db, trackers,
        data_port, announce_port,
        ips,
        dfs_mode, dfs_link_pattern,
        parent=None
    ):
        super(ResourceManager, self).__init__(logname='res', parent=parent)

        self.uid = uid
        self.desc = desc
        self.workdir = workdir
        self.db = db
        self.trackers = trackers

        self.data_port = data_port
        self.announce_port = announce_port
        self.ips = ips

        self.lookupper = Lookup(self.db, dfs_mode, dfs_link_pattern, self)

        self.world = World(uid=self.uid, desc=self.desc, port=data_port, parent=self)
        self.world.enable_daemon_mode()
        self.world.set_lookupper(self.lookupper)

        self.world.set_handles_clean_watchdog(600)  # check handles clean routine

        self.announcer = Announcer(
            self.uid, db, trackers=trackers, ips=self.ips,
            port=announce_port, data_port=data_port, dfs=True, parent=self
        )

    def add_resource(self, uid, head, info):
        if uid.startswith('rbtorrent:'):
            uid = uid.split(':', 1)[1]

        with self.db:
            self.db.query(
                'INSERT OR REPLACE INTO resource (id, type, data, info) VALUES (?, ?, ?, ?)',
                [uid, 'rbtorrent1', buffer(head), buffer(msgpack.dumps(info))]
            )

        if not self.announcer.announce_or_fail(uid, timeout=300):
            self.log.warning('Unable to announce -- timeout')
            raise errors.CopierError('Tracker registration timed out (5min)')

        return True

    def remove_resource(self, uid):
        if uid.startswith('rbtorrent:'):
            uid = uid.split(':', 1)[1]

        with self.db:
            self.db.query(
                'INSERT OR REPLACE INTO announce_stop (tracker, hash, deadline) '
                'SELECT tracker, resource, '
                '   CASE WHEN timestamp <= ? '
                '       THEN ? '
                '       ELSE timestamp '
                '   END '
                'FROM announce WHERE resource = ?',
                [int(time.time()), int(time.time() + 86400), uid]
            )
            self.db.query(
                'DELETE FROM resource WHERE id = ?', [uid]
            )

        [sched.wakeup() for sched in self.announcer.scheduler_loops]

        return True
