import gevent
import resource
import gc
import greenlet

from ..kernel_util.sys.gettime import monoTime
from ..kernel_util.sys.getpeerid import getpeerid


class SkyboneMDSRPC(object):
    def __init__(self, daemon):
        self.daemon = daemon

    def ping(self, extended=False):
        if not extended:
            return True
        return {
            'active': self.daemon.active.isSet(),
            'stage': self.daemon.startup_stage
        }

    def add_resource(self, uid, head, fmap):
        self.daemon.active.wait()
        return self.daemon.resource_mngr.add_resource(uid, head, fmap)

    def remove_resource(self, uid):
        self.daemon.active.wait()
        return self.daemon.resource_mngr.remove_resource(uid)


class SkyboneMDSAdminRPC(object):
    def __init__(self, daemon):
        self.daemon = daemon
        self.log = daemon.log.getChild('rpc.admin')

    def status(self, dbwait=3):
        self.daemon.active.wait()

        with gevent.Timeout(dbwait) as tout:
            try:
                with self.daemon.db.deblock.lock('grab status'):
                    return self._status(db=True)
            except gevent.Timeout as ex:
                if ex != tout:
                    raise
                return self._status(db=False)

    def _status(self, db):
        dbcounters = {}
        if db:
            dbstatus = self.daemon.db.status()
            dbstatus['locked'] = False

            dbcounters = {
                'resource_cnt': self.daemon.db.query_one_col('SELECT COUNT(*) FROM resource'),
            }
        else:
            dbstatus = {}
            dbstatus['locked'] = True
            dbstatus['lock_reason'] = self.daemon.db.deblock.lock_reason
            dbstatus['job'] = self.daemon.db.deblock.job
            dbstatus['job_waiters'] = self.daemon.db.deblock.lock_waiters

            if dbstatus['job']:
                dbstatus['job'] = list(dbstatus['job'])
                meth, args, kwargs, local = dbstatus['job']

                dbstatus['job'] = (meth.__func__.__name__, repr(args)[:100], repr(kwargs)[:100])
            else:
                dbstatus['job'] = (None, None, None)

            dbcounters = {
                'resource_cnt': None,
                'data_cnt': None,
                'data_size': None,
                'file_cnt': None,
                'wait_checking': None
            }

        rusage_self = resource.getrusage(resource.RUSAGE_SELF)
        rusage_children = resource.getrusage(resource.RUSAGE_CHILDREN)

        greenlets_count = 0
        for obj in gc.get_objects():
            if isinstance(obj, greenlet.greenlet):
                greenlets_count += 1

        if hasattr(self.daemon, 'resource_mngr'):
            announcer_stats = self.daemon.resource_mngr.announcer.status(use_db=db)
        else:
            announcer_stats = {}

        return {
            'daemon': {
                'active': self.daemon.active.isSet(),
                'uptime': monoTime() - self.daemon.ts_started,
                'stopping': self.daemon.stopping,
                'rusage': {
                    'self': {
                        'user': rusage_self.ru_utime,
                        'system': rusage_self.ru_stime,
                    },
                    'children': {
                        'user': rusage_children.ru_utime,
                        'system': rusage_children.ru_stime,
                    }
                },
                'greenlets': greenlets_count
            },
            'rpc': self.daemon.rpc.stats,
            'db': {
                'sqlite': dbstatus,
            },
            'file_cache': {
                'resources_count': dbcounters['resource_cnt'],
            },
            'announcer': announcer_stats,
        }

    def evaluate(self, job, code):
        sock = job._RPCJob__conn._Connection__sock.sock
        user_id = getpeerid(sock)[0]

        if user_id == 0:
            ret = None
            exec code
            return ret
        else:
            raise Exception('You are not root')

    def query(self, sql):
        self.daemon.active.wait()
        return self.daemon.db.query(sql, log=False)

    def dbbackup(self):
        self.daemon.active.wait()
        with self.daemon.db.deblock.lock('database backup'):
            self.daemon.db.backup()

    def resource_list(self):
        self.daemon.active.wait()
        result = []
        for uid in self.daemon.db.query_col('SELECT id FROM resource'):
            result.append(uid)
        return result

    def resource_remove(self, uid):
        self.daemon.active.wait()
        self.daemon.resource_mngr.remove_resource(uid)
