#!/skynet/python/bin/python

from __future__ import absolute_import

from . import monkey_patch as skbn_monkey_patch

skbn_monkey_patch()

import argparse

import gevent
try:
    import gevent.coros as coros
except ImportError:
    import gevent.lock as coros

from py import std
import os
import py
import signal
import sys
import time

import struct

try:
    from psutil import Process
except ImportError:
    Process = None  # noqa

from library.sky.hostresolver import Resolver as BCalcResolver

from ..common import Path, determine_app_path

from .component import Component
from .db import Database
from .greenish.deblock import Deblock
from .legacy import migrate_workdir_if_needed
from .logger import log, initialize as initialize_log
from .metrics import MetricPusher
from .resource.manager import ResourceManager
from .procrunner import ProcRunner
from .rpc import SkyboneRPC, SkyboneProcRPC, AdminRPC, NetproxyRPC, InternalRPC, SkybitRPC, DiskIORPC
from .skbn import encryption
from .utils import SlottedDict, update_proc_title, fastbonize_hostname, gen_uds

from ..kernel_util.sys.gettime import monoTime
from ..rpc.server import RPC, Server as RPCServer


class CopierDaemon(Component):
    session = None
    fastbone_mode = None

    def __init__(self, uid, app_path, cfg, (cfg_file, cfg_mtime), user, workdir):  # rpcPort, dataPort, workdir):
        super(CopierDaemon, self).__init__(logname='')
        self.log.debug('CopierDaemon init...')
        self.log.debug('  nofile: %r', std.resource.getrlimit(std.resource.RLIMIT_NOFILE))
        if os.uname()[0].lower() != 'darwin' and sys.platform != 'cygwin':
            self.log.debug('  uids  : %r', std.os.getresuid())
        else:
            self.log.debug('  uids  : (%d, %d)', os.getuid(), os.geteuid())

        try:
            self.port_offset = int(os.environ.get('SKYNET_PORTOFFSET', 0))
        except:
            self.port_offset = 0

        self.singletone_port = cfg.rpc_port + self.port_offset
        self.rpc_sock = Path(workdir).join('rpc.sock')
        self.rpc_sock_mvme = Path(workdir).join('rpc.sock.mvme')

        self.shouldStop = False

        self.path = app_path
        self.cfg = cfg
        self.cfg_file = cfg_file
        self.cfg_mtime = cfg_mtime
        self.user = user

        self.uid = uid
        self.ips = {'bb': None, 'bb6': None, 'fb': None, 'fb6': None}

        self.tsStarted = monoTime()

        std.random.seed()

        self.workdir = Path(workdir).ensure(dir=1)

        self.startup_stage = (time.time(), 'init')
        self.active = gevent.event.Event()
        self.active.clear()

        self._selfcheck = {'ts': 0, 'te': 0, 'running': False, 'db': False, 'netproxy': False}

        singletone_port = self.cfg.proxy.port + self.port_offset

        self.netproxy_uds_sock_path, self.netproxy_uds_sock_abstract = gen_uds(
            'skybone_%d_netproxy_data' % (singletone_port, ),
            'netproxy_data',
            self.workdir,
            allow_abstract=False
        )

        self.skybit_rpc_uds, self.skybit_rpc_uds_abstract = gen_uds(
            'skybone_%d_skybit' % (singletone_port, ),
            'skybit',
            self.workdir
        )
        self.skybit_data_uds, self.skybit_data_uds_abstract = gen_uds(
            'skybone_%d_skybit_data' % (singletone_port, ),
            'skybit_data',
            self.workdir
        )
        self.procrunner = ProcRunner(parent=self)

        self.skybit_proc = None
        self.netproxy_proc = None

        self.trackers = None
        self._tracker_hostnames = None

        self.data_counters = {
            'resource_cnt': None,
            'file_cnt': None,
            'file_size': None,
            'data_size': None
        }

        if os.uname()[0].lower() == 'linux':
            import porto
            self._porto_deblock = Deblock(keepalive=None, logger=self.log.getChild('portodeblock'))
            self._porto = self._porto_deblock.make_proxy(
                porto.Connection(
                    timeout=5,
                    auto_reconnect=True
                ),
            )
        else:
            self._porto = None

        self.porto = None

    def init_db(self):
        db_path = self.workdir.join('copier.db')
        log.debug('Opening database at %s', db_path)

        db_temp_dir = self.workdir.join('temp').ensure(dir=1)

        dbobj = Database(
            db_path.strpath,
            mmap=self._config_kmg_convert(self.cfg.db.mmap),
            temp=db_temp_dir.strpath,
            parent=self
        )

        self.db_deblock = Deblock(keepalive=None, logger=dbobj.log.getChild('deblock'))
        self.db = self.db_deblock.make_proxy(dbobj, put_deblock='deblock')

        self.startup_stage = (time.time(), 'init_db')
        self.db.open(check=self.db.CHECK_IF_DIRTY, force=self.cfg.db.get('drop_db_on_error', True))
        self.db.set_debug(sql=True, transactions=True)

        self.startup_stage = (time.time(), 'migrate_db')
        migrations_path = self.path.join('share', 'db', 'main')

        fw_migrations = {}
        bg_migrations = {}

        if migrations_path.check(exists=1, dir=1):
            for migration_file in migrations_path.listdir():
                if not (migration_file.basename.endswith('.sql') or migration_file.basename.endswith('.py')):
                    continue
                if migration_file.basename.startswith('fw_'):
                    target = fw_migrations
                elif migration_file.basename.startswith('bw_'):
                    target = bg_migrations
                else:
                    continue

                version = int(migration_file.basename.split('_', 2)[1])
                target[version] = migration_file

        self.db.migrate(fw_migrations, bg_migrations)

        self.startup_stage = (time.time(), 'maintain_db')

        def _maintain(**kwargs):
            try:
                with self.db.deblock.lock('dbmaintain'):
                    self.db.maintain(**kwargs)
            except Exception as ex:
                self.log.critical('Maintainance error: %s', str(ex))

        _maintain(vacuum=True, analyze=True, grow=True)

        def _mnt(t, **kwargs):
            while 1:
                gevent.sleep(t)
                if self.shouldStop:
                    break
                _maintain(**kwargs)

        gevent.spawn(_mnt, 3600, vacuum=True, analyze=True)
        gevent.spawn(_mnt, 10, grow=True)

        if not self.uid:
            forced_uid = False
            uid_info = self.db.query_one_col('SELECT value_text FROM dbmaintain WHERE key = ?', ['uid'])
            if uid_info:
                uid_version = int(uid_info[0])
                if uid_version == 1 and uid_info[1] == ':':
                    self.uid = uid_info[2:]
                else:
                    self.uid = None
            else:
                self.uid = None
        else:
            forced_uid = True

        if not self.uid or forced_uid:
            if not self.uid:
                self.uid = struct.pack('!Q', std.random.randint(0, 2 ** 64)).encode('hex')
            else:
                assert len(self.uid) == 16, 'Invalid uid length'
                assert isinstance(struct.unpack('!Q', self.uid.decode('hex'))[0], int), 'Invalid uid'

                self.log.info('Using forced uid: %s', self.uid)

            self.db.query(
                'REPLACE INTO dbmaintain VALUES (?, ?, null)',
                ['uid', '1:' + self.uid]
            )

    def resolve_tracker_hosts(self):
        from .utils import getaddrinfo_g as getaddrinfo
        from _socket import getaddrinfo as _getaddrinfo

        # Will be initialized lazily
        resolver = None

        self.trackers = {'coord': {}}

        for expr, port in ((self.cfg.coord_hosts, self.cfg.coord_hosts_port), ):
            if isinstance(expr, (list, tuple)):
                hostnames = expr
            else:
                if self._tracker_hostnames is None:
                    if resolver is None:
                        # Lazy init
                        resolver = BCalcResolver(check_hosts=False)
                    log.debug('Resolving blinov calc: %s...', expr)
                    hostnames = resolver.resolveHosts(expr)
                    self._tracker_hostnames = hostnames
                    log.debug('  resolved: %s', hostnames)
                else:
                    hostnames = self._tracker_hostnames

            log.debug('Resolving ips from...')
            ipv4_count, ipv6_count = 0, 0

            families = []

            if self.ips['bb6']:
                families.append(std.socket.AF_INET6)

            if self.ips['bb']:
                families.append(std.socket.AF_INET)

            if not families:
                if not self.cfg.network.use_v4:
                    families = [std.socket.AF_INET6]
                else:
                    families = [std.socket.AF_INET6, std.socket.AF_INET]

            for hostname in hostnames:
                ips = []

                for family in families:
                    try:
                        for ipinfo in getaddrinfo(
                            hostname, port,
                            family, std.socket.SOCK_STREAM, std.socket.IPPROTO_IP,
                            getaddrinfo=_getaddrinfo
                        ):
                            ip = ipinfo[4][0]
                            if ipinfo[0] == std.socket.AF_INET:
                                ipv4_count += 1
                                ip = '::ffff:%s' % (ip, )
                            else:
                                ipv6_count += 1

                            ipport = (ip, ipinfo[4][1])
                            if ipport not in ips:
                                ips.append(ipport)
                    except std.socket.gaierror as ex:
                        if ex.errno in (
                            getattr(std.socket, 'EAI_NONAME', None),
                            getattr(std.socket, 'EAI_NODATA', None),
                        ):
                            log.warning('Unable to resolve %s:%d: %s, skipping', hostname, port, ex)
                        else:
                            raise

                self.trackers['coord'][hostname] = ips
            log.debug('  resolved: %s', self.trackers['coord'])

    def _config_kmg_convert(self, value):
        if isinstance(value, basestring):
            meth = (float if '.' in value else int)
            if value.endswith('K'):
                value = 1024 * meth(value.rstrip('K'))
            elif value.endswith('M'):
                value = 1024 * 1024 * meth(value.rstrip('M'))
            elif value.endswith('G'):
                value = 1024 * 1024 * 1024 * meth(value.rstrip('G'))

        return value

    def get_path(self, postfix):
        return self.path.join(postfix).strpath

    def configure_fastbone_mode(self, hostname=None):
        from .utils import getaddrinfo_g as getaddrinfo
        from _socket import getaddrinfo as _getaddrinfo

        self.ips['bb'] = None
        self.ips['bb6'] = None
        self.ips['fb'] = None
        self.ips['fb6'] = None

        if hostname is None:
            hostname = std.socket.gethostname()
        if '.' not in hostname:
            fqhostname = std.socket.getfqdn(hostname)
            if sys.platform == 'cygwin':
                # on cygwin getfqdn can return all DNS prefixes as an address, so we cut it here
                fqhostname = fqhostname.split()[0]
            if fqhostname != 'any.yandex.ru':
                hostname = fqhostname

        log.debug('Detecting our ips (hostname: %s)', hostname)
        found_ipv6 = False

        try:
            log.debug('  resolve %s...' % (hostname, ))
            ips = getaddrinfo(hostname, 0, 0, std.socket.SOCK_STREAM, getaddrinfo=_getaddrinfo)
            for family, _, _, _, ipinfo in ips:
                if family == std.socket.AF_INET:
                    if ipinfo[0] != '127.0.0.1':
                        if not self.cfg.network.use_v4:
                            self.ips['bb'] = None
                            log.debug('    detected bb: %s (but we a not using v4 stack, ignoring)', ipinfo[0])
                        else:
                            self.ips['bb'] = ipinfo[0]
                            log.debug('    detected bb: %s', ipinfo[0])
                elif family == std.socket.AF_INET6:
                    if ipinfo[0] != '::1':
                        if not self.cfg.network.use_v6:
                            self.ips['bb6'] = None
                            log.debug('    detected bb6: %s (but we are not using v6 stack, ignoring)', ipinfo[0])
                        else:
                            self.ips['bb6'] = ipinfo[0]
                            log.debug('    detected bb6: %s', ipinfo[0])
                            found_ipv6 = True
        except std.socket.gaierror as ex:
            if ex.errno in (
                getattr(std.socket, 'EAI_NONAME', None),  # NOT FOUND
                getattr(std.socket, 'EAI_NODATA', None),  # NO ADDRESS ASSOCIATED WITH HOSTNAME (LINUX)
            ):
                log.debug('    not found (errno: %d)', ex.errno)
            else:
                # Allow to copier to die completely in case of any
                # dns errors here. Because starting without fastbone
                # support is critical
                raise

        if self.cfg.network.search_bb6_in_search_yandex_net:
            if not found_ipv6 and hostname.startswith('ws') and hostname.endswith('.yandex.ru'):
                log.debug('  ipv6 was not found, will try legacy ipv6 search...')
                second_hostname = hostname[:-(len('.yandex.ru'))] + '.search.yandex.net'
                try:
                    log.debug('  resolve %s...' % (second_hostname, ))
                    ips = getaddrinfo(second_hostname, 0, 0, std.socket.SOCK_STREAM, getaddrinfo=_getaddrinfo)

                    for family, _, _, _, ipinfo in ips:
                        if family == std.socket.AF_INET:
                            if ipinfo[0] != '127.0.0.1':
                                if self.ips['bb'] != ipinfo[0]:
                                    log.debug(
                                        '    detected bb ip dont match already detected: '
                                        '%s != %s, ignoring extended hostname',
                                        self.ips['bb'], ipinfo[0]
                                    )
                                    found_ipv6 = False
                                    break
                                log.debug('    detected bb: %s', ipinfo[0])
                        elif family == std.socket.AF_INET6:
                            if ipinfo[0] != '::1':
                                found_ipv6 = ipinfo[0]
                                log.debug('    detected bb6: %s', ipinfo[0])

                    if found_ipv6:
                        self.ips['bb6'] = found_ipv6
                except std.socket.gaierror as ex:
                    if ex.errno in (
                        getattr(std.socket, 'EAI_NONAME', None),  # NOT FOUND
                        getattr(std.socket, 'EAI_NODATA', None),  # NO ADDRESS ASSOCIATED WITH HOSTNAME (LINUX)
                    ):
                        log.debug('    not found (errno: %d)', ex.errno)
                    else:
                        # Allow to copier to die completely in case of any
                        # dns errors here. Because starting without fastbone
                        # support is critical
                        raise

        self.hostname = hostname
        self.fbhostname = None

        fb_hostname, fb_ips = fastbonize_hostname(hostname, log)

        if fb_ips:
            if fb_ips[0]:
                self.ips['fb'] = fb_ips[0][0]
            if fb_ips[1]:
                self.ips['fb6'] = fb_ips[1][0]

            self.fastbone_mode = True

            if fb_hostname:
                self.fbhostname = fb_hostname

            return

        self.fastbone_mode = False

    @Component.green_loop(logname='housekeep.dtct_fb')
    def _redetect_fastbone(self, log):
        run_every = 60 * 30
        first_run_delay = 600

        if monoTime() - self.tsStarted > first_run_delay:
            old_fastbone_mode = self.fastbone_mode
            self.configure_fastbone_mode()

            if self.fastbone_mode != old_fastbone_mode:
                log.warning(
                    'Fastbone mode changed %r => %r, exiting',
                    old_fastbone_mode, self.fastbone_mode
                )
                self.stop()
                return run_every

            old_trackers = self.trackers.copy()
            self.resolve_tracker_hosts()

            if old_trackers != self.trackers:
                log.warning('Trackers has been changed:')
                log.warning('  old %r', old_trackers)
                log.warning('  new %r', self.trackers)
                self.stop()
                return run_every
            else:
                log.debug('Trackers reresolved ok')

        return run_every  # run every 30 mins

    @Component.green_loop(logname='housekeep.data_counters')
    def _data_counters(self):
        try:
            self.active.wait()

            ts = time.time()

            with self.db(debug_sql=False):
                dc = self.data_counters
                dc['ready'] = True
                dc['resource_cnt'] = self.db.query_one_col('SELECT COUNT(*) FROM resource')
                dc['file_cnt'] = self.db.query_one_col('SELECT COUNT(*) FROM file')
                dc['data_size'] = self.db.query_one_col('SELECT SUM(size) FROM data')
                dc['file_size'] = self.db.query_one_col(
                    'SELECT SUM(d.size) FROM file f JOIN data d ON d.id = f.data'
                )

                if dc['data_size'] is None:
                    dc['data_size'] = 0

                if dc['file_size'] is None:
                    dc['file_size'] = 0

            te = time.time()
            spent = te - ts

            next_sleep = max(60, min(int(spent) * 10, 1800))
            log.debug('Spent %.2fs on counters, next run in %ds', spent, next_sleep)

            return next_sleep
        except Exception as ex:
            log.warning('Unable to calculate counters: %s', ex)
            return 300

    @Component.green_loop(logname='housekeep.selfcheck')
    def _selfcheck(self, log):
        self._selfcheck['ts'] = time.time()
        self._selfcheck['running'] = True

        try:
            netproxy = False
            db = False

            if self.netproxy_proc:
                netproxy = self.netproxy_proc.ready.wait(timeout=10)
            else:
                netproxy = False

            db_locked_for = self.db.deblock.locked_for

            if db_locked_for > 300:
                db_lock_reason = self.db.deblock.lock_reason
                if db_lock_reason:
                    log.warning('Db warning: (locked for %ds, reason: %s)', db_locked_for, db_lock_reason)
                    db = True
                else:
                    log.critical('Db fail (locked for %ds without reason)', db_locked_for)
                    db = False
            else:
                if db_locked_for > 0:
                    log.debug('Db ok (locked for %ds, reason: %s)', db_locked_for, self.db.deblock.lock_reason)
                    db = True
                else:
                    log.debug('Db ok (not locked)')
                    log.debug('Db waiters: %d', self.db.deblock.lock_waiters)
                    db = self.db.ping()
                    try:
                        if db:
                            # Will raise an err if _vtin not exists
                            self.db.query('SELECT value FROM _vtin', log=False)
                    except:
                        log.debug('Db not ok (missing _vtin)')
                        db = False
        finally:
            self._selfcheck['netproxy'] = netproxy
            self._selfcheck['db'] = db
            self._selfcheck['running'] = False
            self._selfcheck['te'] = time.time()

        log.debug(repr(self._selfcheck))
        return 300

    @Component.green_loop(logname='portoconn')
    def _porto_connect(self, log):
        if not os.uname()[0].lower() == 'linux':
            self.porto = None
            return 86400

        if not self._porto:
            self.porto = None
            return 60

        try:
            self._porto.TryConnect()
            if not self.porto:
                log.info('Successfully connected to porto')
            self.porto = self._porto
        except Exception as ex:
            if self.porto:
                log.warning('Unable to connect to porto: %s', ex)
            self.porto = None

        return 60

    def ping(self, wait_active=None, extended=False):
        if wait_active is None and self.startup_stage[1] != 'done':
            self.log.debug('startup: %s since %s', self.startup_stage[1], time.ctime(self.startup_stage[0]))
            if not extended:
                return True
            return {
                'active': False,
                'stage': self.startup_stage,
                'fastbone_mode': self.fastbone_mode
            }

        # simplest RPC func ever :)
        active = self.active.isSet()
        if wait_active is not None:
            # If we need to wait active state -- wait it
            if not active:
                self.active.wait(timeout=wait_active)
                active = self.active.isSet()
            working = active
        else:
            working = True

        if working:
            if time.time() - self._selfcheck['te'] < 600:
                self.log.debug('selfcheck result is %d seconds old', time.time() - self._selfcheck['ts'])
                working = self._selfcheck['db'] and self._selfcheck['netproxy']
            else:
                if time.time() - self._selfcheck['ts'] > 1800:
                    self.log.critical('selfcheck started %d seconds ago', time.time() - self._selfcheck['ts'])
                    working = False
                else:
                    self.log.warning(
                        'selfcheck is not ready (started %d seconds ago)',
                        time.time() - self._selfcheck['ts']
                    )
                    working = True

        if working:
            if self.cfg_mtime != os.stat(self.cfg_file).st_mtime:
                self.stop()

        self.log.debug('ping result: %r', working)

        if not extended:
            return working

        return {
            'active': working,
            'stage': self.startup_stage,
            'fastbone_mode': self.fastbone_mode
        }

    def stop(self):
        # This will say to us "should stop" and quit immediately
        # avoiding blocking on RPC request
        self.shouldStop = True
        return True

    def shutdown(self):
        self.shouldStop = True  # ensure this set

        super(CopierDaemon, self).stop()
        self.resource_mngr.join()

        self.db.close()
        self.db.deblock.stop()

        if self._porto_deblock:
            self._porto_deblock.stop()

        if len(std.threading.enumerate()) > 1:
            # Give a last try all threads to stop gracefully after analyzing .shouldStop
            # flag. Wait max 5 seconds for them.
            all_threads_stopped = False

            for i in range(50):
                if len(std.threading.enumerate()) == 1:
                    all_threads_stopped = True
                    break
                gevent.sleep(0.1)

            if not all_threads_stopped:
                log.critical('Cant shutdown properly -- some threads still alive!')
                os._exit(1)

        # Last stage -- stop RPC server
        log.normal('Shutdown success')

    def start(self):
        log.debug('Starting RPC server serving %r on %s' % (self, self.rpc_sock))

        if sys.platform == 'cygwin':
            # ignore memory limits for cygwin
            mem_soft_limit = None
            mem_hard_limit = None
            mem_guard = None
        else:
            mem_soft_limit = self._config_kmg_convert(self.cfg.limits.memory.soft)
            mem_hard_limit = self._config_kmg_convert(self.cfg.limits.memory.hard)
            mem_guard = self._config_kmg_convert(self.cfg.limits.memory.guard)

        if 'guard_timed' in self.cfg.limits.memory:
            try:
                guard_timed = self.cfg.limits.memory['guard_timed']
                guard_timed['memory'] = self._config_kmg_convert(guard_timed['memory'])
                interval = guard_timed['interval']

                for key, value in interval.items():
                    if isinstance(value, basestring) and value.count(':') == 2:
                        interval[key] = [int(_) for _ in value.split(':')]
                    else:
                        raise Exception('Invalid type for interval item %s: %r' % (key, value))
            except Exception as ex:
                log.warning('Ignoring timed interval: %s', ex)
                guard_timed = None
        else:
            guard_timed = None

        if mem_soft_limit or mem_hard_limit:
            if not mem_soft_limit:
                mem_soft_limit = mem_hard_limit
            if not mem_hard_limit:
                mem_hard_limit = mem_soft_limit

            memSLimitCur, memHLimitCur = std.resource.getrlimit(std.resource.RLIMIT_AS)
            if (
                (memSLimitCur == -1 or memSLimitCur > mem_soft_limit) and
                (memHLimitCur == -1 or memHLimitCur > mem_hard_limit)
            ):
                log.debug(
                    'Set memory limits (RLIMIT_AS) to (%d -> %s (%d), %d -> %s (%d))' % (
                        memSLimitCur, self.cfg.limits.memory.soft, mem_soft_limit,
                        memHLimitCur, self.cfg.limits.memory.hard, mem_hard_limit
                    )
                )
                try:
                    std.resource.setrlimit(std.resource.RLIMIT_AS, (mem_soft_limit, mem_hard_limit))
                except ValueError as ex:
                    log.warning('Setting RLIMIT_AS failed: %s, ignoring', ex)
            else:
                log.debug(
                    'Not setting memory limits (RLIMIT_AS), currently have lower than needed (%d, %d)' % (
                        memSLimitCur, memHLimitCur
                    )
                )

        if (mem_guard or guard_timed) and Process is not None:
            def _memguard():
                while not self.shouldStop:
                    try:
                        meminfo = Process(std.os.getpid()).get_memory_info()
                    except Exception as ex:
                        log.warning('Failed to get memory usage: %s' % (ex, ))
                        gevent.sleep(60)
                        continue

                    log.debug('RSS memory usage is %d bytes (max allowed %d bytes)' % (meminfo.rss, mem_guard))
                    if mem_guard and meminfo.rss > mem_guard:
                        log.critical('Too much rss memory usage (%d > %d), exiting.' % (meminfo.rss, mem_guard))
                        self.stop()
                        signal.alarm(300)  # alarm to 5 mins
                        break
                    elif guard_timed and (meminfo.rss > guard_timed['memory']):
                        start, end = guard_timed['interval']['start'], guard_timed['interval']['end']
                        dt_now = std.datetime.datetime.now()
                        hms = [dt_now.hour, dt_now.minute, dt_now.second]
                        matched = False
                        if start < end:
                            if end >= hms >= start:
                                matched = True
                        elif start > end:
                            if hms >= start or hms <= end:
                                matched = True

                        if matched:
                            log.critical('Too much rss memory usage (%d > %d), interval %s - %s, exiting.' % (
                                meminfo.rss, mem_guard,
                                start, end
                            ))
                            self.stop()
                            signal.alarm(300)  # alarm to 5 mins
                            break

                        gevent.sleep(1)
                    else:
                        gevent.sleep(300)

            gevent.spawn(_memguard)

        try:
            self.singletone_sock = std.socket.socket(std.socket.AF_INET, std.socket.SOCK_STREAM)
            self.singletone_sock.setsockopt(std.socket.SOL_SOCKET, std.socket.SO_REUSEADDR, 1)
            self.singletone_sock.bind(('127.0.0.1', self.singletone_port))
            self.singletone_sock.listen(1)
        except std.socket.error as err:
            import errno
            if errno.errorcode.get(err.errno, None) == 'EADDRINUSE':
                error_message = 'ERROR: Address 127.0.0.1:%d already in use! Quitting...' % self.singletone_port
                print >> std.sys.stderr, error_message
                log.critical(error_message)
                std.sys.exit(1)
            raise

        try:
            self.rpc_sock.remove()
        except py.error.ENOENT:
            pass

        try:
            self.rpc_sock_mvme.remove()
        except py.error.ENOENT:
            pass

        self.internal_rpc = InternalRPC(self)
        self.diskio_rpc = DiskIORPC(self)
        self.netproxy_rpc = NetproxyRPC(self)
        self.skybit_rpc = SkybitRPC(self)
        self.skbn_rpc = SkyboneRPC(self)
        self.skbn_proc_rpc = SkyboneProcRPC(self)
        self.skbn_rpc_admin = AdminRPC(self)

        self.rpc = RPC(self.log)
        self.rpc_server = RPCServer(self.log, backlog=10, max_conns=1000, unix=self.rpc_sock.strpath)
        self.rpc_server.register_connection_handler(self.rpc.get_connection_handler())

        self.rpc.mount(self.ping)
        self.rpc.mount(self.skbn_rpc.io_semaphore, typ='full')

        # Old SH helpers
        self.rpc.mount(self.skbn_proc_rpc.lock_files, typ='full')

        # DL helpers
        self.rpc.mount(self.skbn_proc_rpc.dl_helper, typ='full')
        self.rpc.mount(self.skbn_proc_rpc.get_trycopy_waiters)

        # Old COMMON helpers
        self.rpc.mount(self.skbn_proc_rpc.cache_load_info_from_db, typ='full')
        self.rpc.mount(self.skbn_proc_rpc.cache_store_resource, typ='full')
        self.rpc.mount(self.skbn_proc_rpc.mon, typ='full')

        # See SKYDEV-1802
        self.rpc.mount(self.skbn_proc_rpc.runjob, typ='full')

        # Various admin meths
        self.rpc.mount(self.skbn_rpc_admin.status)
        self.rpc.mount(self.skbn_rpc_admin.counters)
        self.rpc.mount(self.skbn_rpc_admin.query, typ='full')
        self.rpc.mount(self.skbn_rpc_admin.dbbackup)
        self.rpc.mount(self.skbn_rpc_admin.dbcheck)
        self.rpc.mount(self.skbn_rpc_admin.evaluate, typ='full')
        self.rpc.mount(self.skbn_rpc_admin.check_resources)
        self.rpc.mount(self.skbn_rpc_admin.check_paths)
        self.rpc.mount(self.skbn_rpc_admin.notify)
        self.rpc.mount(self.skbn_rpc_admin.file_move, typ='full')
        self.rpc.mount(self.skbn_rpc_admin.clean_resources)

        self.rpc.mount(self.internal_rpc.logger, name='internal_logger', typ='full')

        self.rpc.mount(self.diskio_rpc.get_paths_by_checksum, name='io_get_paths_by_checksum', silent=True)
        self.rpc.mount(self.diskio_rpc.set_bad_paths, name='io_set_bad_paths', silent=True)
        self.rpc.mount(self.netproxy_rpc.lookup_infohash, name='netproxy_lookup_infohash', typ='full')

        self.rpc.mount(self.skybit_rpc.lookup_resource, name='skybit_lookup_resource', typ='full')

        self.rpc_server.start()

        # Check greenlets
        assert gevent.spawn(lambda: True).get(), 'Greenlets are not working correctly!'

        log.normal('%r started' % self)

        self.configure_fastbone_mode()
        self.resolve_tracker_hosts()

        self.init_db()

        self.startup_stage = (time.time(), 'init_resource_mngr')

        singletone_port = self.cfg.proxy.port + self.port_offset

        netproxy_rpc_uds, netproxy_rpc_uds_abstract = gen_uds(
            'skybone_%d_netproxy' % (singletone_port, ),
            'netproxy',
            self.workdir
        )

        dl_lim = self._config_kmg_convert(self.cfg.limits.speed.dl)
        dl_lim = dl_lim or -1
        ul_lim = self._config_kmg_convert(self.cfg.limits.speed.ul)
        ul_lim = ul_lim or -1

        tcp_send_buffer = self._config_kmg_convert(self.cfg.network.tcp.send_buffer)
        tcp_recv_buffer = self._config_kmg_convert(self.cfg.network.tcp.recv_buffer)

        self.netproxy_proc = self.procrunner.add_proc(
            'netproxy',
            [
                self.get_path('bin/skybone-netproxy'),
                '--master-uds', self.rpc_sock.strpath,
                '--rpc-uds', netproxy_rpc_uds,
            ] + (['--rpc-uds-abstract'] if netproxy_rpc_uds_abstract else []) + [
                '--uid', self.uid,
                '--desc', self.hostname,
                '--port', str(self.cfg.proxy.port + self.port_offset),
                '--port-proxy', str(self.cfg.proxy.port_proxy + self.port_offset),
                '--data-uds', self.netproxy_uds_sock_path,
            ] + (['--data-uds-abstract'] if self.netproxy_uds_sock_abstract else []) + [
                '--skbt-uds', self.skybit_data_uds,
            ] + (['--skbt-uds-abstract'] if self.skybit_data_uds_abstract else []) + [
                '--max-speed-dl', str(dl_lim),
                '--max-speed-ul', str(ul_lim),
                '--tcp-send-buffer', str(tcp_send_buffer),
                '--tcp-recv-buffer', str(tcp_recv_buffer),
                '--max-connections', str(self.cfg.limits.connections.max),
            ],
            rpc_uds=('\0' if netproxy_rpc_uds_abstract else '') + netproxy_rpc_uds,
            root=True
        )

        self.resource_mngr = ResourceManager(
            self.uid,
            self.hostname,
            self.workdir, self.db,
            trackers=self.trackers,
            data_port=self.cfg.proxy.port + self.port_offset,
            announce_port=self.cfg.announce_port + self.port_offset,
            main_lt_port=self.cfg.main_lt_port + self.port_offset,
            proxy_port=self.cfg.proxy.port_proxy + self.port_offset,
            proxy_uds=self.netproxy_uds_sock_path,
            proxy_uds_abstract=self.netproxy_uds_sock_abstract,
            skybit_data_uds=('\0' if self.skybit_data_uds_abstract else '') + self.skybit_data_uds,
            ips=self.ips,
            dl_binary=self.get_path('bin/skybone-dl-legacy'),
            hasher_binary=self.get_path('bin/skybone-hasher'),
            netproxy_proc=self.netproxy_proc,
            net_priorities=self.cfg.priorities,
            max_connections=self.cfg.limits.connections.max,
            dl_write_cache_size=self._config_kmg_convert(self.cfg.dlrs.write_cache_size),
            dl_write_cache_expiry=self.cfg.dlrs.write_cache_expiry,
            dl_preallocate_files=self.cfg.dlrs.preallocate_files,
            filechecker_blacklist=self.cfg.filechecker.blacklist,
            deduplicate_nocheck=self.cfg.deduplicate_nocheck,
            parent=self
        )

        yasmagent_url = self.cfg.get('yasmagent_url')
        self.metric_pusher = MetricPusher(yasmagent_url, parent=self)

        migrate_workdir_if_needed(self)

        self.startup_stage = (time.time(), 'init_subprocs')

        compression_codecs = ','.join(self.cfg.get('allowed_compression_codecs', []))
        encryption_modes = ','.join(self.cfg.get('allowed_encryption_modes', [encryption.PLAIN]))
        self.skybit_proc = self.procrunner.add_proc(
            'skybit',
            [
                self.get_path('bin/skybone-skybit'),
                '--master-uds', self.rpc_sock.strpath,
                '--rpc-uds', self.skybit_rpc_uds,
            ] + (['--rpc-uds-abstract'] if self.skybit_rpc_uds_abstract else []) + [
                '--child',
                '--data-uds', self.skybit_data_uds,
            ] + (['--data-uds-abstract'] if self.skybit_data_uds_abstract else []) + [
                '--proxy-uds', self.netproxy_uds_sock_path,
            ] + (['--proxy-uds-abstract'] if self.netproxy_uds_sock_abstract else []) + [
                '--world-uid', self.resource_mngr.announcer.uid,
                '--world-desc', self.resource_mngr.desc,
            ] + (['--direct-io'] if self.cfg.seed_direct_io else []) +
                # limit of incoming connections per handle
                (['--seeder-conn-limit', str(self.cfg.seeder_conn_limit)] if 'seeder_conn_limit' in self.cfg else []) +
                (['--yasmagent-url', yasmagent_url] if yasmagent_url else []) + 
                (['--allowed-compression-codecs', compression_codecs] if compression_codecs else []) + [
                '--allowed-encryption-modes', encryption_modes,
            ],
            rpc_uds=('\0' if self.skybit_rpc_uds_abstract else '') + self.skybit_rpc_uds,
        )

        super(CopierDaemon, self).start()

        self.skybit_proc.ready.wait()
        self.netproxy_proc.ready.wait()

        self.startup_stage = (time.time(), 'done')

        self.active.set()
        log.debug('Allowed incoming connections')

        try:
            fail_count = 0

            while not self.shouldStop:
                try:
                    gevent.sleep(1)
                    fail_count = 0
                except IOError as err:
                    fail_count += 1
                    if fail_count >= 1024:
                        raise
                    log_level = log.debug if fail_count < 1000 else log.critical
                    log_level('Main daemon hub.switch() error #%d: %s' % (fail_count, err))
        except KeyboardInterrupt:
            log.normal('Caught KeyboardInterrupt (SIGINT), shutting down...')
            self.active.clear()
            self.shutdown()
        else:
            log.normal('ShouldStop flag set, shutting down...')
            self.active.clear()
            self.shutdown()


def load_config(filename, fmt):
    import yaml

    cfg_mtime = os.stat(filename).st_mtime
    cfg = yaml.safe_load(open(filename, 'rb'))
    cfg_mtime_2 = os.stat(filename).st_mtime

    if cfg_mtime != cfg_mtime_2:
        raise Exception('Config mtime changed during loading!')

    if fmt == 'skycore':
        cfg = cfg['subsections']['skynet']['subsections']['services']['subsections']['copier']['config']['config']

    return SlottedDict(cfg), cfg_mtime


def _client(sock):
    from ..rpc.client import RPCClient

    cli = RPCClient(sock, None)
    return cli


def main():
    app_path = determine_app_path()

    parser = argparse.ArgumentParser()
    parser.add_argument('--uid', required=False, help='Force copier to use this uid. Must be 16 bytes hex string!')
    parser.add_argument('-c', '--config', required=True, help='Config file to use')
    parser.add_argument('-cfmt', choices=['ctl', 'skycore'], default='ctl')

    parser.add_argument(
        '-d', '--workdir', required=False, default=os.path.abspath('rbtorrent'),
        help='Working directory'
    )
    parser.add_argument('--skycore-ping', default=None)
    parser.add_argument('-u', '--user', required=False)

    args = parser.parse_args()

    if args.skycore_ping:
        sock = args.skycore_ping + '/rpc.sock'

        try:
            result = _client(sock).call('ping', extended=True).wait()
            print('ping result: %r' % (result, ))
        except Exception as ex:
            print('ping error: %s' % (ex, ))
            return 1

        if result['stage'][1] != 'done':
            in_stage = time.time() - result['stage'][0]
            if in_stage < 86400:
                print(
                    'just %d secs in %s stage, bypassing all checks' % (
                        time.time() - result['stage'][0],
                        result['stage'][1]
                    )
                )
                return 0
            else:
                print(
                    '%d secs in %s stage, too long, reporting as running=False' % (
                        time.time() - result['stage'][0],
                        result['stage'][1]
                    )
                )
                return 1

        active = result.get('active', False)
        if not active:
            print('not active, running=False')
            return 1

        import socket
        import errno

        port = 6881

        def _check_sock(name, addr, family=socket.AF_INET):
            print('check %s socket' % (name, ))

            try:
                sock = socket.socket(family, socket.SOCK_STREAM)
                sock.settimeout(10)
                sock.connect((addr, port))
            except socket.error as ex:
                print('unable to connect to %s socket: %s, running=False' % (name, ex))
                return False
            except Exception as ex:
                print('unknown error while connecting to %s sockect: %s, running=False' % (name, ex))
                return False
            else:
                try:
                    sock.send('a' * (49 + 19))
                    data = sock.recv(1)
                    if data != '':
                        print(
                            '%s socket didnt closed connection (result: %r), running=False' % (
                                name, data
                            )
                        )
                        return False  # copier must close connection asap
                except socket.error as ex:
                    if ex.errno != errno.ECONNRESET:  # con reset by peer
                        print('%s socket not good error: %s, running=False' % (name, ex))
                        return False
                except Exception as ex:
                    print('%s socket unknown error: %s, running=False' % (name, ex))
                    return False

                print('%s sockect ok' % (name, ))
                sock.close()

                return True

        if not _check_sock('bb', '127.0.0.1'):
            return 1

        if result.get('fastbone_mode') is True:
            if not _check_sock('fb', '::1', family=socket.AF_INET6):
                return 1
        else:
            print('bypassing fb socket check')

        print('(finish) runing: True')
        return 0

    if args.user:
        from ..kernel_util.sys.user import userPrivileges as user_privileges  # noqa
    else:
        import contextlib

        @contextlib.contextmanager
        def user_privileges(user):  # noqa
            yield

    # Force kernel.util.sys.user to use green subprocess module
    from . import subprocess_gevent
    std.sys.modules['subprocess'] = subprocess_gevent
    for name, mod in std.sys.modules.iteritems():
        if name.startswith('skybone.kernel_util.sys.user._'):
            if hasattr(mod, 'subprocess'):
                mod.subprocess = subprocess_gevent

    cur_soft, cur_hard = std.resource.getrlimit(std.resource.RLIMIT_NOFILE)[:2]

    if cur_soft < 4096:
        if os.getuid() == 0:
            try:
                std.resource.setrlimit(std.resource.RLIMIT_NOFILE, (4096, 4096))
            except ValueError as ex:
                log.warning('Unable to set NOFILE rlimit with uid=0: %s', ex)
                log.warning('Will keep current soft=%d hard=%d', cur_soft, cur_hard)
    else:
        if cur_hard < 4096:
            target_soft = cur_hard
        else:
            target_soft = 4096

        try:
            std.resource.setrlimit(std.resource.RLIMIT_NOFILE, (target_soft, cur_hard))
        except ValueError as ex:
            log.warning(
                'Unable to set NOFILE rlimit to soft=%d hard=%d with uid!=0: %s',
                target_soft, cur_hard, os.getuid()
            )
            log.warning('Will keep current soft=%d hard=%d', cur_soft, cur_hard)

    with user_privileges(user=args.user):
        std.sys.modules['subprocess'] = subprocess_gevent._subprocess

        cfg, cfg_mtime = load_config(args.config, args.cfmt)

        update_proc_title(cfg, 0, 0)

        # Dont rely on umask we have, because we will play with
        # chmodding files.
        os.umask(cfg.umask)

        try:
            initialize_log()
            daemon = CopierDaemon(
                args.uid, app_path,
                cfg, (args.config, cfg_mtime),
                user=args.user, workdir=args.workdir
            )
            daemon.start()
        except KeyboardInterrupt:
            log.debug('interrupted!')
            print >> std.sys.stderr, 'interrupted!'
            os._exit(1)
        except Exception as e:
            import traceback
            log.debug('Unhandled daemon exception: %s--\nQuit immediately!' % traceback.format_exc(e))

            gevent.sleep(1)

            os._exit(1)

    return 0


if __name__ == '__main__':
    # Run copier/rbtorrent/bin/rbtorrent instead of this!
    std.sys.exit(main())
