import contextlib
import time

try:
    import gevent.coros as coros
except ImportError:
    import gevent.lock as coros

from ..component import Component


class PathCache(Component):
    def __init__(self, master_cli, parent=None):
        super(PathCache, self).__init__(logname='pcache', parent=parent)

        self.master_cli = master_cli
        self.cache = {}  # md5 => [lock, ttl, {path: (mtime, chktime)}]
        self.bads = {}   # md5 => set(paths)

        self.stats = {
            'miss': 0,
            'hit': 0,
        }

    @contextlib.contextmanager
    def md5lock(self, md5):
        if md5 not in self.cache:
            self.cache[md5] = [coros.Semaphore(1), None, 0]

        with self.cache[md5][0]:
            yield (self.cache[md5][1], self.cache[md5][2])
            def linkcount(sem):
                if hasattr(sem, 'linkcount'):
                    return sem.linkcount()
                return len(sem._links)
            if not linkcount(self.cache[md5][0]) and not self.cache[md5][1]:
                # If there is no waiters and no cached paths -- drop this one
                if time.time() > self.cache[md5][2]:
                    self.cache.pop(md5)

    def get_paths(self, md5):
        with self.md5lock(md5) as (paths, ttl):
            if time.time() < ttl and paths:
                # only if ttl is not happen yet and we dont cache
                # negative lookups at all
                self.stats['hit'] += 1
                return set(paths.keys())

            self.stats['miss'] += 1

            if md5 in self.bads:
                bad_paths = self.bads.pop(md5)
                self.master_cli.call(
                    'io_set_bad_paths', md5, bad_paths
                ).wait()

            paths = self.master_cli.call(
                'io_get_paths_by_checksum', md5
            ).wait()

            paths = dict([(path, (mtime, chktime)) for path, mtime, chktime in paths])

            self.cache[md5][1] = paths
            self.cache[md5][2] = time.time() + 60

            return set(paths.keys())

    def bad_file(self, fn, md5, mtime):
        with self.md5lock(md5) as (paths, ttl):
            stored_mtime, stored_chktime = self.cache[md5][1].get(fn, (None, None))
            if stored_mtime is None:
                # No such file in cache -- just do nothing
                return

            self.bads.setdefault(md5, {})

            if fn not in self.bads[md5]:
                self.bads[md5][fn] = (stored_mtime, stored_chktime, mtime)

            paths.pop(fn, None)

            if not paths:
                # If there is no paths left in this md5 cache -- set ttl to 0
                # and md5 cache will be completely whiped after md5lock exits
                self.cache[md5][2] = 0

    @Component.green_loop
    def stats_printer(self, log):
        log.debug(repr(self.stats))
        return 60

    @Component.green_loop
    def send_bads(self, log):
        for md5 in self.bads.iterkeys():
            break
        else:
            return 1

        bad_paths = self.bads.pop(md5)

        self.master_cli.call(
            'io_set_bad_paths', md5, bad_paths
        ).wait()

        return 0
