from __future__ import absolute_import

from functools import wraps

import gevent
from gevent.greenlet import Greenlet as GeventGreenlet

from api.copier import errors

from .logger import log

from ..kernel_util.errors import formatException, saveTraceback


class LinkedExited(Exception):
    pass


class LinkedFailed(LinkedExited):
    """Copied from https://github.com/gevent/gevent/blob/0.13.6/gevent/greenlet.py#L578"""

    msg = "%r failed with %s"

    def __init__(self, source):
        exception = source.exception
        try:
            excname = exception.__class__.__name__
        except:
            excname = str(exception) or repr(exception)
        LinkedExited.__init__(self, self.msg % (source, excname))


class Greenlet(GeventGreenlet):
    def __init__(self, *args, **kwargs):
        GeventGreenlet.__init__(self, *args, **kwargs)
        if getattr(self, '_run', None) is not None:
            self._run = self._safe_run(self._run)

        try:
            self.slocal = gevent.getcurrent().slocal
        except AttributeError:
            self.slocal = {}

    def __nonzero__(self):
        return super(GeventGreenlet, self).__nonzero__()

    def _safe_run(self, func):
        @wraps(func)
        def _wrapper(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except BaseException as err:
                # Dont log some types of errors
                if isinstance(err, (
                    gevent.GreenletExit,
                    errors.Timeout,  # this is our user, not gevent timeouts.
                    errors.ResourceNotAvailable,
                )):
                    raise
                elif isinstance(err, errors.FilesystemError):
                    if err.message == 'No space left on device':
                        raise

                if not getattr(err, '_logged', False):
                    log.critical('Unhandled error during coroutine run: %s' % formatException())
                    err._logged = True

                raise
        return _wrapper

    def _report_error(self, exc_info):
        exception = exc_info[1]
        if isinstance(exception, gevent.GreenletExit):
            self._report_result(exception)
            return

        saveTraceback(exception, blocktitle='GreenLet._report_error()', excInfo=exc_info)
        self._exception = exception

        if self._links and self._notifier is None:
            self._notifier = gevent.core.active_event(self._notify_links)
