from __future__ import division

try:
    import msgpack
except ImportError:
    msgpack = None

import argparse
import hashlib
import pprint
import py
import os
import struct
import sys
import time

from api.copier import errors

from ..kernel_util.console import setProcTitle as set_proc_title  # noqa
from ..kernel_util.errors import formatException

from .utils import human_size

PIECE_SIZE = 1024 * 1024 * 4


def rehash(file_path, check_parent_pid=False, progress_func=None, change_proc_title=False):
    if change_proc_title:
        set_proc_title('skybone-hasher [init]')

    data_path = py.path.local(file_path)
    stat = data_path.stat()

    pieces = []
    num_pieces = int(stat.size / PIECE_SIZE)
    if stat.size % PIECE_SIZE:
        num_pieces += 1

    data_fp = data_path.open(mode='rb')
    timestamp = time.time()
    md5sum = hashlib.md5()

    for idx in xrange(num_pieces):
        if change_proc_title:
            set_proc_title('skybone-hasher [%.2f%% %s/%s] %s' % (
                idx * 100 / num_pieces,
                human_size(idx * PIECE_SIZE),
                human_size(stat.size),
                data_path
            ))

        data = data_fp.read(PIECE_SIZE)
        # from gevent import sleep
        # from time import sleep
        # sleep(1)

        cur_time = time.time()
        last_piece = idx == (num_pieces - 1)

        if check_parent_pid and cur_time - timestamp >= 1:
            if os.getppid() == 1:
                raise Exception('Caller died?')

        if cur_time - timestamp >= 5 or last_piece:  # check once a five seconds
            cur_stat = data_path.stat()
            if stat.mtime != cur_stat.mtime or stat.size != cur_stat.size:
                raise errors.FilesystemError('File %r mtime or size changed during hashing!' % (data_path, ))
            stat, timestamp = cur_stat, cur_time

        pieces.append(hashlib.sha1(data).digest())
        md5sum.update(data)

        if progress_func is not None:
            progress_func(idx, PIECE_SIZE if not last_piece else len(data))

    if change_proc_title:
        set_proc_title('skybone-hasher [100%%] %s' % (data_path, ))

    data_fp.close()

    return (
        PIECE_SIZE,
        tuple(pieces),
        (stat.ino, stat.size, stat.atime, stat.ctime, stat.mtime),
        md5sum.digest(),
    )


def main():
    parser = argparse.ArgumentParser()
    parser.add_argument('file')
    parser.add_argument(
        '-o', dest='output', default='text', choices=('text', 'msgpack'),
        help='Output type'
    )
    parser.add_argument(
        '--progress', action='store_true', default=False,
        help='Report hashing progress'
    )

    args = parser.parse_args()

    try:
        if args.progress:
            def _progress(idx, bytes):
                data = msgpack.dumps(('progress', idx, bytes))
                sys.stdout.write(struct.pack('!I', len(data)))
                sys.stdout.write(data)
                sys.stdout.flush()
        else:
            _progress = None

        result = rehash(args.file, check_parent_pid=True, change_proc_title=True, progress_func=_progress)
        error = None
    except BaseException as err:
        result = None
        error = (err, formatException())

    if args.output == 'text':
        if result:
            pprint.pprint(result)
        elif error:
            print error[1],

    elif args.output == 'msgpack':
        if msgpack is None:
            print >> sys.stderr, 'MsgPack not available'
            exit(2)

        if not result:
            err = [error[0].__class__.__name__] + list(error[0].__reduce__())
            err[1] = err[1].__module__ + '.' + err[1].__name__
            if len(err[2]) > 1:
                err[2] = list(err[2])
                err[2][0] = err[2][0].__module__ + '.' + err[2][0].__name__
                err[2] = tuple(err[2])
                err = tuple(err)

            result = (err, error[1])

        data = msgpack.dumps(result)
        sys.stdout.write(struct.pack('!I', len(data)))
        sys.stdout.write(data)

    if error:
        exit(1)
