from __future__ import print_function, division

import msgpack
import py
import sys
import os
import gevent

from collections import OrderedDict

from ..logger import log as root_log
from ..resource.hasher import Hasher
from ..bencode import bencode

Path = py.path.local

from pprint import pprint as pp

ZERO_MD5 = 'd41d8cd98f00b204e9800998ecf8427e'
ZERO_SHA1_PIECES = '\x92\x01\x92\x00\xda\x00(da39a3ee5e6b4b0d3255bfef95601890afd80709'


def _store_data(db, size, md5, sha1_blocks):
    db.query(
        'INSERT INTO data (id, legacy_id, size, md5, sha1_blocks) VALUES (?, ?, ?, ?, ?)', (
            md5,
            True,
            int(size),
            md5,
            buffer(sha1_blocks)
        )
    )
    return md5


def _store_file(db, path, data_id, mtime):
    db.query(
        'INSERT INTO file (path, data, mtime, chktime) VALUES (?, ?, ?, ?)', (
            path, data_id,
            int(mtime) if mtime is not None else -1,
            0
        )
    )


def migrate(workdir, db):
    log = root_log.getChild('migrate_0_1')

    log.info('Migrating workdir 0 => 1')
    state_file = workdir.join('copier.state')
    if not state_file.check(exists=1):
        log.info('no state file found, nothing to migrate')
        return

    state = msgpack.load(state_file.open(mode='rb'))
    assert state['type'] == 'CopierDaemon', 'invalid state type!'

    torrents = {}
    resources = {}

    for torrent in state['torrents']:
        if torrent['type'] == 'MetaTorrent':
            resources[torrent['infoHash']] = torrent
            torrent.pop('infoHash')
        elif torrent['type'] == 'Torrent':
            torrents[torrent['infoHash']] = torrent
            torrent.pop('infoHash')

    added_files = set([r[0] for r in db.query('SELECT path FROM file')])
    added_data = set([r[0] for r in db.query('SELECT id FROM data')])
    added_resources = set([r[0] for r in db.query('SELECT id FROM resource')])

    with db(debug_sql=False, debug_transactions=True, transaction=False):
        try:
            db.begin()
            total = len(resources)

            stats = OrderedDict((
                ('skipped', OrderedDict((
                    ('resources', 0),
                    ('no torrent info', 0),
                    ('no md5 for data', 0),
                    ('no watcher for touch file', 0),
                    ('explicit directory', 0),
                ))),
                ('resources', 0),
                ('data', 0),
                ('files', 0),
                ('dirs', 0),
                ('symlinks', 0)
            ))

            for idx, (resource_id, resource) in enumerate(resources.iteritems()):
                gevent.sleep()

                if idx % 1000 == 0:
                    log.info('  Done %d/%d', idx, total)

                if resource_id in added_resources:
                    continue

                sha1_blocks = resource['metadata']['info']['pieces']
                metadata = {
                    'structure': resource['structure'],
                    'torrents': resource['torrents']
                }
                metadata_bin = bencode(metadata)

                assert len(metadata_bin) == resource['metadata']['info']['length']

                missing_files = False

                explicit_dirs = set()

                for fn, props in resource['structure'].iteritems():
                    if '/' not in fn:
                        continue

                    if props['resource']['type'] in ('torrent', 'touch', 'symlink'):
                        explicit_dirs.add(fn.rsplit('/', 1)[0])

                resource_datas = []

                for fn, props in resource['structure'].iteritems():
                    if props['resource']['type'] == 'torrent':
                        torrent_id = props['resource']['id']
                        torrent_info = torrents.get(torrent_id, None)
                        if not torrent_info:
                            stats['skipped']['no torrent info'] += 1
                            log.debug(
                                '  No torrent info found for file %s (torrent_id:%s), skipping',
                                fn, torrent_id
                            )
                            missing_files = True
                            continue

                        fn_md5_raw = torrent_info['md5Hash']
                        fn_path = Path(torrent_info['dataPath'])

                        if not fn_md5_raw:
                            stats['skipped']['no md5 for data'] += 1

                            log.debug(
                                '  Resource: %s, file %s => %s: no md5 hash, skipping',
                                resource_id, torrent_id, fn_path
                            )
                            missing_files = True
                            continue

                        fn_md5 = fn_md5_raw.encode('hex')

                        sha1_blocks = msgpack.Packer().pack(
                            Hasher.convert_sha1_blocks_from_lt_form(
                                torrent_info['metadata']['info']['pieces'],
                                torrent_info['metadata']['info']['piece length']
                            )
                        )

                        if fn_md5 not in added_data:
                            data_id = _store_data(db, torrent_info['metadata']['info']['length'], fn_md5, sha1_blocks)
                            added_data.add(fn_md5)
                            stats['data'] += 1
                        else:
                            data_id = fn_md5

                        if fn_path.strpath not in added_files:
                            _store_file(
                                db, fn_path.strpath, data_id,
                                torrent_info['tsFinished']
                            )
                            stats['files'] += 1
                            added_files.add(fn_path.strpath)

                        perms = 0755 if props['executable'] else 0644
                        resource_datas.append((fn, data_id, None, False, perms))

                    elif props['resource']['type'] == 'symlink':
                        symlink_target = props['resource']['data'].split(':', 1)[1]
                        stats['symlinks'] += 1
                        resource_datas.append((fn, None, symlink_target, False, -1))

                    elif props['resource']['type'] == 'mkdir':
                        if fn in explicit_dirs:
                            stats['skipped']['explicit directory'] += 1
                        else:
                            for dr in explicit_dirs:
                                if dr.startswith(fn.rstrip('/') + '/'):
                                    stats['skipped']['explicit directory'] += 1
                                    break
                            else:
                                # not found
                                stats['dirs'] += 1
                                resource_datas.append((fn, None, None, True, -1))

                    elif props['resource']['type'] == 'touch':
                        watcher = resource['watchers'].get(fn, None)
                        if not watcher:
                            log.debug(
                                '  Resource: %s, file %s (0 bytes): watcher not found',
                                resource_id, fn
                            )
                            stats['skipped']['no watcher for touch file'] += 1
                            missing_files = True
                            continue

                        if ZERO_MD5 not in added_data:
                            data_id = _store_data(db, 0, ZERO_MD5, ZERO_SHA1_PIECES)
                            stats['data'] += 1
                            added_data.add(ZERO_MD5)
                        else:
                            data_id = ZERO_MD5

                        fn_path = Path(watcher[0][0])
                        if fn_path.strpath not in added_files:
                            _store_file(
                                db, fn_path.strpath, data_id,
                                resource['watchers'][fn][0][4]
                            )
                            stats['files'] += 1
                            added_files.add(fn_path.strpath)

                        resource_datas.append((fn, data_id, None, False, -1))
                    else:
                        pp(props['resource'])
                        log.warning('  Unknown resource type: %r', props['resource'])
                        os._exit(1)
                        missing_files = True
                        continue

                if missing_files:
                    stats['skipped']['resources'] += 1
                    log.debug(
                        '  Where was some missing items in resource %s, skipping', resource_id
                    )
                    continue
                else:
                    if not any(r[1] for r in resource_datas):
                        log.warning('  No data found in resource %s, skipping', resource_id)
                        for name, _, symlink, directory in resource_datas:
                            log.debug('    path: %s, symlink %s, directory %r)', name, symlink, directory)
                        continue

                    resource_data = msgpack.dumps({
                        'structure': resource['structure'],
                        'torrents': resource['torrents'],
                        'torrent_info': resource['metadata']['info'],
                    })

                    db.query(
                        'INSERT INTO resource (id, type, data, torrents_count) '
                        'VALUES (?, ?, ?, ?)',
                        [
                            resource_id, 'rbtorrent1',
                            buffer(resource_data),
                            len(resource['torrents']) + 1,
                        ]
                    )

                    added_resources.add(resource_id)
                    stats['resources'] += 1

                    db.execute_many(
                        'INSERT INTO resource_data (resource, name, data, symlink, directory, perms) '
                        'VALUES (?, ?, ?, ?, ?, ?)',
                        [
                            (
                                resource_id,
                                name,
                                data,
                                symlink,
                                directory,
                                xperms
                            )
                            for (name, data, symlink, directory, xperms) in resource_datas
                        ],
                    )

        except Exception as ex:
            ei = sys.exc_info()
            log.info('Got unhandled exception: %s, rollbacking transaction...', str(ex))
            db.rollback()
            raise ei[0], ei[1], ei[2]

        else:
            log.info('Migration finished. Stats:')

            for key, value in stats.iteritems():
                if isinstance(value, dict):
                    log.info('  %s:', key)
                    for key2, value2 in value.iteritems():
                        log.info('    %s: %s', key2, value2)
                else:
                    log.info('  %s: %s', key, value)

            log.info('Migrate OK, commiting transaction')
            db.commit()
            db.analyze()
