import sys
import os
import gevent

from api.logger import SkynetLoggingHandler

from ..kernel_util import logging
from ..kernel_util.functional import singleton


handler = SkynetLoggingHandler(app='copier', filename='copier.log')
handlererr = SkynetLoggingHandler(app='copier', filename='copier-errors.log')
handlercli = SkynetLoggingHandler(app='copier', filename='copier-client.log')

root_logger = logging.getLogger()

log = root_logger.getChild('copier')
log.normal = log.info


class SmartLoggerAdapter(logging.LoggerAdapter):
    def __init__(self, logger, extra):
        super(SmartLoggerAdapter, self).__init__(logger, extra)
        self.name = logger.name

    def get_child(self, *args, **kwargs):
        return type(self)(self.logger.getChild(*args, **kwargs), self.extra)

    def get_parent(self):
        get_parent = getattr(self.logger, 'get_parent', None)
        parent = get_parent() if get_parent else self.logger.parent
        if parent is None:
            return None
        return type(self)(parent, self.extra)

    def get_local(self, key):
        grn = gevent.getcurrent()
        local = getattr(grn, 'slocal', getattr(grn, 'tlocal', {}))
        return local.get(key)

    getChild = get_child


class JobUidAdapter(SmartLoggerAdapter):
    def process(self, msg, kwargs):
        resource_uid = self.get_local('resource_uid')
        if not resource_uid:
            resource = self.get_local('resource')
            if resource:
                resource_uid = resource.uid

        if resource_uid and resource_uid not in msg:
            msg = '[resid:%s] %s' % (resource_uid, msg)

        job_uid = self.get_local('job_uid')
        job_name = self.get_local('job_name')

        if job_name == 'resource_info':
            job_name = 'info'
        elif job_name == 'resource_download':
            job_name = 'dl'
        elif job_name == 'resource_create':
            job_name = 'create'

        if job_uid:
            msg = '[jb]{%s:%s}  %s' % (str(job_uid).split('-', 1)[0], job_name, msg)

        return msg, kwargs


log = JobUidAdapter(log, {})
log.normal = log.info


def fix_logger_name(name):
    deprecated_name = 'ya.skynet.services.copier.rbtorrent'
    new_name = 'skbn'

    if name.startswith(deprecated_name):
        name = new_name + name[len(deprecated_name):]

    return name


def adapt_logger(logger):
    return JobUidAdapter(logger, {})


@singleton
def initialize(hdl=None):
    logging.renameLevels(1)

    root_logger.setLevel(logging.DEBUG)
    root_logger.normal = root_logger.info

    if hdl is None:
        root_logger.addHandler(handler)
        handlererr.setLevel(logging.WARN)
        root_logger.addHandler(handlererr)
    else:
        root_logger.addHandler(hdl)

    if os.getenv('SKYNET_LOG_STDOUT', False) == '1':
        os.environ.pop('SKYNET_LOG_STDOUT', None)

        console_handler = logging.StreamHandler(sys.stderr)
        console_handler.setLevel(logging.DEBUG)
        formatter = logging.ColoredFormatter(
            fmt=' %(levelname)s %(asctime)s %(name)s  %(message)s',
            fmtDatetime='%H:%M:%S',
            fmtName='[%(name)-30s]',
            fmtLevelname='[%(levelname)-1s]',
            fmtAsctime='%(datetime)s.%(msecs)003d',
            hungryLevels={
                'info': ['message'],
                'debug': ['message'],
                'warning': ['message', 'asctime', 'name'],
                'error': ['message', 'name']
            }
        )
        console_handler.setFormatter(formatter)
        root_logger.addHandler(console_handler)


@singleton
def initialize_client():
    logging.renameLevels(1)

    clilog = root_logger.getChild('skybone.client')
    clilog.setLevel(logging.DEBUG)
    clilog.normal = clilog.info
    clilog.addHandler(handlercli)
    clilog.propagate = False

    return clilog
