from __future__ import absolute_import

import bisect
import copy
import itertools
import json
import operator
import socket
import time
import urllib2

from abc import ABCMeta, abstractmethod

from .component import Component
from .utils import gevent_urlopen


class IMetric(object):
    __metaclass__ = ABCMeta

    def __init__(self, suffix):
        self.suffix = suffix

    @abstractmethod
    def __nonzero__(self):
        pass

    @abstractmethod
    def get_value(self):
        pass

    @abstractmethod
    def update(self, new_value):
        pass

    @abstractmethod
    def reset(self):
        pass


class Counter(IMetric):
    def __init__(self, suffix, update_func):
        super(Counter, self).__init__(suffix)
        self._update_func = update_func
        self.reset()

    def __nonzero__(self):
        return self.value != 0

    def get_value(self):
        return self.value

    def update(self, new_value):
        self.value = self._update_func(self.value, new_value)

    def reset(self):
        self.value = 0


class Histogram(IMetric):
    def __init__(self, intervals):
        super(Histogram, self).__init__('hgram')
        assert len(intervals) <= 50, 'Yasm allows up to 50 custom buckets in hgrams'
        self._buckets = [[x, 0] for x in intervals]
        assert self._buckets == sorted(self._buckets)

    def __nonzero__(self):
        return any(v for k, v in self._buckets)

    def get_value(self):
        return copy.deepcopy(self._buckets)

    # accepts either a number or a histogram
    def update(self, new_value):
        if isinstance(new_value, list):
            assert len(self._buckets) == len(new_value)
            for cur, new in zip(self._buckets, new_value):
                assert cur[0] == new[0], 'Mismatching histogram intervals'
                cur[1] += new[1]
        else:
            pos = bisect.bisect(self._buckets, [new_value, float('inf')])
            # buckets[pos - 1][0] <= [new_value, inf] < buckets[pos][0]
            if pos:
                pos -= 1
            self._buckets[pos][1] += 1

    def reset(self):
        for bucket in self._buckets:
            bucket[1] = 0


IO_HGRAM_INTERVALS = (
    0.0, 1.0, 2.0, 5.0, 10.0, 20.0, 30.0, 40.0, 50.0, 60.0,
    80.0,  100.0, 120.0, 140.0, 160.0, 180.0, 200.0, 225.0, 250.0, 275.0,
    300.0, 325.0, 350.0, 375.0, 400.0, 450.0, 500.0, 550.0, 600.0, 650.0,
    700.0, 800.0, 900.0, 1000.0, 1250.0, 1500.0, 1750.0, 2000.0, 2250.0, 2500.0,
    2750.0, 3000.0, 3500.0, 4000.0, 5000.0, 6000.0, 8000.0, 10000.0, 15000.0, 30000.0,
)

# 10 * (1.15^i) for i in range(-20, 30)
DL_HGRAM_INTERVALS = (
    0.61, 0.7, 0.81, 0.93, 1.1, 1.2, 1.4, 1.6, 1.9, 2.1,
    2.5, 2.8, 3.3, 3.8, 4.3, 5.0, 5.7, 6.6, 7.6, 8.7,
    10.0, 12.0, 13.0, 15.0, 17.0, 20.0, 23.0, 27.0, 31.0, 35.0,
    40.0, 47.0, 54.0, 62.0, 71.0, 81.0, 94.0, 110.0, 120.0, 140.0,
    160.0, 190.0, 220.0, 250.0, 290.0, 330.0, 380.0, 440.0, 500.0, 580.0
)


class MetricPusher(Component):
    PUSH_INTERVAL = 5
    PUSH_TIMEOUT = 2

    def __init__(self, yasmagent_url=None, parent=None):
        super(MetricPusher, self).__init__(logname='metrics', parent=parent)
        self.active = yasmagent_url is not None
        if not self.active:
            return

        self._counters = {
            'p2p_downloads_succeeded':   [Counter('summ', operator.add)],
            'p2p_downloads_failed':      [Counter('summ', operator.add)],
            'dfs_downloads_succeeded':   [Counter('summ', operator.add)],
            'dfs_downloads_failed':      [Counter('summ', operator.add)],
            'downloads_failed_no_peer':  [Counter('summ', operator.add)],
            'downloads_failed_fs_error': [Counter('summ', operator.add)],
            'quick_check_failed':        [Counter('summ', operator.add)],
            'convert_path_errors':       [Counter('summ', operator.add)],
            'local_bytes': [Counter('tmmm', operator.add)],
            'total_bytes': [Counter('tmmm', operator.add)],
            # seconds per piece read/write
            'skybit_io_read_ms': [Histogram(IO_HGRAM_INTERVALS)],  # skybone-skybit
            'dl_io_read_ms':     [Histogram(IO_HGRAM_INTERVALS)],  # skybone-dl
            'dl_io_write_ms':    [Histogram(IO_HGRAM_INTERVALS)],  # skybone-dl

            'dl_limited_crossdc_recv_speed_mb': [Histogram(DL_HGRAM_INTERVALS)],
            'dl_nolimit_crossdc_recv_speed_mb': [Histogram(DL_HGRAM_INTERVALS)],
        }

        self._yasmagent_url = yasmagent_url
        self.add_loop(self._push_counters, logname='push')

    def update_counter(self, key, value):
        if not self.active:
            return
        for counter in self._counters[key]:
            counter.update(value)

    def _push_counters(self, log):
        next_push_time = time.time() + self.PUSH_INTERVAL

        values = [
            {
                'name': name + '_' + counter.suffix,
                'val': counter.get_value()
            }
            for name, counters in self._counters.iteritems()
            for counter in counters if counter
        ]
        for counter in itertools.chain(*self._counters.values()):
            counter.reset()

        if not values:
            return max(0, next_push_time - time.time())

        data = [{
            'tags': {'itype': 'rtcsys', 'prj': 'skybone'},
            'values': values
        }]
        try:
            response = gevent_urlopen(
                self._yasmagent_url,
                data=json.dumps(data),
                headers={'Content-Type': 'application/json'},
                timeout=self.PUSH_TIMEOUT
            )
            json_response = json.load(response)
            if json_response['status'] == 'ok':
                log.debug('Pushed metrics to yasmagent')
            else:
                log.warning(
                    'Push to yasmagent failed: error %s (%s)',
                    json_response.get('error_code'),
                    json_response.get('error')
                )
        except (urllib2.URLError, socket.error, ValueError, KeyError) as ex:
            log.warning('Push to yasmagent failed: %s: %s', type(ex).__name__, ex)

        return max(0, next_push_time - time.time())
