from __future__ import print_function, division

import argparse
import errno
import faulthandler
import gevent
try:
    import gevent.coros as coros
except ImportError:
    import gevent.lock as coros
import gevent.event
import gevent.queue
import gevent.socket
import logging
import msgpack
import os
import socket
import struct
import sys
import threading
import time

try:
    from setproctitle import setproctitle
except ImportError:
    setproctitle = None  # noqa

from .component import Component
from .procrunner import ChildProc

if os.getuid() == 0:
    from ..kernel_util.sys.user import userPrivileges as user_privileges  # noqa
else:
    import contextlib

    @contextlib.contextmanager
    def user_privileges(user=None):
        yield

from ..rpc.server import RPC, Server as RPCServer


PROTO_BITTORRENT = 1
PROTO_SKYTORRENT = 2


class Bucket(Component):  # {{{
    """
    Leaking bucket with specified rate in B/s and max allowed burst.

    Burst is the minimal refill speed.

    In 8192 bursts:

        - leak 100 bytes (left 100)
        - leak 100 bytes (left 0)
        - wait burst / rate
        - refill 8192
        - leak 100 bytes (left 8092)
        - etc.

    So, it will look like we send 8192 bytes, then sleep 8192/rate seconds, then
    again send 8192 bytes, etc.

    This is done for 2 reasons:
        - much less function calls in python code (25-30% less cpu usage)
        - we will never send too small packets over network ;)
    """

    def __init__(self, rate, burst=8192, parent=None):
        self.rate = rate
        self.burst = burst
        self.capacity = rate
        self.last_leak = 0
        self._lock = coros.Semaphore(1)

        super(Bucket, self).__init__(parent=parent)

    def _leak(self, amount):
        now = time.time()
        elapsed = now - self.last_leak

        self.capacity = min(self.rate, self.capacity + int(self.rate * elapsed))
        self.last_leak = now

        allowed = min(amount, self.capacity)

        self.capacity -= allowed

        return allowed

    def leak(self, amount):
        with self._lock:
            allowed = 0
            need_more = amount

            allowed += self._leak(need_more)
            need_more = amount - allowed

            if need_more > 0:
                need_wait = max(self.burst, need_more) / self.rate
                gevent.sleep(need_wait)

                allowed += self._leak(need_more)

            return allowed
# }}}


class InfohashLookupCache(Component):  # {{{
    def __init__(self, skbt_sock, parent=None):
        self.skbt_sock = skbt_sock
        self.active = {}

        super(InfohashLookupCache, self).__init__(logname='hashlook', parent=parent)

    def activate(self, infohash, pid, endpoint, net_priority):
        self.log.info('[resid:%s] CONNECT to %r (pid: %r)', infohash[:8], endpoint, pid)
        self.active[infohash] = (endpoint, {'net_priority': net_priority, 'pid': pid})

    def deactivate(self, infohash):
        self.log.info('[resid:%s] DISCONNECT', infohash[:8])
        self.active.pop(infohash, None)

    def lookup(self, infohash, skybit):
        if infohash in self.active:
            return self.active[infohash]

        # We use 0x20 (LOW) network priority by default here.
        # This has one caveat:
        # 1) start dl with High prio
        # 2) some incoming conn arrives
        # 3) we will set 0x20 (Low) on that incoming connection
        # 4) but all outgoing connections will have High prio
        return self.skbt_sock, {'net_priority': 0x20}
# }}}


class NetProxy(Component):
    # Init {{{
    def __init__(
        self,
        master_cli,
        uid, desc, port, rpc_uds, proxy_port, data_sock, skbt_sock,
        limits=(None, None), tcp_buffers=(0, 0), parent=None
    ):
        super(NetProxy, self).__init__(logname='netproxy', parent=parent)

        self.master_cli = master_cli

        self.uid = uid
        self.desc = desc
        self.ready = gevent.event.Event()

        self.rpc = None
        self.rpc_obj = None
        self.rpc_uds = rpc_uds
        self.rpc_server = None

        self.hashcache = InfohashLookupCache(skbt_sock, parent=self)

        # Main data socket
        self.sock = gevent.socket.socket(socket.AF_INET6, socket.SOCK_STREAM)
        self.sock.setsockopt(socket.IPPROTO_IPV6, socket.IPV6_V6ONLY, 0)
        self.sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)

        self.tcp_send_buffer, self.tcp_recv_buffer = tcp_buffers

        if self.tcp_send_buffer > 0:
            self.log.info('Setting tcp tcp send buffer to %d', self.tcp_send_buffer)
            self.sock.setsockopt(socket.SOL_SOCKET, socket.SO_SNDBUF, self.tcp_send_buffer)

        if self.tcp_recv_buffer > 0:
            self.log.info('Setting tcp tcp recv buffer to %d', self.tcp_recv_buffer)
            self.sock.setsockopt(socket.SOL_SOCKET, socket.SO_RCVBUF, self.tcp_recv_buffer)

        self.sock.setsockopt(socket.IPPROTO_TCP, socket.TCP_NODELAY, 1)

        # Proxy socket
        self.sock2 = gevent.socket.socket(socket.AF_INET, socket.SOCK_STREAM)
        self.sock2.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        self.sock2.setsockopt(socket.IPPROTO_TCP, socket.TCP_NODELAY, 1)
        self.sock3 = gevent.socket.socket(socket.AF_INET6, socket.SOCK_STREAM)
        self.sock3.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        self.sock3.setsockopt(socket.IPPROTO_TCP, socket.TCP_NODELAY, 1)
        self.sock4 = gevent.socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
        self.sock4.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)

        self.port = port
        self.proxy_port = proxy_port
        self.data_sock = data_sock

        self.bucket_dl = self.bucket_ul = None

        if limits[0]:
            self.bucket_dl = Bucket(limits[0], parent=self)

        if limits[1]:
            self.bucket_ul = Bucket(limits[1], parent=self)

        if limits[2]:
            self.max_connections = limits[2]

        self.noslot_workers = coros.Semaphore(500)

        self._incoming_connections = 0

        self._stats = {
            'infohash_lookup': {
                'cache_size': 0,      # update on demand
                'cache_hits': 0,
                'cache_miss': 0,
                'found': 0,           # hash founds
                'missing': 0,         # hash misses
                'spent': 0,
            },
            'incoming': {
                'connects': 0,
                'connect_fails': 0,
                'noslot_count': 0,
                'no_infohash': 0,
                'torrent': {
                    'connects': 0,
                    'connect_fails': 0,
                    'no_infohash': 0,
                },
                'skybit': {
                    'connects': 0,
                    'connect_fails': 0,
                    'no_resource': 0,
                },
                'invalid_handshake': 0,  # unable to parse torrent handshake properly
                'local': 0,           # connections from local machine
            },
            'proxy': {
                'invalid_proto': 0,   # invalid socks5 proto
                'cmd_connect_cnt': 0,  # socks5 connect cmds
                'connect_fail': 0,    # unable to connect requested outside address
            },
            'uds': {
                'connects': 0,
                'connect_fails': 0,
            },

            'bytes_in': 0,            # bytes from outside to us
            'bytes_out': 0,           # bytes from us to outside
            'bytes_in_torrent': 0,
            'bytes_out_torrent': 0,
            'bytes_in_skybit': 0,
            'bytes_out_skybit': 0,
        }
    # }}}

    def start(self, *args, **kwargs):  # {{{
        self.rpc_obj = NetproxyRPC(self)
        self.rpc = RPC(self.log)
        self.rpc_server = RPCServer(self.log, backlog=gevent.socket.SOMAXCONN, max_conns=1000, unix=self.rpc_uds)
        self.rpc_server.register_connection_handler(self.rpc.get_connection_handler())

        self.rpc.mount(self.rpc_obj.ping)
        self.rpc.mount(self.rpc_obj.stats)
        self.rpc.mount(self.rpc_obj.connect, typ='full')

        self.rpc_server.start()

        self.sock.bind(('', self.port))
        self.sock.listen(socket.SOMAXCONN)

        self.log.info('Listening (main) on :%d', self.port)

        self.sock2.bind(('127.0.0.1', self.proxy_port))
        self.sock2.listen(socket.SOMAXCONN)

        self.sock3.bind(('::1', self.proxy_port))
        self.sock3.listen(socket.SOMAXCONN)

        self.log.info('Listening (proxy) on %s:%d', '::1', self.proxy_port)

        if not self.data_sock.startswith('\0'):
            try:
                os.unlink(self.data_sock)
            except:
                pass

        self.sock4.bind(self.data_sock)
        self.sock4.listen(socket.SOMAXCONN)

        if not self.data_sock.startswith('\0'):
            os.chmod(self.data_sock, 0o666)

        self.log.info('Listening (data) on %s', self.data_sock)

        ret = super(NetProxy, self).start(*args, **kwargs)

        self.ready.set()

        return ret
    # }}}

    # Utils {{{
    def _read(self, sock, nbytes):
        data = sock.recv(nbytes)
        if len(data) != nbytes:
            return ''
        return data

    def _sock_proxy(self, s1, s2, out, name=None, log_prefix=''):
        while True:
            try:
                bucket = self.bucket_ul if out else self.bucket_dl
                data = s1.recv(8192)
                datalen = len(data)

                if bucket is not None:
                    bucket.leak(datalen)

            except socket.error as ex:
                if ex.errno not in (
                    errno.ECONNRESET,
                ):
                    self.log.debug('%sR | %s', log_prefix, str(ex))
                return

            if not data:
                return

            if not out:
                self._stats['bytes_in'] += datalen
                if name:
                    self._stats['bytes_in_' + name] += datalen

            try:
                s2.sendall(data)
            except socket.error as ex:
                if ex.errno not in (
                    errno.ECONNRESET,
                ):
                    self.log.debug('%sW | %s', log_prefix, str(ex))
                return

            if out:
                self._stats['bytes_out'] += len(data)
                if name:
                    self._stats['bytes_out_' + name] += len(data)

    def _fmt_peer_name(self, sock):
        pname = sock.getpeername()
        if isinstance(pname, basestring):
            return 'UDS'

        ip, port = pname[:2]
        if ip.startswith('::ffff:'):
            ip = ip[7:]
        if ':' in ip:
            ip = '[' + ip + ']'
        return '%s:%d' % (ip, port)

    def _fw_sock(self, sock1, sock2, infohash, sock_timeout=5, name=None):
        # sock1 is always we
        # sock2 is always remote

        sock1.settimeout(sock_timeout)
        sock2.settimeout(sock_timeout)

        try:
            log_prefix1 = '[trnt:%s]  %s => %s | ' % (
                infohash[:8], self._fmt_peer_name(sock1), self._fmt_peer_name(sock2)
            )
            log_prefix2 = '[trnt:%s]  %s <= %s | ' % (
                infohash[:8], self._fmt_peer_name(sock2), self._fmt_peer_name(sock1)
            )
        except socket.error as ex:
            if ex.errno == errno.ENOTCONN:
                pass
            else:
                raise
        else:
            grn1 = gevent.spawn(self._sock_proxy, sock1, sock2, out=True, name=name, log_prefix=log_prefix1)
            grn2 = gevent.spawn(self._sock_proxy, sock2, sock1, out=False, name=name, log_prefix=log_prefix2)

            ev = gevent.event.Event()

            grn1.rawlink(lambda _: ev.set())
            grn2.rawlink(lambda _: ev.set())

            ev.wait()

            grn1.kill()
            grn2.kill()

        sock1.close()
        sock2.close()

    def _socks5_pack_addr(self, ip, port):
        if ip.startswith('::ffff:'):
            ip = ip[7:]

        if ':' in ip:
            family = socket.AF_INET6
        else:
            family = socket.AF_INET

        ip_packed = socket.inet_pton(family, ip)

        return ip_packed + struct.pack('!H', port)
    # }}}

    # Proxy worker {{{
    def proxy_conn_worker(self, sock, peer):  # {{{
        sock.settimeout(1)

        try:
            data = self._read(sock, 3)
        except socket.error as ex:
            self.log.warning('Got error while reading proto version: %s', str(ex))
            sock.close()
            return

        if not data:
            self.log.warning('Got EOF while reading proto version')
            sock.close()
            return

        if data[0] != '\x05':
            self.log.warning('Invalid proto version: %r', data)
            self._stats['proxy']['invalid_proto'] += 1
            sock.close()
            return

        if data[1] != '\x01' or data[2] != '\x00':
            self.log.warning('Unusual 2-3 bytes: %r', data[1:2])
            self._stats['proxy']['invalid_proto'] += 1
            sock.close()
            return

        sock.sendall('\x05\x00')

        try:
            data = self._read(sock, 4)
        except socket.error as ex:
            self.log.warning('Got error while reading proto command: %s', str(ex))
            sock.close()
            return

        if not data:
            self.log.warning('Got EOF while reading proto command')
            sock.close()
            return

        if data[0] != '\x05':
            self.log.warning('Unusual 4th byte: %r', data[0])
            self._stats['proxy']['invalid_proto'] += 1
            sock.close()
            return

        cmd = data[1]
        addr_type = data[3]

        if addr_type == '\x01':  # ipv4
            v6, read_family, read_bytes = False, socket.AF_INET, 4
        elif addr_type == '\x04':
            v6, read_family, read_bytes = True, socket.AF_INET6, 16
        else:
            self.log.warning('Unusual addr_type %r', addr_type)
            self._stats['proxy']['invalid_proto'] += 1
            sock.close()
            return

        try:
            dst_addr = socket.inet_ntop(read_family, self._read(sock, read_bytes))
            dst_port = struct.unpack(
                '!H', self._read(sock, 2)
            )[0]
        except socket.error as ex:
            self.log.warning('Got error while reading host/port: %s', str(ex))
            sock.close()
            return
        except struct.error:
            self.log.warning('Got EOF while reading host/port')
            sock.close()
            return

        if cmd == '\x01':
            # child connect request
            self._proxy_cmd_connect(sock, dst_addr, dst_port, v6)
        else:
            assert 0, 'Unable to handle cmd -- unknown (%r)' % (cmd, )
    # }}}
    # }}}

    def _proxy_cmd_connect(self, sock, addr, port, v6):  # {{{
        self._stats['proxy']['cmd_connect_cnt'] += 1

        try:
            if v6:
                fsock = socket.socket(gevent.socket.AF_INET6)
            else:
                fsock = socket.socket()

            if self.tcp_send_buffer > 0:
                fsock.setsockopt(socket.SOL_SOCKET, socket.SO_SNDBUF, self.tcp_send_buffer)

            if self.tcp_recv_buffer > 0:
                fsock.setsockopt(socket.SOL_SOCKET, socket.SO_RCVBUF, self.tcp_recv_buffer)

            fsock.setsockopt(socket.IPPROTO_TCP, socket.TCP_NODELAY, 1)

            fsock.setblocking(0)
            end = time.time() + 5
            while True:
                err = fsock.getsockopt(socket.SOL_SOCKET, socket.SO_ERROR)
                if err:
                    raise socket.error(err, os.strerror(err))
                result = fsock.connect_ex((addr, port))
                if not result or result == errno.EISCONN:
                    break
                elif (result in (errno.EWOULDBLOCK, errno.EINPROGRESS, errno.EALREADY)):
                    timeleft = end - time.time()
                    if timeleft <= 0:
                        raise socket.error('timed out')
                    gevent.socket.wait_write(fsock.fileno(), timeout=timeleft)
                else:
                    raise socket.error(result, os.strerror(result))
            fsock = gevent.socket.socket(_sock=fsock)
        except socket.error as ex:
            self.log.warning('%s:%d | Unable to connect: %s', addr, port, str(ex))
            self._stats['proxy']['connect_fail'] += 1
            sock.close()
            return

        try:
            if v6:
                sock.sendall('\x05\x00\x00\x04' + self._socks5_pack_addr(*fsock.getsockname()[:2]))
            else:
                sock.sendall('\x05\x00\x00\x01' + self._socks5_pack_addr(*fsock.getsockname()[:2]))

            proto, data = self._parse_handshake(sock)
            if not proto:
                self._stats['incoming']['invalid_handshake'] += 1
                sock.close()
                return

            infohash, _, payload = data

            fsock.sendall(payload)
        except socket.error as ex:
            self.log.warning('%s', str(ex))
            sock.close()
            fsock.close()
            return

        self._fw_sock(sock, fsock, infohash, name='torrent')
    # }}}

    def skybit_proxy_conn_worker(self, sock, peer):  # {{{
        self._stats['uds']['connects'] += 1

        payload = sock.recv(40)
        if len(payload) < 40:
            sock.close()
            return

        infohash, af, ip_packed, port, net_priority = struct.unpack('!20sB16sHB', payload)

        if af == socket.AF_INET:
            ip_packed = ip_packed[:4]
        ip = socket.inet_ntop(af, ip_packed)
        infohash = infohash.encode('hex')

        v6 = af == socket.AF_INET6

        try:
            if v6:
                fsock = socket.socket(gevent.socket.AF_INET6)
            else:
                fsock = socket.socket()

            if net_priority:
                if os.uname()[0].lower() == 'linux':  # does not work on darwin
                    try:
                        fsock.setsockopt(socket.IPPROTO_IP, socket.IP_TOS, net_priority)
                    except socket.error as ex:
                        if 'Microsoft' in os.uname()[3] and ex.errno == errno.EINVAL:
                            # May fail on WSL
                            pass
                        else:
                            raise
                if v6:
                    try:
                        fsock.setsockopt(socket.IPPROTO_IPV6, socket.IPV6_TCLASS, net_priority)
                    except socket.error as ex:
                        if ex.errno in (errno.EPERM, ):
                            # This could fail if we not allowed to set TCLASS, e.g. in WSL
                            pass
                        else:
                            raise

            if self.tcp_send_buffer > 0:
                fsock.setsockopt(socket.SOL_SOCKET, socket.SO_SNDBUF, self.tcp_send_buffer)

            if self.tcp_recv_buffer > 0:
                fsock.setsockopt(socket.SOL_SOCKET, socket.SO_RCVBUF, self.tcp_recv_buffer)

            fsock.setsockopt(socket.IPPROTO_TCP, socket.TCP_NODELAY, 1)

            fsock.setblocking(0)
            end = time.time() + 5
            while True:
                err = fsock.getsockopt(socket.SOL_SOCKET, socket.SO_ERROR)
                if err:
                    raise socket.error(err, os.strerror(err))
                result = fsock.connect_ex((ip, port))
                if not result or result == errno.EISCONN:
                    break
                elif (result in (errno.EWOULDBLOCK, errno.EINPROGRESS, errno.EALREADY)):
                    timeleft = end - time.time()
                    if timeleft <= 0:
                        raise socket.error('timed out')
                    gevent.socket.wait_write(fsock.fileno(), timeout=timeleft)
                else:
                    raise socket.error(result, os.strerror(result))
            fsock = gevent.socket.socket(_sock=fsock)
        except socket.error as ex:
            self.log.warning('%s:%d | Unable to connect: %s', ip, port, str(ex))
            self._stats['uds']['connect_fails'] += 1

            strerror = ex.strerror
            if not strerror:
                strerror = str(ex)

            this_errno = ex.errno
            if not isinstance(this_errno, int):
                this_errno = 0

            try:
                sock.sendall(struct.pack('!BHH', 0, this_errno, len(strerror)))
                sock.sendall(strerror)
            except socket.error as ex:
                pass

            sock.close()
            return
        else:
            try:
                sock.sendall(struct.pack('!BHH', 1, 0, 0))
            except socket.error:
                sock.close()
                return

        self._fw_sock(sock, fsock, infohash, sock_timeout=60, name='skybit')
    # }}}

    # Main worker {{{
    def _parse_handshake_skybit(self, payload, sock):
        try:
            llen = sock.recv(4)
        except socket.error as ex:
            self.log.warning('Unable to receive handshake: %s', str(ex))
            return None, (None, '')

        if len(llen) < 4:
            self.log.warning('Unable to receive handshake (wrong llen)')
            return None, (None, '')

        payload += llen
        llen = struct.unpack('!I', llen)[0]

        if llen > 4096:
            self.log.warning('Unable to receive handshake (too much llen: %d)', llen)
            return None, (None, '')

        unpacker = msgpack.Unpacker()
        while llen:
            data = sock.recv(min(8192, llen))
            if not data:
                self.log.warning('Unable to receive handshake (connection closed)')
                return None, (None, '')

            payload += data
            llen -= len(data)
            unpacker.feed(data)

        try:
            msgtype, message = unpacker.next()
        except StopIteration:
            return None, (None, '')

        if msgtype != 'SKYBIT_HANDSHAKE_1':
            self.log.warning('Unable to receive handshake: invalid message %r', msgtype)
            return None, (None, '')

        if message.get('version', None) != 1:
            self.log.warning('Unable to receive handshake: invalid version %r', message.get('version', None))
            return None, (None, '')

        return PROTO_SKYTORRENT, (message, payload)

    def _parse_handshake(self, sock):
        try:
            payload = sock.recv(6)
        except socket.error as ex:
            self.log.warning('Unable to receive handshake: %s', str(ex))
            return None, (None, '')

        if len(payload) < 6:
            return None, (None, payload)

        if payload == 'SKYBIT':
            return self._parse_handshake_skybit(payload, sock)

        try:
            payload = payload + sock.recv(62)
        except socket.error as ex:
            self.log.warning('Unable to receive handshake: %s', str(ex))
            return None, (None, '')

        if len(payload) < 68:
            return None, (None, payload)

        pstrlen, pstr, reserved, infohash, peer_id = struct.unpack('!B19s8s20s20s', payload)

        if pstrlen == 97 and pstr == 'aaaaaaaaaaaaaaaaaaa':
            # pings from ctl.py
            return None, (None, payload)

        if pstrlen == 19:
            proto = None
            if pstr == 'BitTorrent protocol':
                proto = PROTO_BITTORRENT

            if proto:
                return proto, (infohash.encode('hex'), (reserved, peer_id), payload)

        self.log.warning('Invalid handshake: pstrlen=%r pstr=%r', pstrlen, pstr)
        return None, (None, payload)

    def noslot_worker(self, sock, peer):
        self._stats['incoming']['noslot_count'] += 1

        sock.settimeout(1)
        proto, data = self._parse_handshake(sock)
        if not proto:
            self._stats['incoming']['invalid_handshake'] += 1
            sock.close()
            return

        if proto == PROTO_SKYTORRENT:
            self._stats['incoming']['skybit']['connects'] += 1

            message, payload = data

            with gevent.Timeout(1) as tout:
                infohash_bytes = message['uid']
                response_msg = (
                    'SKYBIT_HANDSHAKE_1',
                    {
                        'version': 1,
                        'uid': infohash_bytes,
                        'world_uid_hex': self.uid.decode('hex'),
                        'world_uid_str': None,
                        'world_desc': self.desc,
                        'capabilities': ['skybit'],
                    }
                )
                response_msg = msgpack.dumps(response_msg)
                response = struct.pack('!6sI', 'SKYBIT', len(response_msg))
                sock.sendall(response)
                sock.sendall(response_msg)

                response_msg = msgpack.dumps(('NO_SLOT', 1))
                response = struct.pack('!I', len(response_msg))
                sock.sendall(response)
                sock.sendall(response_msg)

                # Allow peer to send us ACTIVATE message and close conn by itself
                try:
                    sock.recv(8192)
                except gevent.Timeout as ex:
                    if ex != tout:
                        raise
                finally:
                    sock.close()

            return

        else:
            self._stats['incoming']['torrent']['connects'] += 1
            # If libtorrent comes to us -- just close connection without
            # any details
            sock.close()
            return

    def main_conn_worker(self, sock, peer):
        sock.settimeout(1)

        self._stats['incoming']['connects'] += 1

        if peer[0].startswith('::ffff:'):
            peer = (peer[0][7:], peer[1])
            v6 = False
        else:
            v6 = ':' in peer[0]
        proto, data = self._parse_handshake(sock)
        if not proto:
            self._stats['incoming']['invalid_handshake'] += 1
            sock.close()
            return

        if proto == PROTO_SKYTORRENT:
            self._stats['incoming']['skybit']['connects'] += 1
            message, payload = data
            infohash = message['uid'].encode('hex')
        else:
            infohash, payload_parsed, payload = data
            self._stats['incoming']['torrent']['connects'] += 1

        local = sock.getpeername()[0] == sock.getsockname()[0]

        if local:
            self._stats['incoming']['local'] += 1

        try:
            endpoint, opts = self.hashcache.lookup(infohash, proto == PROTO_SKYTORRENT)
        except gevent.Timeout:
            sock.close()
            return

        if 'net_priority' in opts and opts['net_priority'] is not None:
            if os.uname()[0].lower() == 'linux':  # does not work on darwin
                try:
                    sock.setsockopt(socket.IPPROTO_IP, socket.IP_TOS, opts['net_priority'])
                except socket.error as ex:
                    if 'Microsoft' in os.uname()[3] and ex.errno == errno.EINVAL:
                        # May fail on WSL
                        pass
                    else:
                        raise
            if v6:
                try:
                    sock.setsockopt(socket.IPPROTO_IPV6, socket.IPV6_TCLASS, opts['net_priority'])
                except socket.error as ex:
                    if ex.errno in (errno.EPERM, ):
                        # This could fail if we not allowed to set TCLASS, e.g. in WSL
                        pass
                    else:
                        raise

        if not endpoint:
            self._stats['incoming']['no_infohash'] += 1
            if proto == PROTO_SKYTORRENT:
                # We have no head for this -- report to peer and disconnect
                self._stats['incoming']['skybit']['no_resource'] += 1
                sock.close()
                return
            else:
                self._stats['incoming']['torrent']['no_infohash'] += 1
                sock.close()
                return

        if isinstance(endpoint, (list, tuple)):
            endpoint = tuple(endpoint)
            if v6:
                af = gevent.socket.AF_INET6
            else:
                af = gevent.socket.AF_INET
            set_nodelay = True
        else:
            af = gevent.socket.AF_UNIX
            set_nodelay = False

        if proto == PROTO_SKYTORRENT:
            # Prefix payload with peer info
            if v6:
                peer_af = gevent.socket.AF_INET6
            else:
                peer_af = gevent.socket.AF_INET

            payload = struct.pack(
                '!20sB16sH',
                infohash.decode('hex'),
                peer_af,
                gevent.socket.inet_pton(peer_af, peer[0]),
                peer[1]
            ) + payload

        try:
            if 'pid' in opts and opts['pid'] is not None:
                # If we have separate pid -- detect if we are in different net namespace
                # and move into pid's if differs to connect incoming connection into skybone-dl
                # inside isolated environment
                from .setns import setns, CLONE_NEWNET
                pid = opts['pid']
                nsfn = '/proc/%d/ns/net' % (pid, )
                onsfn = '/proc/%d/ns/net' % (os.getpid(), )

                has_ns = False

                with user_privileges():
                    has_ns = os.path.exists(onsfn) and os.path.exists(nsfn)

                    if has_ns:
                        with open(onsfn) as onsfd:
                            with open(nsfn) as nsfd:
                                # Switch namespace
                                setns(nsfd.fileno(), CLONE_NEWNET)

                                # Create socket in new namespace
                                fsock = socket.socket(af)

                            # Switch namespace back
                            setns(onsfd.fileno(), CLONE_NEWNET)

                if not has_ns:
                    # If we had no ns, create regular socket with current user (not root!)
                    fsock = socket.socket(af)
            else:
                # No separate pid -- create regular user socket
                fsock = socket.socket(af)

            if set_nodelay:
                fsock.setsockopt(socket.IPPROTO_TCP, socket.TCP_NODELAY, 1)

            fsock.setblocking(0)
            end = time.time() + 5

            while True:
                err = fsock.getsockopt(socket.SOL_SOCKET, socket.SO_ERROR)
                if err:
                    raise socket.error(err, os.strerror(err))

                result = fsock.connect_ex(endpoint)

                if not result or result == errno.EISCONN:
                    break
                elif (result in (errno.EWOULDBLOCK, errno.EINPROGRESS, errno.EALREADY)):
                    timeleft = end - time.time()
                    if timeleft <= 0:
                        raise socket.error('timed out')
                    gevent.sleep(0.1)
                    gevent.socket.wait_write(fsock.fileno(), timeout=timeleft)
                else:
                    raise socket.error(result, os.strerror(result))
            fsock = gevent.socket.socket(_sock=fsock)
        except socket.error as ex:
            if isinstance(endpoint, (list, tuple)):
                self.log.warning(
                    '[trnt:%s]  %s:%d | Unable to connect: %s',
                    infohash[:8], endpoint[0], endpoint[1], str(ex)
                )
            else:
                self.log.warning(
                    '[resource:%s]  %s | Unable to connect: %s',
                    infohash[:8], endpoint, str(ex)
                )

            self._stats['incoming']['connect_fails'] += 1
            if proto == PROTO_SKYTORRENT:
                self._stats['incoming']['skybit']['connect_fails'] += 1
            else:
                self._stats['incoming']['torrent']['connect_fails'] += 1
            sock.close()
            return

        self._stats['bytes_in'] += len(payload)

        # Since we want to avoid situation of connecting to secure sockets (not skybone downbloaders)
        # we add extra prefix here. So the total payload length will be 38 + 18 bytes (56)
        # See SKYDEV-1806 for attack example without this addition
        payload = 'SKYBONE_NETPROXY_1' + payload

        try:
            fsock.settimeout(1)
            fsock.sendall(payload)
        except socket.error as ex:
            self.log.warning('%s', str(ex))
            sock.close()
            fsock.close()
            return

        if proto == PROTO_SKYTORRENT:
            sock_timeout = 60
        else:
            sock_timeout = 5

        self._fw_sock(
            fsock, sock, infohash,
            sock_timeout,
            name='skybit' if proto == PROTO_SKYTORRENT else 'torrent'
        )

    def _main_conn_worker_done(self, grn):
        self._incoming_connections -= 1
    # }}}

    # Accept loops {{{
    @Component.green_loop
    def main_accept(self, log):
        while True:
            sock, peer = self.sock.accept()
            if (
                self._incoming_connections < self.max_connections
            ):
                self._incoming_connections += 1
                grn = gevent.spawn(self.main_conn_worker, sock, peer)
                grn.rawlink(self._main_conn_worker_done)
            else:
                # If we have no more incoming connection slots -- block here until
                # noslot worker will be available.
                self.noslot_workers.acquire()
                grn = gevent.spawn(self.noslot_worker, sock, peer)
                grn.rawlink(lambda grn: self.noslot_workers.release())

            del sock, peer
        return 1

    @Component.green_loop
    def proxy_accept(self, log):
        while True:
            sock, peer = self.sock2.accept()
            gevent.spawn(self.proxy_conn_worker, sock, peer)
            del sock, peer
        return 1

    @Component.green_loop
    def proxy_accept_v6(self, log):
        while True:
            sock, peer = self.sock3.accept()
            gevent.spawn(self.proxy_conn_worker, sock, peer)
            del sock, peer
        return 1

    @Component.green_loop
    def skybit_proxy_accept(self, log):
        while True:
            sock, peer = self.sock4.accept()
            gevent.spawn(self.skybit_proxy_conn_worker, sock, peer)
            del sock, peer
        return 1

    @Component.green_loop
    def check(self):
        try:
            if self.ready.isSet():
                if not os.path.exists(self.data_sock):
                    self.log.critical('No data socket on disk -- exit immidiately')
                    os._exit(1)
        except:
            self.log.critical('Unable to check data socket on disk -- exit immidiately')
            os._exit(1)

        return 30
    # }}}


class NetproxyRPC(object):
    def __init__(self, netproxy):
        self.netproxy = netproxy

    def ping(self):
        return self.netproxy.ready.wait() and os.path.exists(self.netproxy.data_sock)

    def stats(self):
        stats = self.netproxy._stats
        return stats

    def connect(self, job, uid, sock, net_priority, pid):
        try:
            job.state(True)
            self.netproxy.hashcache.activate(uid, pid, sock, net_priority)
            assert job.feedback
        finally:
            self.netproxy.hashcache.deactivate(uid)


def _child_check_parent():
    while True:
        if os.getppid() == 1:
            os._exit(1)
        time.sleep(1)


def _main():  # {{{
    faulthandler.enable()

    if setproctitle is not None:
        setproctitle('skybone-netproxy')

    thr = threading.Thread(target=_child_check_parent)
    thr.daemon = True
    thr.start()

    handler = logging.StreamHandler(sys.stdout)
    handler.setLevel(logging.DEBUG)

    root_handler = logging.getLogger('')
    root_handler.setLevel(logging.DEBUG)
    root_handler.addHandler(handler)

    parser = argparse.ArgumentParser()
    parser.add_argument('--master-uds', required=True)
    parser.add_argument('--rpc-uds')
    parser.add_argument('--rpc-uds-abstract', action='store_true')
    parser.add_argument('--uid')
    parser.add_argument('--desc')
    parser.add_argument('--port', type=int)
    parser.add_argument('--port-proxy', type=int)
    parser.add_argument('--data-uds')
    parser.add_argument('--data-uds-abstract', action='store_true')
    parser.add_argument('--skbt-uds')
    parser.add_argument('--skbt-uds-abstract', action='store_true')
    parser.add_argument('--max-speed-dl', type=int)
    parser.add_argument('--max-speed-ul', type=int)
    parser.add_argument('--tcp-send-buffer', type=int)
    parser.add_argument('--tcp-recv-buffer', type=int)
    parser.add_argument('--max-connections', type=int)

    args = parser.parse_args()

    cproc = ChildProc(name='netproxy', master_uds=args.master_uds)
    cproc.start()

    if args.max_speed_dl == -1:
        args.max_speed_dl = None
    if args.max_speed_ul == -1:
        args.max_speed_ul = None

    nproxy = NetProxy(
        master_cli=cproc.master_cli,
        uid=args.uid,
        desc=args.desc,
        port=args.port,
        rpc_uds=('\0' if args.rpc_uds_abstract else '') + args.rpc_uds,
        proxy_port=args.port_proxy,
        data_sock=('\0' if args.data_uds_abstract else '') + args.data_uds,
        skbt_sock=('\0' if args.skbt_uds_abstract else '') + args.skbt_uds,
        limits=(args.max_speed_dl, args.max_speed_ul, args.max_connections),
        tcp_buffers=(args.tcp_send_buffer, args.tcp_recv_buffer)
    )

    if setproctitle is not None:
        setproctitle('skybone-netproxy %d:%d (limits: dl %s, ul %s)' % (
            args.port, args.port_proxy,
            args.max_speed_dl or 'none',
            args.max_speed_ul or 'none',
        ))

    try:
        nproxy.start()
        nproxy.join()
    except KeyboardInterrupt:
        sys.stderr.write('Interrupted\n')
        nproxy.stop()
        nproxy.join()
        cproc.stop()
        cproc.join()
        os._exit(1)


def main():
    with user_privileges('skynet'):
        return _main()
# }}}
