import errno
import gevent
try:
    import gevent.coros as coros
except ImportError:
    import gevent.lock as coros
import gevent.event
import logging
import os
import signal
import time

from . import subprocess_gevent as subproc, utils
from .component import Component

from ..rpc.client import RPCClientGevent
from ..kernel_util.sys.user import userPrivileges as user_privileges  # noqa


class Proc(Component):
    def __init__(self, procrunner, logname, args, rpc_uds=None, root=False, parent=None):
        self.args = args
        self.procrunner = procrunner
        self.proc = None
        self.rpc_uds = rpc_uds
        self.root = root
        self.cli = None
        self.ready = gevent.event.Event()

        self._fails_in_a_row = 0

        super(Proc, self).__init__(logname=logname, parent=parent)

    def _stream_reader(self, name, stream):
        buff = ''
        log = self.log.getChild(name)

        while True:
            gevent.socket.wait_read(stream.fileno())
            data = stream.read()

            if not data:
                break

            buff += data

            while '\n' in buff:
                part, buff = buff.split('\n', 1)
                if name == 'stdout':
                    log.debug(name + ': ' + part)
                else:
                    log.warning(name + ': ' + part)

    def _preexec(self):
        if self.root and utils.has_root():
            user_privileges(user='root', store=False).__enter__()
            return

        if os.uname()[0] == 'Darwin':
            uid = os.getuid()
            euid = os.geteuid()

            if not self.root and uid != euid and utils.has_root():
                user_privileges(user=euid, store=False).__enter__()
                return

    @Component.green_loop(logname='runner')
    def runner(self, log):
        log.debug('Runnning proc %s', ' '.join(self.args))
        self.proc = subproc.Popen(
            self.args, stdout=subproc.PIPE, stderr=subproc.PIPE,
            preexec_fn=self._preexec, close_fds=True
        )

        stdout_grn = gevent.spawn(self._stream_reader, 'stdout', self.proc.stdout)
        stderr_grn = gevent.spawn(self._stream_reader, 'stderr', self.proc.stderr)

        if self.rpc_uds:
            self.cli = RPCClientGevent(self.rpc_uds, port=None)
        else:
            self.ready.set()

        try:
            if self.cli:
                last_ping = time.time()

                while self.proc.poll() is None:
                    try:
                        self.cli.call('ping').wait()
                        last_ping = time.time()
                        self._fails_in_a_row = 0
                        self.ready.set()
                    except Exception as ex:
                        log.debug('ping fail: %s', ex)

                        if time.time() - last_ping >= 60:
                            self.proc.kill()

                        gevent.sleep(0.3)
                    else:
                        # Wait 600 seconds or until process terminates.
                        for i in range(600):
                            gevent.sleep(1)
                            if self.proc.poll() is not None:
                                break

                log.warning('process exited')
            else:
                ts = time.time()
                self.proc.wait(check_interval=1.0)
                if time.time() - ts > 30:
                    # If process was working for at least 30 seconds -- consider it was "okayish"
                    self._fails_in_a_row = 0
        finally:
            self._fails_in_a_row += 1

            self.kill_proc()
            self.ready.clear()

            if self.cli:
                self.cli.stop()
                self.cli = None

            stdout_grn.kill()
            stderr_grn.kill()

        if self._fails_in_a_row < 5:
            logger = log.debug
        elif self._fails_in_a_row < 15:
            logger = log.warning
        else:
            logger = log.error

        sleeptime = min(30, self._fails_in_a_row)
        logger(
            'process failed %d times in a row, sleepin %d secs before restart...',
            self._fails_in_a_row, sleeptime
        )
        return sleeptime

    def start(self):
        super(Proc, self).start()

    def _kill_proc(self, proc, sig):
        if utils.has_root() and self.root:
            pid = os.fork()
            if not pid:
                try:
                    if os.uname()[0].lower() == 'darwin' or os.uname()[0].lower().startswith('cygwin'):
                        os.setreuid(0, 0)
                    else:
                        os.setresuid(0, 0, 0)
                    os.setuid(0)
                    proc.send_signal(sig)
                finally:
                    os._exit(0)
            else:
                while True:
                    try:
                        ret = os.waitpid(pid, os.WNOHANG)
                        if ret[0] == 0:  # on bsd ret[1] may be -115, which is okay
                            gevent.sleep(0.1)
                        else:
                            break
                    except OSError as ex:
                        if ex.errno != errno.ECHILD:
                            raise
                        break
        else:
            proc.send_signal(sig)

    def kill_proc(self, block=True, sig=signal.SIGKILL):
        if self.proc:
            while self.proc.poll() is None:
                self._kill_proc(self.proc, sig)

                if not block:
                    return

                gevent.sleep(0.01)

            self.proc = None

    def stop(self):
        self.kill_proc(block=False, sig=signal.SIGINT)
        return super(Proc, self).stop()

    def join(self):
        if self.proc is not None:
            with gevent.Timeout(10) as tout:
                try:
                    self.proc.wait()
                except gevent.Timeout as ex:
                    if ex != tout:
                        raise

                    self.kill_proc(block=True)

        return super(Proc, self).join()


class ProcRunner(Component):
    def __init__(self, parent=None):
        super(ProcRunner, self).__init__(logname='procrun', parent=parent)

        self.procs = {}

    def add_proc(self, name, args, rpc_uds=None, root=False):
        proc = Proc(self, name, args, rpc_uds=rpc_uds, root=root, parent=self)
        return proc


class ChildLogHandler(logging.Handler):
    def __init__(self, job):
        self.job = job
        self.log = coros.Semaphore(1)
        super(ChildLogHandler, self).__init__()

    def emit(self, record):
        with self.lock:
            try:
                self.job.send((
                    record.name, record.levelno, record.msg, record.args
                ))
            except TypeError:
                self.job.send((
                    record.name, record.levelno, record.msg % record.args, ()
                ))


class ChildProc(Component):
    def __init__(self, name, master_uds, parent=None):
        self.name = name

        self.master_uds = master_uds
        self.master_cli = RPCClientGevent(self.master_uds, port=None)
        self.logger_grn = None

        self._orig_log_handlers = None
        self._redirect_logging_worker_grn = None

        super(ChildProc, self).__init__(logname=name + '.hlpr', parent=parent)

    def start(self):
        self.master_cli.connect()
        self.master_cli.name(self.name)

        self.redirect_logging()
        return super(ChildProc, self).start()

    def stop(self):
        if self._redirect_logging_worker_grn:
            if self.job:
                self.job.send(None)
                self.job = None

            self._redirect_logging_worker_grn.join(timeout=3)
            self._redirect_logging_worker_grn.kill()

        self.master_cli.stop()
        return super(ChildProc, self).stop()

    def redirect_logging_worker(self, job):
        try:
            job.wait()
            if self.job is not None:
                raise Exception('We should not be here')
        except gevent.GreenletExit:
            self.redirect_logging(revert=True)
            return
        except:
            self.redirect_logging(revert=True)
            self.job = None
            gevent.sleep(1)
            self.redirect_logging()

    def redirect_logging(self, revert=False):
        root_logger = logging.getLogger('')
        if revert:
            root_logger.handlers[:] = self._orig_log_handlers[:]
            self._orig_log_handlers = None
            self.log.info('Child logger detached')
        else:
            self.job = self.master_cli.call('internal_logger')

            assert self.job.next() == 'ready'

            self._orig_log_handlers = root_logger.handlers[:]
            root_logger.handlers[:] = []
            job_handler = ChildLogHandler(self.job)
            job_handler.setLevel(logging.DEBUG)
            root_logger.addHandler(job_handler)

            self._redirect_logging_worker_grn = gevent.spawn(self.redirect_logging_worker, self.job)

            self.log.info('Child logger attached')
