import time
import py

import gevent

from ..component import Component


class Checker(Component):
    def __init__(
        self, db, locker, cache, announcer,
        blacklist=[], parent=None
    ):
        self.db = db
        self.locker = locker
        self.cache = cache
        self.announcer = announcer
        self.blacklist = blacklist

        self.check_file_every = 3600 * 3
        self.limit_minimal_sleep_between_checks = 0.1  # 100ms or 10 files/sec
        self.limit_files_checked_at_once = 150

        self.stats = {
            'files_checked': 0,
            'files_removed': 0,
            'files_removed_by_bad_mtime': 0,
            'files_removed_by_bad_checksum': 0,
        }

        super(Checker, self).__init__(logname='checker', parent=parent)

    def _check_failed_path_parts(self, item, log):
        fail = True
        failed_part = None
        failed_part_noaccess = None

        for part in reversed(item.path.dirpath().parts()):
            try:
                if part.check(dir=1):
                    break
                else:
                    failed_part = part
            except py.error.EACCES:
                failed_part_noaccess = part

        if failed_part:
            log.debug('Parent directory not exists as well: %s', failed_part)
            if failed_part in self.blacklist:
                log.debug(
                    'File check blacklisted (%s -- parent directory %s dissapeared, blacklisted)',
                    item.path, failed_part
                )
                fail = False
                paths = []
            else:
                paths = self.cache.get_paths_in_dir(failed_part)
                for path in paths:
                    log.debug(
                        'File check failed (%s -- parent directory %s dissapeared)',
                        path, failed_part
                    )
        else:
            if failed_part_noaccess:
                log.debug('We have no access to parent directory: %s', failed_part_noaccess)

                if failed_part_noaccess in self.blacklist:
                    log.debug(
                        'File check failed (%s -- no access to parent directory %s, blacklisted)',
                        item.path, failed_part_noaccess
                    )
                    fail = False
                    paths = []
                else:
                    paths = self.cache.get_paths_in_dir(failed_part_noaccess)
                    for path in paths:
                        log.debug(
                            'File check failed (%s -- no access to parent directory %s)',
                            path, failed_part_noaccess
                        )
            else:
                paths = [item.path.strpath]

        return paths, fail

    def _check_files(self, files_to_check, minimal_sleep_between_checks, log):
        schedule_reannounce = False

        log.debug('Will check %d files', len(files_to_check))

        for idx, item in enumerate(files_to_check):
            next_check = time.time() + minimal_sleep_between_checks
            with self.locker.paths([item.path.strpath], log=False) as bulk_locker:
                assert bulk_locker.next().next() == item.path.strpath

                old_chktime = item.chktime
                with self.db(debug_sql=False):
                    if self.cache.get_item_check_time(item) is None:
                        # maybe we removed this via file -> data -> resource -> datas -> files
                        # chain while checked previous items
                        continue

                try:
                    item.check(validate=True)
                except Exception as ex:
                    log.debug('File check failed (%s -- %s: %s)' % (item.path, type(ex).__name__, ex))
                    fail = True
                else:
                    fail = False

                self.stats['files_checked'] += 1

                with self.db(debug_sql=False, transaction=False):
                    current_chktime = self.cache.get_item_check_time(item)
                    if current_chktime == old_chktime:
                        if fail:
                            paths, fail = self._check_failed_path_parts(item, log)

                            self.stats['files_removed'] += len(paths)

                            dropped_paths, dropped_resources = self.cache.delete_paths(paths)
                            if dropped_resources:
                                schedule_reannounce = True

                        if not fail:
                            self.cache.update_file_check_time(item)
                    else:
                        if fail:
                            log.warning('Item chktime changed in db, will not do anything')

            sleep_time = next_check - time.time()
            if sleep_time > 0:
                gevent.sleep(sleep_time)

        if schedule_reannounce:
            [sched.wakeup() for sched in self.announcer.scheduler_loops]

    @Component.green_loop(logname='housekeep.files')
    def _check_files_loop(self, log):
        while True:
            with self.db(debug_sql=False):
                files_to_check = self.cache.get_files_to_check(
                    count=self.limit_files_checked_at_once,
                    max_chktime=0.1
                )

            if not files_to_check:
                break

            self._check_files(
                files_to_check,
                0,
                log
            )

            gevent.sleep()

        next_loop_run = time.time() + (
            self.limit_files_checked_at_once *
            self.limit_minimal_sleep_between_checks
        )

        with self.db(debug_sql=False):
            files_to_check = self.cache.get_files_to_check(
                count=self.limit_files_checked_at_once,
                max_chktime=(time.time() - self.check_file_every)
            )

        if files_to_check:
            self._check_files(
                files_to_check,
                self.limit_minimal_sleep_between_checks,
                log,
            )

        return max(0, next_loop_run - time.time())

    def on_bad_checksum(self, path, md5):
        with self.db(debug_sql=False):
            stored_md5 = self.db.query_one_col(
                'SELECT d.md5 FROM data d JOIN file f ON f.data = d.id WHERE f.path = ?',
                [path]
            )

            if stored_md5 is not None and stored_md5 != md5:
                self.log.debug(
                    'File check failed (%s -- %s)',
                    path,
                    'bad_checksum %s != %s' % (
                        stored_md5, md5
                    )
                )

                self.db.query('DELETE FROM file WHERE path = ?', [path])
                self.stats['files_removed_by_bad_checksum'] += 1

    def on_bad_mtime(self, path, mtime):
        with self.db(debug_sql=False):
            stored_mtime = self.db.query_one_col(
                'SELECT mtime FROM file WHERE path = ?',
                [path]
            )

            if stored_mtime is not None and stored_mtime != mtime:
                self.log.debug(
                    'File check failed (%s -- %s)',
                    path,
                    'bad_mtime %r != %r' % (
                        stored_mtime, mtime
                    )
                )
                self.db.query('DELETE FROM file WHERE path = ?', [path])
                self.stats['files_removed_by_bad_mtime'] += 1

    def on_bad_file(self, path):
        with self.db(debug_sql=False):
            self.db.query('UPDATE file SET chktime = 0 WHERE path = ?', [path])
