import os
import py
import time

from ..utils import Path

from api.copier import errors


class ResourceItem(object):
    __slots__ = 'file', 'symlink', 'directory'

    def __init__(self, **kwargs):
        for key in self.__slots__:
            setattr(self, key, kwargs.pop(key, None))

        for key, value in kwargs.items():
            setattr(self, key, value)

    @classmethod
    def from_path(cls, path):
        """ Initialize object from path in filesystem. """

        if path.check(link=1):
            self = cls.symlink(path=path, symlink=path.readlink())
        elif path.check(dir=1):
            self = cls.directory()
        else:
            self = cls.file(path=path)

        return self

    @classmethod
    def from_dict(cls, data):
        if data['type'] == 'file':
            self = cls.file.from_dict(data)
        elif data['type'] == 'symlink':
            self = cls.symlink.from_dict(data)
        elif data['type'] == 'directory':
            self = cls.directory.from_dict(data)

        return self

    def to_dict(self):
        result = dict((key, getattr(self, key)) for key in self.__slots__)
        result['type'] = None
        return result


class ResourceFile(ResourceItem):
    __slots__ = 'path', 'inode', 'data', 'size', 'mtime', 'perms', 'md5', 'sha1_blocks', 'chktime'

    def check(self, validate=False):
        if not os.path.isabs(self.path.strpath):
            raise errors.FilesystemError('File or folder %r expected to be absolute path' % self.path)

        try:
            if self.path.check(exists=0, link=1):
                raise errors.FilesystemError('Symlink %r points to non-existent file or folder %r' % (
                    self.path, self.path.readlink()
                ))
            if self.path.check(exists=0, link=0):
                # If target does not exist *AND* not a symlink
                raise errors.FilesystemError('File or folder %r does not exist.' % self.path)
        except py.error.EACCES:
            # Whoops, we completely have no access to this
            raise errors.FilesystemError('No read access to %r' % self.path)

        # Finally, check what item is readable itself by us, but do not bother
        # checking symlinks -- if we can reach it, we can read it.
        if not self.path.check(exists=0, link=1) and not os.access(self.path.strpath, os.R_OK):
            raise errors.FilesystemError('No read access to %r' % self.path)

        if validate:
            assert self.inode is not None, 'missing inode'
            assert self.mtime is not None, 'missing mtime'
            assert self.size is not None, 'missing size'

            stat = self.path.stat()
            if stat.size != self.size:
                raise errors.FilesystemError('size changed %d => %d' % (self.size, stat.size))
            if int(stat.mtime) != int(self.mtime):
                raise errors.FilesystemError('mtime changed %d => %d' % (self.mtime, stat.mtime))
            if stat.ino != self.inode:
                raise errors.FilesystemError('inode changed %d => %d' % (self.inode, stat.ino))

            self.chktime = int(time.time())

        return True

    def stat(self):
        stat = self.path.stat()
        self.mtime = stat.mtime
        self.inode = stat.ino
        self.size = stat.size
        self.perms = stat.mode & 0777

    def make_torrent(self):
        """
        Generate torrent_info structure for libtorrent
        based on self.sha1_blocks.

        This is the main thing which is passed to libtorrent (together with path to
        store files). Bencoded result is the .torrent file in regular torrent clients :).
        """
        assert self.sha1_blocks is not None, 'sha1_blocks not set'
        assert self.sha1_blocks[0] == 1, 'invalid structure version, only v1 supported, got %r' % (
            self.sha1_blocks[0],
        )
        piece_length = self.sha1_blocks[1][0]
        assert all(block[0] == piece_length for block in self.sha1_blocks[1:-1])
        if len(self.sha1_blocks) > 2:
            assert piece_length == 4 * 1024 * 1024

        ti = {
            'name': 'data',
            'piece length': 4 * 1024 * 1024,
            'pieces': ''.join([piece[1] for piece in self.sha1_blocks[1:]]),
            'length': self.size,
        }

        return ti

    def __repr__(self):
        return '<ResourceFile at %s [size=%s mtime=%s perms=%s]>' % (
            self.path if self.path is not None else '?',
            self.size if self.size is not None else '?',
            self.mtime if self.mtime is not None else '?',
            oct(self.perms) if self.perms is not None else '?'
        )

    def to_dict(self):
        result = super(ResourceFile, self).to_dict()
        result['type'] = 'file'
        result['path'] = str(result['path'])
        return result

    @classmethod
    def from_dict(cls, data):
        assert data.pop('type', 'unknown') == 'file'
        self = cls(**data)
        self.path = Path(self.path)
        return self


class ResourceSymlink(ResourceItem):
    __slots__ = 'path', 'symlink'

    def check(self, validate=False):
        return True

    def stat(self):
        return

    def to_dict(self):
        result = super(ResourceSymlink, self).to_dict()
        result['type'] = 'symlink'
        result['path'] = str(result['path'])
        return result

    @classmethod
    def from_dict(cls, data):
        assert data.pop('type', 'unknown') == 'symlink'
        self = cls(**data)
        self.path = Path(self.path)
        return self


class ResourceDirectory(ResourceItem):
    __slots__ = ()

    def check(self, validate=False):
        return True

    def stat(self):
        return

    def to_dict(self):
        result = super(ResourceDirectory, self).to_dict()
        result['type'] = 'directory'
        return result

    @classmethod
    def from_dict(cls, data):
        assert data.pop('type', 'unknown') == 'directory'
        self = cls(**data)
        return self


ResourceItem.file = ResourceFile
ResourceItem.symlink = ResourceSymlink
ResourceItem.directory = ResourceDirectory
