import collections
import contextlib
import functools
try:
    import gevent.coros as coros
except ImportError:
    import gevent.lock as coros
import gevent.queue
import os
import sys
import time
import traceback

from ..component import Component


class BulkLocker(Component):
    """
    Bulk locker

    It locks paths, torrents or files in steps:
        1. Yield locked items which was not locked at all
        2. Every second yield locked items or do not yield anything if was not locked

    Usage:

        >>> locker = BulkLocker()
        >>> path1, path2 = '/etc/path1', '/etc/path2'
        >>> with locker.paths([path1, path2]) as bulk_locker:
        ...    for locked_paths in bulk_locker:
        ...       # do something with locked paths
        ...       pass

    Since BulkLocker is gevent-aware only, deadlock cant occur here.
    """

    def __init__(self, *types, **kwargs):
        super(BulkLocker, self).__init__(logname='blklck', parent=kwargs.get('parent', None))

        self.locks = dict(
            (key, {}) for key in types
        )
        self.locks_backref = {}

    def __getattr__(self, key):
        if key in self.locks:
            return functools.partial(self.lock, key)
        raise AttributeError('%r has no attribute %r' % (self, key))

    def _locker(self, kind, items, locked, log=True):
        unlocked = set()
        seen = set()

        for item in items:
            lock = self.locks[kind].get(item)

            if not lock:
                lock = coros.Semaphore(1)
                self.locks[kind][item] = lock
                self.locks_backref[lock] = (kind, item, None)
                quick = True
            else:
                quick = False

            if (item, lock) in seen:
                # already locked
                continue

            seen.add((item, lock))

            if quick:
                # lock could be already acquired only if it was acquired by hand
                # which is not supported at all.
                assert lock.acquire(blocking=False)
                locked.append((item, lock))
            else:
                unlocked.add((item, lock))

        if log:
            self.log.debug('Quicklock %d %s, wait %d %s', len(locked), kind, len(unlocked), kind)

            for item in unlocked:
                self.log.debug('  wait : %s', item[0])

        if unlocked:
            lock_queue = gevent.queue.Queue()

        try:
            for item, lock in unlocked:
                backref = self.locks_backref[lock]

                deq = backref[2]
                if deq is None:
                    deq = collections.deque()
                    self.locks_backref[lock] = (backref[0], backref[1], deq)

                deq.append(lock_queue.put)

            if locked:
                yield (item for item, lock in locked)

            if not unlocked:
                return

            while unlocked:
                deadline = time.time() + 0.5
                pairs = []
                while time.time() < deadline and unlocked:
                    timeout = deadline - time.time()
                    try:
                        itemlock_pair = lock_queue.get(timeout=timeout)
                        if log:
                            self.log.debug('  lock (w): %s', itemlock_pair[0])
                        assert itemlock_pair[1].acquire(blocking=False)
                        unlocked.discard(itemlock_pair)
                        pairs.append(itemlock_pair)
                        locked.append(itemlock_pair)
                    except gevent.queue.Empty:
                        break

                if pairs:
                    if log:
                        self.log.debug('Locked %d %s', len(pairs), kind)
                    yield (item for item, lock in pairs)

        except BaseException as ex:
            # Oops, something happened
            # This will possibly be timeout during lock_queue.get
            # Or any other not-us exception raised during yielding
            #
            # We need to grab everything from queue (if any)
            # Fill up locked list (this will be cleaned later)
            # And remove any outstanding waiters
            if unlocked:
                if log:
                    self.log.debug('Error: %s: %s, cleaning up...', type(ex).__name__, ex)

                exc = sys.exc_info()

                while True:
                    try:
                        # First, grab all locks which we can grab without blocking
                        # This will be "good locks", which should be unlocked as usual
                        # (that is done in finally block in lock() meth)
                        itemlock_pair = lock_queue.get_nowait()
                        assert itemlock_pair[1].acquire(blocking=False)
                        unlocked.discard(itemlock_pair)
                        locked.append(itemlock_pair)
                    except gevent.queue.Empty:
                        break

                # Everything we wait, but not yet locked -- just discard
                if unlocked:
                    for item, lock in unlocked:
                        self.locks_backref[lock][2].remove(lock_queue.put)

                    if log:
                        self.log.debug('Discarded %d %s locks (error: %s)', len(unlocked), kind, str(ex))

                raise exc[0], exc[1], exc[2]
            raise

    @contextlib.contextmanager
    def lock(self, kind, items, log=True):
        if log:
            self.log.debug('Lock %d %s', len(items), kind)

        locked = []
        ex = None

        try:
            gena = self._locker(kind, items, locked, log=log)
            yield gena
        except BaseException as ex:
            # Rethrow exception to locker generator right now, or it will raise GeneratorExit
            # but rpc job will be destroyed at that moment and we will not be able to log
            # job id properly
            ei = sys.exc_info()
            gena.throw(ei[0], ei[1], ei[2])

            # Actually this point should not be never met, since locker generator reraises exceptions
            # by itself.
            raise
        finally:
            if log:
                self.log.debug(
                    'Releasing %d %s locks (%s)', len(locked), kind,
                    'ok' if not ex else 'error: %s' % (ex, )
                )

            wakeups = []
            try:
                for item, lock in locked:
                    backref = self.locks_backref.get(lock)

                    if backref:
                        kind, item, waiters = backref

                        if waiters:
                            waiter = waiters.popleft()
                            lock.release()
                            wakeups.append((waiter, (item, lock)))
                        else:
                            # nobody waits this lock anymore
                            self.locks_backref.pop(lock)
                            self.locks[kind].pop(item)
            finally:
                # Since we rely on waiters very heavily and waiter
                # will switch current greenlet out (coz, there is queue.put)
                # we are running all waiters even if errors occur (at least 100 times =))
                # and if error was -- reraise it at the end.
                error = False

                for w in wakeups:
                    for i in range(50):  # 5 secs of stale max
                        try:
                            w[0](w[1])
                        except BaseException:
                            self.log.warning('Unable to wakeup waiter: %s (try #%d)', traceback.format_exc(), i + 1)
                            time.sleep(0.1)  # do not switch current greenlet out
                            error = True
                        else:
                            break

                    if error:
                        self.log.critical('Unable to wakeup waiters, emergency exit')
                        os._exit(1)
