import gevent
try:
    import gevent.coros as coros
except ImportError:
    import gevent.lock as coros
import time

from .announcer import Announcer
from .cache import Cache
from .hasher import Hasher
from .locker import BulkLocker
from .resource import Resource
from .checker import Checker

from ..component import Component
from ..utils import Path


class ResourceManager(Component):
    def __init__(
        self, uid, desc, workdir, db, trackers,
        data_port, announce_port, main_lt_port, proxy_port,
        proxy_uds, proxy_uds_abstract,
        skybit_data_uds,
        ips,
        dl_binary, hasher_binary,
        netproxy_proc,
        net_priorities,
        max_connections,
        dl_write_cache_size,
        dl_write_cache_expiry,
        dl_preallocate_files,
        filechecker_blacklist,
        deduplicate_nocheck,
        parent=None
    ):
        super(ResourceManager, self).__init__(logname='res', parent=parent)

        self.uid = uid
        self.desc = desc
        self.workdir = workdir
        self.db = db
        self.trackers = trackers
        self.data_port = data_port
        self.skybit_data_uds = skybit_data_uds
        self.ips = ips
        self.has_fb = self.ips['fb'] or self.ips['fb6']

        self.cache = Cache(db, parent=self)
        self.hasher = Hasher(hasher_binary, parent=self)
        self.locker = BulkLocker('paths', 'resources', parent=self)
        self.dl_binary = dl_binary
        self.dl_write_cache_size = dl_write_cache_size
        self.dl_write_cache_expiry = dl_write_cache_expiry
        self.dl_preallocate_files = dl_preallocate_files
        self.deduplicate_nocheck = deduplicate_nocheck

        dl_max = 30
        self.dl_sem = coros.Semaphore(dl_max)
        self.dl_slots = range(1, dl_max + 1)

        self.netproxy_proc = netproxy_proc

        self._max_connections = max_connections

        self._main_lt_port = main_lt_port
        self._proxy_port = proxy_port
        self._proxy_uds = proxy_uds
        self._proxy_uds_abstract = proxy_uds_abstract

        self.announcer = Announcer(
            self.uid, db, trackers=trackers, ips=self.ips,
            port=announce_port, data_port=self.data_port, parent=self
        )

        self.filechecker = Checker(
            db=self.db,
            locker=self.locker,
            cache=self.cache,
            announcer=self.announcer,
            blacklist=[Path(path) for path in filechecker_blacklist],
            parent=self
        )

        self.active_state_skbt = {}  # uid => (skbt_uds)

        self.active_heads = {}  # uid => gevent.event.AsyncResult

        self.stats = {
            'lt': {
                'too_many_conns': 0,
            },
            'downloaders': {
                'dl': 0,
                'ul': 0,
                'cp': 0,
                'dd': 0,
                'skybit': {
                    'ok': 0,
                    'error': 0,
                },
            }
        }

    def start(self):
        ret = super(ResourceManager, self).start()

        return ret

    @Component.green_loop(logname='housekeep.metacache')
    def _cleanup_metacache_loop(self, log):
        # Once we stored cache for 1 hour -- clean it
        cleanup_cache_after_secs = 3600

        next_check = time.time() + 1800

        metacache_dir = self.workdir.join('metacache')
        if not metacache_dir.check(exists=1) or not metacache_dir.check(dir=1):
            self.log.warning('Metacache dir is not exists or not a directory')
            return 300

        cleaned = 0

        for parent in metacache_dir.listdir():
            if not parent.check(exists=1):
                # was removed since last listdir(), this is not a problem
                gevent.sleep(1)
                continue

            if len(parent.basename) != 2:
                log.warning('Removing invalid metacache directory (invalid name): %s', parent)
                parent.remove()
                gevent.sleep(1)
                continue

            if not parent.check(dir=1):
                log.warning('Removing invalid metacache item (not a dir): %s', parent)
                parent.remove()
                gevent.sleep(1)
                continue

            metadirs = parent.listdir()
            if not metadirs:
                changed_ago = time.time() - parent.stat().mtime
                if changed_ago > 60:
                    log.debug(
                        'Removing empty metacache directory: %s (mtime %d seconds ago)',
                        parent, changed_ago
                    )
                    parent.remove()
                gevent.sleep(1)
                continue

            for metadir in metadirs:
                with self.locker.resources([metadir.basename], log=False) as bulk_locker:
                    assert bulk_locker.next().next() == metadir.basename

                    if not metadir.check(exists=1):
                        # was removed while we were sleeping, not a problem
                        gevent.sleep(1)
                        continue

                    if not metadir.check(dir=1):
                        log.warning('Removing invalid metacache item (not a dir): %s', metadir)
                        metadir.remove()
                        gevent.sleep(1)
                        continue

                    # Take into account both inode and contents change
                    last_change = max(metadir.stat().mtime, metadir.stat().ctime)
                    changed_ago = time.time() - last_change
                    if changed_ago < cleanup_cache_after_secs:
                        gevent.sleep(1)
                        continue

                    log.debug('Removing metacache %s (changed %ds ago)', metadir, changed_ago)
                    metadir.remove()
                    cleaned += 1

        if cleaned:
            log.info('Cleaned %d metacache dirs', cleaned)

        return max(next_check - time.time(), 0)

    def lookup_infohash(self, infohash, skybit):
        if skybit:
            if infohash in self.active_state_skbt:
                endpoint = self.active_state_skbt[infohash]
                self.log.debug('[resid:%s]  Skybit Lookup (found child at uds:%s)', infohash[:8], endpoint)

                # Cache for just 1 sec, because downloader may die soon
                return endpoint, 1

            else:
                self.log.debug('[resid:%s]  Lookup (forcing skybit daemon)', infohash[:8])
                # In skybit mode we send any connection to main skybit daemon
                # So, it will handle everything itself.
                return self.skybit_data_uds, 120

        else:
            self.log.warning('[trnt:%s]  We do not support LT lookups anymore', infohash[:8])
            # Negative lookups should be cached for 1 sec max
            return None, 1

    def lookup_resource(self, uid):
        self.log.debug('[resid:%s] Lookup resource from skybit', uid[:8])
        resource = self._on_lt_unknown_hash(uid)
        if resource:
            head = resource.data.dbdict()
            head.pop('torrent_info')
            return head
        return None

    def _on_lt_unknown_hash(self, infohash):
        """
        It is a bit weird to have this logic in manager, but there is no easy way
        to do this now.

        Later we should either review this logic, either store infohash in db,
        so cache will be able to lookup by itself, either store resource id in libtorrent
        and tell about that resource in connect packet.
        """
        with self.db:
            # First try to find exact resource
            with self.db(transaction=False, debug_sql=False):
                resource = Resource.from_id(infohash, self.cache, self.log)

            if resource:
                if not resource.check(self.cache):
                    # If check failed we probably removed some infohashes from db (including resource itself)
                    # Thus, wakeup scheduler loops
                    [sched.wakeup() for sched in self.announcer.scheduler_loops]
                    resource = None
                else:
                    return resource

        return resource
