import collections
import hashlib
import logging
import time
import cEcdsa

from api.copier import errors

from .file import ResourceItem
from .. import bencode
from ..utils import Path
from ..logger import SmartLoggerAdapter


class ResourceLoggerAdapter(SmartLoggerAdapter):
    def process(self, msg, kwargs):
        uid = self.extra.get('resource_uid')
        if not uid:
            uid = self.extra['resource'].uid

        return (
            '[resid:%s] %s' % (uid[:8], msg),
            kwargs
        )


class Resource(object):
    """ Resource of type rbtorrent1

    uid: resource id
    items: {'path/file': <ResourceItem>}
    data: <Rbtorrent1>
    """

    __slots__ = 'uid', 'items', 'data', 'sign_key', 'log'

    class RbTorrent1(object):
        """
        structure: {'path/in/resource', <descriptoin>'}
        torrents: {'infohash': <item ti>}
        torrent_info: <metatorrent ti>
        """

        __slots__ = 'structure', 'torrents', 'torrent_info', 'sign_key'

        def __init__(self, **kwargs):
            if 'torrent_info' not in kwargs:
                kwargs['torrent_info'] = None

            for key in self.__slots__:
                if key == 'sign_key' and not kwargs.get('sign_key'):
                    continue

                setattr(self, key, kwargs[key])

        def infohash(self):
            return hashlib.sha1(bencode.bencode(self.torrent_info)).hexdigest()

        def dbdict(self):
            result = {}

            for key in self.__slots__:
                if key == 'sign_key' and (not hasattr(self, key) or not getattr(self, key)):
                    continue
                else:
                    result[key] = getattr(self, key)

            return result

        def get_torrent(self, infohash):
            return self.torrents[infohash]

        def get_binary_data(self):
            return bencode.bencode({
                'structure': self.structure,
                'torrents': self.torrents,
            })

        def info(self):
            info = {'size': 0, 'files': [], 'dirs': [], 'symlinks': []}
            total_size = 0

            for path, item in self.structure.items():
                if item['type'] == 'file':
                    item_size = None

                    if item['resource']['type'] == 'symlink':
                        info['symlinks'].append({
                            'path': path,
                            'link': item['resource']['data'].split(':', 1)[1],
                        })
                    else:
                        if item['resource']['type'] == 'torrent':
                            torrent = self.torrents[item['resource']['id']]
                            item_size = torrent['info']['length']
                            total_size += item_size
                        elif item['resource']['type'] == 'touch':
                            item_size = 0

                        if item_size is not None:
                            info['files'].append({
                                'path': path,
                                'size': item_size,
                                'executable': any([item['mode'][n] in ('x', 's') for n in range(2, 9, 3)]),
                                'md5sum': (
                                    item['md5sum'].encode('hex')
                                    if ('md5sum' in item and item['md5sum'])
                                    else None
                                )
                            })
                elif item['type'] == 'dir':
                    info['dirs'].append({
                        'path': path,
                    })

            info['size'] = total_size
            return info

    def __init__(self, uid, items, data=None, logger=None):
        self.uid = uid
        self.items = items

        base_logger = logger.getChild('resource') if logger else logging.getLogger('resource')

        self.log = ResourceLoggerAdapter(base_logger, {'resource': self})

        if data:
            self.data = self.RbTorrent1(**data)
        else:
            self.data = None

    @classmethod
    def _generate_rbtorrent1(cls, structure, torrents, sign_key):
        """
        Generates data of type RbTorrent1
        Resulting data is the Resource.rbtorrent1 class with fields:
         - structure: dict with torrent structure
         - torrents: dict with all torrents inside resource
         - torrent_info: ti data for libtorrent (so, he will be able to download it)
        """
        rbtorrent_dict = {'structure': structure, 'torrents': torrents}

        if sign_key:
            rbtorrent_dict['sign_key'] = sign_key.public_key().to_raw()

        rbtorrent_binary = bencode.bencode(rbtorrent_dict)

        piece_size = 4 * 1024 * 1024
        num_pieces = len(rbtorrent_binary) // piece_size
        if len(rbtorrent_binary) % piece_size:
            num_pieces += 1

        pieces = []
        for idx in xrange(num_pieces):
            start = idx * piece_size
            end = start + piece_size

            data = rbtorrent_binary[start:end]
            pieces.append(hashlib.sha1(data).digest())

        return cls.RbTorrent1(
            structure=structure,
            torrents=torrents,
            torrent_info={
                'name': 'metadata',
                'piece length': piece_size,
                'pieces': ''.join(pieces),
                'length': len(rbtorrent_binary)
            },
            sign_key=(sign_key.public_key().to_raw()) if sign_key else None
        )

    def _generate(self):
        """
        Generate resource of type "RbTorrent1".

        This one takes all items in this resource, put them into
        binary form, generates torrent (aka "MetaTorrent") and calculates
        sha1 hash of that torrent ("infohash").
        """

        assert not hasattr(self, 'data'), 'Resource already generated'

        structure = {}
        torrents = {}

        counts = collections.defaultdict(int)

        for name, item in self.items.iteritems():
            item_dict = {
                'path': name,
                'resource': {},
            }

            if isinstance(item, ResourceItem.file):
                item_dict['type'] = 'file'
                item_dict['md5sum'] = item.md5
                item_dict['size'] = item.size
                item_dict['executable'] = item.perms & 0o111 != 0
                item_dict['mode'] = 'rwxr-xr-x' if item_dict['executable'] else 'rw-r--r--'

                if item.size > 0:
                    torrent_info = item.make_torrent()
                    infohash = hashlib.sha1(bencode.bencode(torrent_info)).digest()

                    item_dict['resource']['type'] = 'torrent'
                    item_dict['resource']['id'] = infohash.encode('hex')

                    torrents[infohash.encode('hex')] = {'info': torrent_info}
                else:
                    item_dict['resource']['type'] = 'touch'

                structure[name] = item_dict

                counts['file'] += 1
                counts['size'] += item.size

            elif isinstance(item, ResourceItem.directory):
                item_dict['type'] = 'dir'
                item_dict['size'] = -1
                item_dict['executable'] = True
                item_dict['mode'] = 'rwxr-xr-x'
                item_dict['resource']['type'] = 'mkdir'
                item_dict['resource']['data'] = 'mkdir'
                item_dict['resource']['id'] = 'mkdir:%s' % (name, )

                structure[name] = item_dict

                counts['directory'] += 1

            elif isinstance(item, ResourceItem.symlink):
                item_dict['type'] = 'file'
                item_dict['size'] = 0
                item_dict['mode'] = ''
                item_dict['executable'] = True
                item_dict['resource']['type'] = 'symlink'
                item_dict['resource']['data'] = 'symlink:%s' % (item.symlink, )

                structure[name] = item_dict

                counts['symlink'] += 1

        if counts['file'] == 0:
            ex = errors.UnshareableResource(
                'Cowardly refusing creating resource without any non-zero files '
                '(files: %(file)d, dirs: %(directory)d, symlinks: %(symlink)d)' % counts
            )
            raise ex

        self.data = self._generate_rbtorrent1(structure, torrents, self.sign_key)
        return hashlib.sha1(bencode.bencode(self.data.torrent_info)).hexdigest(), counts

    @classmethod
    def create(cls, files, db, cache, hasher, locker, logger, progress=None, cache_dict=None, sign=False):
        """
        Create new resource, grabbing some info from cache if possible.
        Files with missing metadata will be rehashed

        Arguments:
            files: list of tuples with 2 items
            - real path in filesystem
            - "name" for resource (i.e. path inside resource)
        """

        self = cls.__new__(cls)
        self = cls(None, {}, logger=logger)

        del self.data
        del self.uid

        if sign:
            if isinstance(sign, bool):
                sign_key = cEcdsa.Key.generate(256)
                self.sign_key = sign_key
            else:
                self.sign_key = cEcdsa.Key.from_raw(sign)
        else:
            self.sign_key = None

        files = set(files)

        file_names = collections.defaultdict(list)
        for rpath, name in files:
            # some normalization
            #  - some/path/../with/../doubles => some/doubles
            #  - some/path/../../with/megadoubles => with/megadoubles
            #  - ../../path => path
            #  - ./some/curdir => some/curdir
            #  - some/dir//////// => some/dir
            #  - some////other//weird => some/other/weird
            parts = name.split('/')
            skip = 0
            for idx, part in reversed(list(enumerate(parts))):
                if part in ('', '.'):
                    del parts[idx]
                elif part == '..':
                    skip += 1
                    del parts[idx]
                elif skip:
                    skip -= 1
                    del parts[idx]

            name = '/'.join(parts)

            file_names[rpath].append(name)

        if not locker:
            import contextlib

            class Locker(object):
                @contextlib.contextmanager
                def paths(self, paths):
                    yield [paths]
            locker = Locker()

        state = {
            'total_bytes': 0,
            'hashed_bytes': 0,
            'hashed_files': 0
        }

        if progress:
            for rpath, _ in files:
                if not rpath.check(link=1):
                    state['total_bytes'] += rpath.stat().size

            def _on_progress(done_files, hashed_bytes):
                state['hashed_bytes'] += hashed_bytes
                progress('hash_files', state['hashed_bytes'], state['total_bytes'])

            _on_progress(0, 0)
        else:
            _on_progress = None

        with locker.paths([rpath for rpath, _ in files]) as bulk_locker:
            for paths in bulk_locker:
                items = []
                for path in paths:
                    item = ResourceItem.from_path(path)
                    item.check()
                    item.stat()

                    items.append(item)

                    for name in file_names[path]:
                        self.items[name] = item

                cache.load_info_from_db(items, cache_dict=cache_dict)

                need_hashing = []
                for item in items:
                    if isinstance(item, ResourceItem.file):
                        if item.data is None:
                            need_hashing.append(item)
                        elif progress:
                            _on_progress(0, item.size)

                    elif isinstance(item, (ResourceItem.directory, ResourceItem.symlink)):
                        pass
                    else:
                        raise errors.CopierError('Dont know what to do with item type %r' % (type(item), ))

                for item, (md5, sha1_blocks, (_, size, _, _, mtime)) in hasher.rehash_in_bulk(
                    need_hashing, progress=_on_progress
                ):
                    item.size = size
                    item.mtime = mtime
                    item.chktime = int(time.time())
                    item.md5 = md5
                    item.sha1_blocks = sha1_blocks

            self.uid, counts = self._generate()
            cache.store_resource(self)

        return self, {'counts': counts}

    @classmethod
    def from_id(cls, uid, cache, logger, load_items=True):
        typ, data = cache.get_resource(uid)
        if typ is None and data is None:
            return

        assert typ == 'rbtorrent1'

        if load_items:
            items = cache.get_resource_items(uid)
        else:
            items = None

        return cls(uid, items, data, logger)

    @classmethod
    def get_item_by_infohash(cls, infohash, cache, logger):
        uid, typ, data = cache.get_resource_by_child_infohash(infohash)

        if typ is None and data is None:
            return None, None

        assert typ == 'rbtorrent1'

        for name, info in data['structure'].iteritems():
            if info['resource']['type'] == 'torrent' and info['resource']['id'] == infohash:
                ti = data['torrents'][infohash]['info']
                break
        else:
            assert 0

        return cache.get_item_by_resource_and_item_name(uid, name), ti

    def check(self, cache):
        failed_items = {}
        failed_items_by_data = {}

        for name, item in self.items.iteritems():
            try:
                item.check(validate=True)
            except Exception as ex:
                if isinstance(item, ResourceItem.file):
                    item_desc = '%s: %s' % (name, item.path)
                else:
                    item_desc = name

                self.log.debug(
                    '%s failed check (%s)', item_desc, str(ex)
                )
                failed_items[name] = item
                failed_items_by_data.setdefault(item.data, (item, []))[1].append(name)

        if not failed_items:
            return True

        failed_paths = set(item.path for item in failed_items.itervalues())

        for data, path_alternatives in cache.get_path_alternatives(
            failed_items_by_data.keys()
        ).items():
            for path, inode, mtime in path_alternatives:
                if path in failed_paths:
                    continue
                item, names = failed_items_by_data[data]
                item.path = Path(path)
                item.mtime = mtime
                item.inode = inode

                try:
                    self.log.debug('Checking alternative path %s', item.path)
                    item.check(validate=True)
                except Exception as ex:
                    for name in names:
                        self.log.debug(
                            '%s failed check (%s: %s)',
                            name, item.path, str(ex)
                        )
                    failed_paths.add(item.path)
                else:
                    for name in names:
                        del failed_items[name]
                    cache.update_file_check_time(item)
                    break

        if failed_paths:
            cache.delete_paths(failed_paths)

        if failed_items:
            self.log.info('Check failed (still has failed_items)')
            return False

        self.log.info('Check ok (using alternatives)')
        return True

    @classmethod
    def head(cls, uid, db, cache, locker, logger):
        log = ResourceLoggerAdapter(logger, {'resource_uid': uid})
        log.info('head request')

        # First try to look into db, maybe we have that resource
        resource = cls.from_id(uid, cache, logger, load_items=False)
        if resource:
            return resource.data

    def to_dict(self):
        result = {}
        for key in self.__slots__:
            if key == 'sign_key' and not hasattr(self, 'sign_key'):
                continue

            value = getattr(self, key)

            if key == 'data':
                if isinstance(value, self.RbTorrent1):
                    result['data'] = value.dbdict()
                    result['data_type'] = 'rbtorrent1'

            elif key == 'items':
                result['items'] = {}
                for name, item in value.iteritems():
                    result['items'][name] = item.to_dict()

            elif key == 'log':
                continue

            else:
                result[key] = value

        return result

    @classmethod
    def from_dict(cls, data):
        assert 'uid' in data
        assert 'items' in data
        assert 'data' in data
        assert 'data_type' in data
        assert data['data_type'] == 'rbtorrent1'

        seen = {}
        for name, item in data['items'].iteritems():
            if 'path' in item:
                if item['path'] not in seen:
                    seen[item['path']] = ResourceItem.from_dict(item)
                data['items'][name] = seen[item['path']]
            else:
                data['items'][name] = ResourceItem.from_dict(item)

        self = cls(data['uid'], data['items'], data=data['data'])
        return self

    @classmethod
    def xxfrom_id(cls, uid, cache, logger, load_items=True):
        typ, data = cache.get_resource(uid)
        if typ is None and data is None:
            return

        assert typ == 'rbtorrent1'

        if load_items:
            items = cache.get_resource_items(uid)
        else:
            items = None

        return cls(uid, items, data, logger)
