from __future__ import absolute_import

import collections
import gc
import gevent
import greenlet
import logging
import msgpack
import os
import resource
import sys
import time

from api.copier import errors

try:
    import gevent.coros as coros
except ImportError:
    import gevent.lock as coros

from . import subprocess_gevent as subproc

from .resource.file import ResourceItem
from .resource.resource import Resource

from ..kernel_util.sys.gettime import monoTime
from ..kernel_util.sys.user import userPrivileges as user_privileges  # noqa
from ..kernel_util.sys.portoslave import same_container

from .skbn import encryption
from .utils import Path, has_root

from ..rpc.utils import pid_starttime


iosem = coros.Semaphore(1)


def get_caller_creds(job):
    return job._RPCJob__conn._Connection__sock.peer_creds


def get_caller_uid(job):
    return get_caller_creds(job)['uid']


def get_caller_pid(job):
    return get_caller_creds(job)['pid']


def get_caller_container(portoconn, job):
    """
        returns resolved (bool), container name (string or None)

        if not resolved -- unable to grab container if porto is not available
        if resolved and None -- we are in same container as caller

        final case is resolved and container name detected via porto
    """
    pid = get_caller_pid(job)

    if same_container(pid, files=True, procs=False):
        return True, None

    if portoconn:
        return True, portoconn.LocateProcess(pid).name

    return False, None


def disallow_other_containers(portoconn, job, allow_only_root=False):
    """
        Raises errors.ApiError if job comes from different container.
    """
    resolved, container = get_caller_container(portoconn, job)
    if not resolved:
        # Silly check, but safest if some dump people will change logic in get_caller_container()
        assert container is None

        raise errors.ApiError(
            'RPC method not allowed from other container (%s) -- '
            'we are unable to locate processes without porto' % (
                container,
            )
        )

    if allow_only_root:
        if container is not None and container != '/':
            raise errors.ApiError('RPC method not allowed from non-root porto container (%s)' % (container, ))

        creds = get_caller_creds(job)
        if creds['starttime'] != pid_starttime(creds['pid']):
            raise errors.ApiError('RPC method not allowed -- we are unable to recheck client pid')

    elif container is not None:
        # Silly check, but raises if portoconn is not available here
        assert portoconn is not None

        our_container = portoconn.LocateProcess(os.getpid()).name

        if container != our_container:
            raise errors.ApiError('RPC method not allowed from other porto container (%s)' % (container, ))


def get_caller_lxc(job):
    """
        returns LXC container name (string or `None` in case of no LXC container detected)

        if not resolved -- unable to determine LXC container
        final case is resolved and LXC container name detected
    """
    if os.uname()[0].lower() != 'linux':
        return

    pid = get_caller_pid(job)

    try:
        with open("/".join(["", "proc", str(pid), "cgroup"]), "r") as fh:
            for line in fh:
                parts = line.split(":", 3)
                if len(parts) == 3 and parts[0] == "1" and parts[2].startswith("/lxc/"):
                    return parts[2].rstrip()[5:]
    except (OSError, IOError):
        pass


def get_lxc_root(name):
    """ returns LXC container's root location for the given container name """

    try:
        with open("/".join(["", "var", "lib", "lxc", name, "config"]), "r") as fh:
            for l in fh:
                parts = l.split("=", 2)
                if len(parts) == 2 and parts[0].strip() == "lxc.rootfs":
                    return parts[1].strip().rstrip("/") + "/"
    except (OSError, IOError):
        pass


class SkyboneRPC(object):
    def __init__(self, daemon):
        self.daemon = daemon

    def io_semaphore(self, job):
        self.daemon.active.wait()

        aq = gevent.queue.Queue()

        cn = [0]

        def _acquirer():
            while True:
                aq.get()
                iosem.acquire()
                cn[0] += 1
                job.state(True)

        grn = gevent.spawn(_acquirer)

        try:
            while True:
                op = job.feedback

                if op == 'acquire':
                    aq.put(1)
                elif op == 'release':
                    iosem.release()
                    cn[0] -= 1
        finally:
            grn.kill()
            grn.join()

            for _ in range(cn[0]):
                iosem.release()


class SkyboneProcRPC(object):
    def __init__(self, daemon):
        self.daemon = daemon
        self.log = daemon.log.getChild('rpc.skybone')

        # resid => set of pids
        # This map tracks skybone-dl processes that haven't yet finished the trycopy stage.
        self._trycopy_waiters = collections.defaultdict(set)

    def get_trycopy_waiters(self, resid):
        return list(self._trycopy_waiters.get(resid, []))

    def lock_files(self, job, files):
        self.daemon.active.wait()
        job.state(('start', None))

        converted = self._cv_paths(job, files)
        converted_back = {}

        filelist = []
        for path in files:
            if path in converted:
                filelist.append(converted[path])
                converted_back[converted[path]] = path
            else:
                filelist.append(path)

        with self.daemon.resource_mngr.locker.paths(filelist) as bulk_locker:
            for paths in bulk_locker:
                result = []
                for path in paths:
                    # If somebody locked py.path.local() object before us, we will
                    # get it, instead plain string. Thus, will fail to send state
                    # via msgpack. So, convert list of files to plain strings here
                    if str(path) in converted_back:
                        path = converted_back[path]
                    result.append(path)
                job.state(('locked', result))

            job.state(('done', None))
            done = job.feedback
            assert done

    def cache_load_info_from_db(self, job, items):
        self.daemon.active.wait()
        for idx, item in enumerate(items):
            items[idx] = ResourceItem.from_dict(item)

        self.daemon.resource_mngr.cache.load_info_from_db(items)

        for idx, item in enumerate(items):
            items[idx] = item.to_dict()

        return items

    def _porto_lxc_convert_paths(self, job, paths):
        self.log.info('Attempt to convert %d path(s) from porto/lxc', len(paths))

        convert_method = 'unknown'
        result = {}

        for path in paths:
            result[path] = None

        if self.daemon.porto:
            convert_method = 'porto'

            try:
                resolved, container = get_caller_container(self.daemon.porto, job)
            except Exception as ex:
                self.log.warning('We have porto, but unable to connect to it: %s' % (ex, ))
                resolved, container = False, False

            if resolved and container:
                try:
                    my_container = self.daemon.porto.LocateProcess(os.getpid()).name
                except Exception as ex:
                    self.log.warning('Unable to locate child process thru porto: %s' % (ex, ))
                else:
                    result = {}

                    for path in paths:
                        try:
                            newpath = self.daemon.porto.ConvertPath(path, container, my_container)
                        except Exception as ex:
                            self.log.warning('Unable to convert path: %s', ex)
                        else:
                            result[path] = newpath

            elif not resolved:
                # We are in different containers, but unable to grab caller container via porto.
                # Just continue here adding file as-is. This would work if container rootfs is **the same** as
                # ours. But will not work if rootfs differs.
                pass
        else:
            # Check the process is running in LXC
            lxc_root = None
            lxc = get_caller_lxc(job)

            if lxc:
                convert_method = 'lxc'
                self.log.debug('Detected call from LXC container %r', lxc)

                try:
                    lxc_root = get_lxc_root(lxc)
                except Exception as ex:
                    self.log.warning('Unable to detect lxc root: %s', ex)
                    lxc_root = None

                if lxc_root:
                    result = {}
                    for path in paths:
                        newpath = lxc_root + path
                        result[path] = newpath

        return result, convert_method

    def _cv_check_path_iteminfo(self, path, iteminfo):
        if os.path.exists(path):
            try:
                fstat = os.stat(path)
            except:
                return False

            if 'size' in iteminfo and iteminfo['size'] != fstat.st_size:
                return False

            if 'mtime' in iteminfo and iteminfo['mtime'] != fstat.st_mtime:
                return False

            return True
        else:
            return False

    def _cv_check_path_inodev(self, path, inode, dev):
        if os.path.exists(path):
            try:
                fstat = os.stat(path)
            except:
                return False

            if fstat.st_ino != inode or fstat.st_dev != dev:
                return False

            return True
        else:
            return False

    def _cv_paths(self, job, paths):
        # Paths is a dict with key=path and value (inode, dev)
        # Second for of paths is a list of dicts (iteminfo's)
        # Result is a dict with {orig_path: cv_path}

        self.daemon.active.wait()

        cnt_ok = 0  # count items which we do not need to convert
        cnt_cv = 0  # count items which need conversion
        cnt_po = 0  # count items converted
        cnt_ee = 0  # count items failed to convert

        need_convert = {}
        result = {}

        for path, iteminfo in paths.iteritems():
            if isinstance(iteminfo, dict):
                if self._cv_check_path_iteminfo(path, iteminfo):
                    cnt_ok += 1
                else:
                    need_convert[path] = None
                    cnt_cv += 1
            else:
                if self._cv_check_path_inodev(path, iteminfo[0], iteminfo[1]):
                    cnt_ok += 1
                else:
                    need_convert[path] = None
                    cnt_cv += 1

        if need_convert:
            paths_to_convert = []
            for path, iteminfo in paths.iteritems():
                if isinstance(iteminfo, dict):
                    if 'path' in iteminfo:
                        if iteminfo['path'] not in need_convert:
                            continue
                        paths_to_convert.append(iteminfo['path'])
                else:
                    paths_to_convert.append(path)

            if paths_to_convert:
                converted_paths, convert_method = self._porto_lxc_convert_paths(job, paths_to_convert)

                for path, iteminfo in paths.iteritems():
                    if isinstance(iteminfo, dict):
                        if 'path' not in iteminfo or iteminfo['path'] not in need_convert:
                            continue
                    else:
                        if path not in need_convert:
                            continue

                    if isinstance(iteminfo, dict):
                        newpath = converted_paths[iteminfo['path']]

                        if newpath and self._cv_check_path_iteminfo(newpath, iteminfo):
                            result[iteminfo['path']] = newpath
                            iteminfo['path'] = newpath
                            cnt_po += 1
                        else:
                            cnt_ee += 1
                    else:
                        newpath = converted_paths[path]

                        if newpath and self._cv_check_path_inodev(newpath, iteminfo[0], iteminfo[1]):
                            result[path] = newpath
                            cnt_po += 1
                        else:
                            cnt_ee += 1

                self.log.info(
                    'Resource path convert: %d ok, %d need conversion, '
                    '%d converted, %d failed to convert (method:%s)',
                    cnt_ok, cnt_cv, cnt_po, cnt_ee, convert_method
                )
                if cnt_ee:
                    self.daemon.metric_pusher.update_counter('convert_path_errors', 1)
            else:
                self.log.info('Resource path conversion needed by we found no paths to convert')
        else:
            self.log.info('Resource path convertion not needed')
        return result

    def _convert_alternative_paths(self, job, alternatives):
        if os.uname()[0].lower() != 'linux':
            return

        resolved, container = get_caller_container(self.daemon.porto, job)
        if not resolved or not container:
            return
        root_path = self.daemon.porto.GetProperty(container, 'root_path')

        for paths in alternatives.itervalues():
            new_paths = []
            for path, inode, mtime in paths:
                if path.startswith(root_path):
                    new_paths.append((path[len(root_path):], inode, mtime))
            paths.extend(new_paths)

    def cache_store_resource(self, job, data):
        self.daemon.active.wait()

        paths_to_convert = {}

        for item, iteminfo in data['items'].iteritems():
            if 'path' in iteminfo:
                path = iteminfo['path']
                paths_to_convert[path] = iteminfo

        converted = self._cv_paths(job, paths_to_convert)

        for item, iteminfo in data['items'].iteritems():
            if 'path' not in iteminfo:
                continue
            path = iteminfo['path']
            if path in converted:
                iteminfo['path'] = converted[path]

        if not self.daemon.cfg.loose_path_checking:
            # Check paths trying to just open them in readonly mode
            # Do this only if loose path checking not set in global configuration
            for item, iteminfo in data['items'].iteritems():
                if 'path' in iteminfo:
                    path = iteminfo['path']

                    if 'mtime' in iteminfo:  # do not check symlinks, etc.
                        try:
                            open(path, mode='rb').close()
                        except:
                            raise errors.FilesystemError(
                                'Skybone daemon is unable to open %s in read mode -- check perms' % (
                                    path,
                                )
                            )

        resource_obj = Resource.from_dict(data)
        self.daemon.resource_mngr.cache.store_resource(resource_obj)
        # announce the resource only if it isn't already announced
        if (
            not self.daemon.resource_mngr.announcer.is_resource_announced(resource_obj.uid)
            and not self.daemon.resource_mngr.announcer.announce_or_fail(resource_obj.uid, timeout=300)
        ):
            self.log.warning('Unable to announce -- timeout')
            raise errors.CopierError('Tracker registration timed out (5min)')
        return resource_obj.uid

    def dl_helper(self, job, resid, dest, dest_inode):
        # Pid is used in netproxy.connect and moving to pid's net namespace if needed
        # Used only on linux
        if os.uname()[0].lower() == 'linux':
            pid = get_caller_pid(job)
        else:
            pid = None

        sock = None

        _resource_locked = None
        _resource_locker = None

        _np_job = None

        _paths_locked = None
        _paths_locker = None

        _dest_nocv = None  # path from dl without any conversion
        _dest_a = None     # path from dl
        _dest_b = None     # path in dom0

        resource_head = None

        hr = None

        def _cv_path(path):
            if _dest_a is None or _dest_b is None:
                raise Exception('check_dest need to be evaluated first')

            if _dest_a == _dest_b:
                return Path(path)

            return Path('/'.join((_dest_b.strpath, str(path)[len(_dest_a.strpath):])))

        try:
            while True:
                msg = job.feedback
                if msg[0] == 'set_sock':
                    sock = msg[1]
                    self.log.debug('Set sock %r', sock)
                    job.state(('sock_set', sock))
                    continue

                if msg[0] == 'get_props':
                    self.daemon.active.wait()
                    proxy_uds = (
                        ('\0' if self.daemon.resource_mngr._proxy_uds_abstract else '') +
                        self.daemon.resource_mngr._proxy_uds
                    )

                    self.log.debug('Sending props...')
                    job.state(('props', {
                        'uid': self.daemon.resource_mngr.announcer.uid,
                        'desc': self.daemon.hostname,
                        'ips': self.daemon.ips,
                        'trackers': self.daemon.trackers,
                        'main_port': self.daemon.resource_mngr.data_port,
                        'proxy_uds': proxy_uds,
                        'announcer_state': self.daemon.resource_mngr.announcer.get_state(),
                        'priorities': self.daemon.cfg.priorities,
                        'loose_path_checking': self.daemon.cfg.loose_path_checking,
                        'max_write_chunk': self.daemon.cfg.get('max_write_chunk', None),
                        'allowed_compression_codecs': self.daemon.cfg.get('allowed_compression_codecs', []),
                        'allowed_encryption_modes': self.daemon.cfg.get('allowed_encryption_modes', [encryption.PLAIN])
                    }))
                    continue

                if msg[0] == 'lock_resource':
                    assert sock is not None
                    get_head_if_ready = msg[1]

                    class HeadReady(Exception):
                        pass

                    self._trycopy_waiters[resid].add(pid)
                    _resource_locker = self.daemon.resource_mngr.locker.resources([resid])
                    _resource_locked = _resource_locker.__enter__()

                    def _wait_active_head(resid, grn):
                        self.log.debug('  bg waiter spawned')
                        while True:
                            ev = self.daemon.resource_mngr.active_heads.get(resid, None)
                            self.log.debug('  bg waiter wait ev %r', ev)

                            if ev:
                                head = ev.get()
                                self.log.debug("  bg waiter got head %r", head)

                                if head:
                                    grn.kill(HeadReady(head), False)
                                    return
                                else:
                                    gevent.sleep(1)
                            else:
                                gevent.sleep(5)

                    if get_head_if_ready:
                        get_head_waiter = gevent.spawn(_wait_active_head, resid, gevent.getcurrent())

                    try:
                        assert _resource_locked.next().next() == resid
                    except HeadReady as ex:
                        job.state(('resource_locked', resid, ex.args[0].dbdict()))
                        continue
                    finally:
                        if get_head_if_ready:
                            get_head_waiter.kill()

                    _np_job = self.daemon.netproxy_proc.cli.call('connect', resid, sock, None, pid)

                    try:
                        assert _np_job.next(), 'netproxy resid lock failed'
                        hr = self.daemon.resource_mngr.active_heads[resid] = gevent.event.AsyncResult()
                    except:
                        _np_job.send(True)
                        _np_job.wait()
                        raise

                    job.state(('resource_locked', resid, None))
                    continue

                if msg[0] == 'remove_trycopy_waiter':
                    self._trycopy_waiters[resid].remove(pid)
                    job.state(('trycopy_waiter_removed', ))
                    continue

                if msg[0] == 'get_file_alternatives':
                    self.daemon.active.wait()
                    assert resource_head is not None
                    assert isinstance(resource_head, dict)

                    self.log.info('Grabbing path alternatives...')

                    with self.daemon.db(transaction=False):
                        resource_obj = Resource.from_id(resid, self.daemon.resource_mngr.cache, job.log.logger)
                        if resource_obj and resource_obj.check(self.daemon.resource_mngr.cache):
                            # We have this resource on this machine and it completes
                            # checking. Grab all path alternatives.
                            alternatives = self.daemon.resource_mngr.cache.get_path_alternatives(
                                [
                                    item.data for item in resource_obj.items.itervalues()
                                    if isinstance(item, ResourceItem.file)
                                ]
                            )
                        else:
                            # Do not announce stop for current resource
                            self.daemon.db.query('DELETE FROM announce_stop WHERE hash = ?', (resid, ))

                            # If resource fails checking -- we should announce that
                            [sched.wakeup() for sched in self.daemon.resource_mngr.announcer.scheduler_loops]

                            # So, we dont have that resource on this machine, but
                            # have already received resource head. Try to search path
                            # alternatives by md5 checksum in this case (md5 checksum should
                            # match data_id if legacy_id is True)
                            alternatives = self.daemon.resource_mngr.cache.get_path_alternatives(
                                [
                                    item['md5sum'].encode('hex')
                                    for item in resource_head['structure'].itervalues()
                                    if (
                                        item['type'] == 'file' and
                                        item['resource']['type'] != 'symlink'
                                    )
                                ]
                            )

                        self._convert_alternative_paths(job, alternatives)
                        job.state(('alternatives', alternatives))
                    continue

                if msg[0] == 'lock_files':
                    self.daemon.active.wait()

                    self.log.info('Lock all files...')

                    assert resource_head is not None

                    if self.daemon.cfg.loose_path_checking:
                        # In loose path checking mode _dest_b may be None
                        # In that cose - lock files by original dl paths
                        if _dest_b is not None:
                            _base_path = _dest_b
                        else:
                            _base_path = _dest_nocv
                    else:
                        # If loose path checking is not turned on - we must have real dom0
                        # path (_dest_b) resolved properly for locks to work
                        assert _dest_b is not None
                        _base_path = _dest_b

                    path_types = dict([
                        (
                            _base_path.join(path),
                            path_opt['type'],
                        )
                        for path, path_opt in resource_head['structure'].iteritems()
                    ])

                    # Lock all files, but not directories
                    paths_to_lock = [p for p, typ in path_types.iteritems() if typ not in ('dir', )]

                    _paths_locker = self.daemon.resource_mngr.locker.paths(paths_to_lock)
                    _paths_locked = _paths_locker.__enter__()

                    locked_paths = 0

                    for paths in _paths_locked:
                        for path in paths:
                            locked_paths += 1

                    job.state(('paths_locked', ))
                    continue

                if msg[0] == 'check_announces':
                    assert _resource_locked
                    is_announced = self.daemon.resource_mngr.announcer.is_resource_announced(resid)
                    if is_announced:
                        self.log.debug('Found scheduled announces for resource')
                    job.state(('announces_checked', is_announced))
                    continue

                if msg[0] == 'check_dest':
                    dest, dest_inode, dest_dev = msg[1], msg[2], msg[3]
                    dest = _dest_nocv = Path(dest)

                    self.log.debug('Checking destination path %s', dest)

                    need_conv = False

                    dest_dir_ok = False
                    dest_ino_ok = False
                    dest_perm_ok = True

                    try:
                        if not dest.check(dir=1):
                            need_conv = True
                        else:
                            dest_dir_ok = True

                        if dest.stat().ino != dest_inode or dest.stat().dev != dest_dev:
                            need_conv = True
                        else:
                            dest_ino_ok = True
                    except Exception:
                        need_conv = True
                        dest_perm_ok = False

                    if need_conv:
                        dest_converted, convert_method = self._porto_lxc_convert_paths(job, [dest.strpath])
                        new_dest = dest_converted[dest.strpath]
                        if new_dest and new_dest != dest:
                            self.log.debug(
                                '  check dest: ok [dir %s, ino %s, perm %s], converted (%s => %s)',
                                dest_dir_ok, dest_ino_ok, dest_perm_ok, dest, new_dest
                            )
                            _dest_a = dest
                            _dest_b = dest = Path(new_dest)
                        else:
                            self.log.debug(
                                '  check dest: not ok [dir %s, ino %s, perm %s], unable to convert %s',
                                dest_dir_ok, dest_ino_ok, dest_perm_ok, dest
                            )
                    else:
                        _dest_a = _dest_b = dest
                        self.log.debug('  check dest: ok, no need to convert prefix')

                    _dest_a_strpath = _dest_a.strpath if _dest_a else None
                    _dest_b_strpath = _dest_b.strpath if _dest_b else None

                    job.state(('dest_checked', _dest_a_strpath, _dest_b_strpath))
                    continue

                if msg[0] == 'get_head':
                    # Get head from cache if we have it already
                    self.daemon.active.wait()
                    resid = msg[1]
                    head = Resource.head(
                        resid, self.daemon.db,
                        self.daemon.resource_mngr.cache,
                        self.daemon.resource_mngr.locker,
                        self.log
                    )
                    if head:
                        job.state(('head', head.dbdict()))
                    else:
                        job.state(('head', None))
                    continue

                if msg[0] == 'set_head':
                    assert hr is not None
                    hr.set(Resource.RbTorrent1(**msg[1]))
                    resource_head = msg[1]
                    job.state(('head_set', True))
                    continue

                if msg[0] == 'check_paths':
                    failed = []
                    paths = list(msg[1])

                    for path in paths:
                        path = Path(path)
                        if (_dest_a is None or _dest_b is None) and self.daemon.cfg.loose_path_checking:
                            converted_path = path
                        else:
                            converted_path = _cv_path(path)
                        try:
                            converted_path.open(mode='rb')
                        except Exception:
                            failed.append((path.strpath, converted_path.strpath))

                    job.state(('paths_checked', failed))
                    continue

                if msg[0] == 'store_resource':
                    data = msg[1]
                    self.cache_store_resource(job, data)
                    job.state(('resource_stored', ))
                    continue

                if msg[0] == 'update_metric':
                    name, value = msg[1], msg[2]
                    self.daemon.metric_pusher.update_counter(name, value)
                    continue

                if msg[0] == 'done':
                    assert msg[1]
                    break

                self.log.critical('Invalid message %r', msg)
                raise Exception('Invalid message %r' % (msg, ))
        finally:
            self._trycopy_waiters[resid].discard(pid)
            if not self._trycopy_waiters[resid]:
                del self._trycopy_waiters[resid]
            ei = sys.exc_info()
            if _resource_locked:
                _resource_locker.__exit__(ei[0], ei[1], ei[2])
            if _paths_locked:
                _paths_locker.__exit__(ei[0], ei[1], ei[2])
            if hr is not None:
                if not hr.ready():
                    hr.set(None)
                if self.daemon.resource_mngr.active_heads.get(resid, None) == hr:
                    self.daemon.resource_mngr.active_heads.pop(resid, None)
            if _np_job is not None:
                _np_job.send(True)
                _np_job.wait()

    def runjob(self, job, cmd, stdin, *args):
        daemon_bin_dir = sys.argv[0].rsplit(os.sep, 1)[0]
        cmd_bin_dir = cmd.rsplit(os.sep, 1)[0]

        if cmd_bin_dir != daemon_bin_dir:
            cmd_bin_dir = os.path.realpath(cmd_bin_dir)
            daemon_bin_dir = os.path.realpath(daemon_bin_dir)

        assert cmd_bin_dir == daemon_bin_dir, 'We can run only our own binaries!'

        cmd_name = cmd.rsplit(os.sep, 1)[1]

        assert cmd_name in ('skybone-sh', 'skybone-dl'), 'We can run only skybone-dl and skybone-sh scripts!'

        uid = get_caller_uid(job)

        self.log.info('Running %s  %s (has_root=%r, want_uid=%r)', cmd, ' '.join(args).strip(), has_root(), uid)

        def _preexec():
            if has_root():
                # Switch user privs only if we have root
                # In other case -- run process in the same perms as copier itself
                user_privileges(user=uid, limit=False).__enter__()

        proc = subproc.Popen(
            [cmd] + list(args),
            stdout=subproc.PIPE,
            stdin=subproc.PIPE,
            stderr=subproc.PIPE,
            preexec_fn=_preexec,
            close_fds=True
        )

        def _kill():
            self.log.info('Killing proc')

            reraise = None

            if not has_root():
                proc.kill()
                proc.wait()
                return

            while True:
                try:
                    forkpid = os.fork()
                except BaseException:
                    if not reraise:
                        reraise = sys.exc_info()
                    gevent.sleep(0.1)

                if not forkpid:
                    try:
                        user_privileges(user=0, limit=False).__enter__()
                        proc.kill()
                        proc.wait()
                    finally:
                        os._exit(0)
                else:
                    try:
                        while True:
                            if os.waitpid(forkpid, os.WNOHANG) == (0, 0):
                                gevent.sleep(0.1)
                                continue
                            break

                    except BaseException:
                        if not reraise:
                            reraise = sys.exc_info()
                        continue

                    break

        stdout_grn = None
        stderr_grn = None

        try:
            if stdin:
                for i in range((len(stdin) / 4096) + 1):
                    gevent.socket.wait_write(proc.stdin.fileno())
                    proc.stdin.write(stdin[i * 4096:(i + 1) * 4096])

            proc.stdin.close()

            def _fw_stream(stream_name, stream):
                while True:
                    gevent.socket.wait_read(stream.fileno())
                    data = stream.read(4096)
                    if not data:
                        return

                    if stream_name == 'stderr':
                        lines = data.rstrip().split('\n')
                        for line in lines:
                            self.log.warning('stderr: %s' % (line.rstrip(), ))

                    job.state(('stream', stream_name, data))

            stdout_grn = gevent.spawn(_fw_stream, 'stdout', proc.stdout)
            stderr_grn = gevent.spawn(_fw_stream, 'stderr', proc.stderr)

            self.log.debug('Stdout/stderr readers spawned')

            proc.wait()

            self.log.debug('process exited with code %r', proc.returncode)

            stdout_grn.join()
            stderr_grn.join()

            self.log.debug('stdout/stderr readers joined')

            return proc.returncode

        except BaseException as ex:
            import traceback

            self.log.critical('Unhandled exception: %s', ex)
            self.log.critical(traceback.format_exc())

            if proc.poll() is None:
                _kill()
            raise

        finally:
            self.log.info('Finished')
            if proc.poll() is None:
                _kill()

    def mon(self, job, subproc):
        self.daemon.active.wait()

        if not subproc and os.uname()[0].lower() == 'linux' and os.getresuid()[2] == 0:
            pid = get_caller_pid(job)
            forkpid = os.fork()
            if not forkpid:
                try:
                    user_privileges(user=0, store=False).__enter__()
                    my_cgroups = {}
                    for line in open('/proc/%d/cgroup' % (os.getpid(), ), 'rb'):
                        num, controller, cgroup = line.strip().split(':', 2)
                        if controller in ('freezer', 'devices', ''):
                            # SKYDEV-1467: Do not MOVE process from freezer/devices/cgroup2
                            continue
                        my_cgroups[controller] = cgroup

                    for controller, cgroup in my_cgroups.items():
                        tasks_file = '/sys/fs/cgroup/%s/%s/tasks' % (controller, cgroup)
                        if os.path.exists(tasks_file):
                            try:
                                open(tasks_file, 'ab').write(str(pid))
                            except:
                                pass
                finally:
                    os._exit(0)

            reraise = None
            while True:
                try:
                    if os.waitpid(forkpid, os.WNOHANG) == (0, 0):
                        gevent.sleep(0.1)
                        continue
                    break
                except BaseException:
                    if not reraise:
                        reraise = sys.exc_info()

            if reraise:
                raise reraise[0], reraise[1], reraise[2]

        assert job.feedback
        return True


class InternalRPC(object):
    def __init__(self, daemon):
        self.daemon = daemon
        self.log = daemon.log.getChild('rpc.netproxy')
        self.ext_log = logging.getLogger('ext')

    def logger(self, job):
        job.state('ready')

        while True:
            task = job.feedback

            if task is None:
                return

            name, levelno, message, args = task
            rlog = self.ext_log.getChild(name)
            getattr(rlog, {
                10: 'debug',
                20: 'info',
                30: 'warn',
                40: 'error',
                50: 'critical'
            }[levelno])(message, *args)


class NetproxyRPC(object):
    def __init__(self, daemon):
        self.daemon = daemon

    def lookup_infohash(self, job):
        while True:
            args, kwargs = job.feedback
            ret = self.daemon.resource_mngr.lookup_infohash(*args, **kwargs)
            job.state(ret)


class DiskIORPC(object):
    def __init__(self, daemon):
        self.daemon = daemon

    def get_paths_by_checksum(self, md5, limit=100):
        with self.daemon.db(debug_sql=False):
            return self.daemon.resource_mngr.cache.get_path_alternatives_one(md5, limit=limit)

    def set_bad_paths(self, md5, paths):
        with self.daemon.db(debug_sql=False):
            for fn, (mtime, chktime, new_mtime) in paths.iteritems():
                self.daemon.db.query('DELETE FROM file WHERE path = ? AND mtime = ?', [fn, mtime])


class SkybitRPC(object):
    def __init__(self, daemon):
        self.daemon = daemon

    def lookup_resource(self, job):
        job.state('ready')

        while True:
            args, kwargs = job.feedback
            ret = self.daemon.resource_mngr.lookup_resource(*args, **kwargs)
            job.state(ret)


class AdminRPC(object):
    def __init__(self, daemon):
        self.daemon = daemon
        self.log = daemon.log.getChild('rpc.admin')

    def status(self, dbwait=3):
        self.daemon.active.wait()

        with gevent.Timeout(dbwait) as tout:
            try:
                with self.daemon.db.deblock.lock('grab status'):
                    return self._status(db=True)
            except gevent.Timeout as ex:
                if ex != tout:
                    raise
                return self._status(db=False)

    def _status(self, db):
        dbcounters = {}
        if db:
            dbstatus = self.daemon.db.status()
            dbstatus['locked'] = False

            dbcounters = {
                'resource_cnt': self.daemon.db.query_one_col('SELECT COUNT(*) FROM resource'),
                'data_cnt': self.daemon.db.query_one_col('SELECT COUNT(*) FROM data'),
                'data_size': self.daemon.db.query_one_col('SELECT SUM(size) FROM data'),
                'file_cnt': self.daemon.db.query_one_col('SELECT COUNT(*) FROM file'),
                'wait_checking': self.daemon.db.query_one_col(
                    'SELECT COUNT(*) FROM file WHERE chktime < ?',
                    [time.time() - 3600 * 3]
                ),
            }
            if not dbcounters['data_size']:
                dbcounters['data_size'] = 0
        else:
            dbstatus = {}
            dbstatus['locked'] = True
            dbstatus['lock_reason'] = self.daemon.db.deblock.lock_reason
            dbstatus['job'] = self.daemon.db.deblock.job
            dbstatus['job_waiters'] = self.daemon.db.deblock.lock_waiters

            if dbstatus['job']:
                dbstatus['job'] = list(dbstatus['job'])
                meth, args, kwargs, local = dbstatus['job']

                dbstatus['job'] = (meth.__func__.__name__, repr(args)[:100], repr(kwargs)[:100])
            else:
                dbstatus['job'] = (None, None, None)

            dbcounters = {
                'resource_cnt': None,
                'data_cnt': None,
                'data_size': None,
                'file_cnt': None,
                'wait_checking': None
            }

        rusage_self = resource.getrusage(resource.RUSAGE_SELF)
        rusage_children = resource.getrusage(resource.RUSAGE_CHILDREN)

        greenlets_count = 0
        for obj in gc.get_objects():
            if isinstance(obj, greenlet.greenlet):
                greenlets_count += 1

        if hasattr(self.daemon, 'resource_mngr'):
            resource_mngr_stats = self.daemon.resource_mngr.stats.copy()
            resource_mngr_stats['gc'] = self.daemon.resource_mngr.filechecker.stats.copy()
            announcer_stats = self.daemon.resource_mngr.announcer.status(use_db=db)
        else:
            resource_mngr_stats = {}
            announcer_stats = {}

        downloaders_stats = resource_mngr_stats.pop('downloaders', {})
        downloaders_skybit_stats = downloaders_stats.get('skybit', {})

        try:
            netproxy_stats = self.daemon.netproxy_proc.cli.call('stats').wait(timeout=3)
        except:
            netproxy_stats = {}

        skybit_proc_stats = self.daemon.skybit_proc.cli.call('stats').wait()

        return {
            'daemon': {
                'active': self.daemon.active.isSet(),
                'uptime': monoTime() - self.daemon.tsStarted,
                'stopping': self.daemon.shouldStop,
                'rusage': {
                    'self': {
                        'user': rusage_self.ru_utime,
                        'system': rusage_self.ru_stime,
                    },
                    'children': {
                        'user': rusage_children.ru_utime,
                        'system': rusage_children.ru_stime,
                    }
                },
                'greenlets': greenlets_count
            },
            'rpc': self.daemon.rpc.stats,
            'db': {
                'sqlite': dbstatus,
            },
            'resource_mngr': resource_mngr_stats,
            'file_cache': {
                'resources_count': dbcounters['resource_cnt'],
                'data_count': dbcounters['data_cnt'],
                'data_size': dbcounters['data_size'],
                'files_count': dbcounters['file_cnt'],
                'wait_checking': dbcounters['wait_checking'],
            },
            'skybit': {
                'stats': {
                    'ok': downloaders_skybit_stats.get('ok', None),
                    'error': downloaders_skybit_stats.get('error', None),
                },
                'seeder': skybit_proc_stats,
            },
            'announcer': announcer_stats,
            'proxy': netproxy_stats,
        }

    def counters(self, query):
        if not self.daemon.active.wait(timeout=30):
            raise Exception('Copier is not ready (we wait 30s), try again later')

        result = {}

        for block, counters in query:
            blockresult = result[block] = {}

            if block == 'rusage':
                rusage_self = resource.getrusage(resource.RUSAGE_SELF)
                rusage_chld = resource.getrusage(resource.RUSAGE_CHILDREN)

            elif block == 'proxy':
                netproxy_stats = self.daemon.netproxy_proc.cli.call('stats').wait()

            elif block == 'file_cache':
                filecache_stats = self.daemon.data_counters

            for counter in counters:
                if block == 'file_cache':
                    # Report filecache stats only if we counted them at least once.
                    if filecache_stats.get('ready', False):
                        if counter == 'resource_count':
                            blockresult[counter] = filecache_stats['resource_cnt']
                        elif counter == 'file_count':
                            blockresult[counter] = filecache_stats['file_cnt']
                        elif counter == 'data_size':
                            blockresult[counter] = filecache_stats['data_size']
                        elif counter == 'file_size':
                            blockresult[counter] = filecache_stats['file_size']
                elif block == 'rusage':
                    if counter == 'uptime':
                        blockresult[counter] = int(monoTime() - self.daemon.tsStarted)
                    elif counter == 'self_system':
                        blockresult[counter] = rusage_self.ru_stime
                    elif counter == 'self_user':
                        blockresult[counter] = rusage_self.ru_utime
                    elif counter == 'chld_system':
                        blockresult[counter] = rusage_chld.ru_stime
                    elif counter == 'chld_user':
                        blockresult[counter] = rusage_chld.ru_utime

                elif block == 'proxy':
                    if counter == 'bytes_in':
                        blockresult[counter] = netproxy_stats['bytes_in']
                    elif counter == 'bytes_ou':
                        blockresult[counter] = netproxy_stats['bytes_out']
                    elif counter == 'bytes_in_skbt':
                        blockresult[counter] = netproxy_stats['bytes_in_skybit']
                    elif counter == 'bytes_ou_skbt':
                        blockresult[counter] = netproxy_stats['bytes_out_skybit']
                    elif counter == 'bytes_in_torrent':
                        blockresult[counter] = netproxy_stats['bytes_in_torrent']
                    elif counter == 'bytes_ou_torrent':
                        blockresult[counter] = netproxy_stats['bytes_out_torrent']
                    elif counter == 'connects_in':
                        blockresult[counter] = netproxy_stats['incoming']['connects']
                    elif counter == 'connects_in_torrent':
                        blockresult[counter] = netproxy_stats['incoming']['torrent']['connects']
                    elif counter == 'connects_in_skbt':
                        blockresult[counter] = netproxy_stats['incoming']['skybit']['connects']
                    elif counter == 'connects_ou':
                        blockresult[counter] = (
                            netproxy_stats['proxy']['cmd_connect_cnt'] +
                            netproxy_stats['uds']['connects']
                        )
                    elif counter == 'connects_ou_torrent':
                        blockresult[counter] = netproxy_stats['proxy']['cmd_connect_cnt']
                    elif counter == 'connects_ou_skbt':
                        blockresult[counter] = netproxy_stats['uds']['connects']

        return result

    def evaluate(self, job, code):
        disallow_other_containers(self.daemon.porto, job)

        user_id = get_caller_uid(job)

        if user_id == 0:
            ret = None
            exec code
            return ret

        pipe = os.pipe()
        pid = os.fork()
        try:
            if not pid:
                try:
                    try:
                        ret = None
                        user_privileges(user=os.geteuid(), store=False).__enter__()
                        exec code
                        ret = msgpack.dumps((True, ret))
                    except BaseException as ex:
                        ret = msgpack.dumps((False, ex.__class__.__name__, str(ex)))
                    finally:
                        os.write(pipe[1], ret)
                        os._exit(0)
                except BaseException:
                    os._exit(0)
            else:
                while True:
                    if os.waitpid(pid, os.WNOHANG) == (0, 0):
                        gevent.sleep(0.1)
                        continue
                    break
                gevent.socket.wait_read(pipe[0])
                data = os.read(pipe[0], 4096)
                data = msgpack.loads(data)
                if data[0]:
                    return data[1]
                else:
                    raise Exception('%s: %s' % (data[1], data[2]))
        finally:
            os.close(pipe[0])
            os.close(pipe[1])

    def check_paths(self, paths):
        self.daemon.active.wait()
        if isinstance(paths, (list, tuple)):
            paths = set(paths)
        else:
            assert isinstance(paths, set)

        self.log.debug('check_paths(%d items)', len(paths))

        return list(paths.difference(
            self.daemon.db.query_col(
                'SELECT path FROM file WHERE path IN (??)',
                [list(paths)],
                log=False
            )
        ))

    def check_resources(self, resources):
        self.daemon.active.wait()
        if isinstance(resources, (list, tuple)):
            resources = set(resources)
        else:
            assert isinstance(resources, set)

        self.log.debug('check_resources(%d items)', len(resources))

        return list(resources.difference(
            self.daemon.db.query_col(
                'SELECT id FROM resource WHERE id IN (??)',
                [list(resources)],
                log=False
            )
        ))

    def notify(self, paths):
        with self.daemon.db(transaction=False):
            ts = time.time()
            self.daemon.db.execute_many(
                'UPDATE file SET chktime = 0 WHERE path GLOB ?',
                [(path + '*',) for path in paths]
            )

        te = time.time()
        self.log.debug('Set chktime=0 of %d path(s) in %0.4fs', len(paths), te - ts)

        return True

    def query(self, job, sql):
        self.daemon.active.wait()

        disallow_other_containers(self.daemon.porto, job, allow_only_root=True)

        return self.daemon.db.query(sql, log=False)

    def dbbackup(self):
        self.daemon.active.wait()
        with self.daemon.db.deblock.lock('database backup'):
            self.daemon.db.backup()

    def dbcheck(self, full_resource_check=False, fix=False):
        self.daemon.active.wait()
        with self.daemon.db, self.daemon.db.deblock.lock('database consistency check'):
            stale_data_no_files = self.daemon.db.query_col(
                'SELECT d.id FROM data d '
                'LEFT JOIN file f ON f.data = d.id WHERE f.data IS NULL'
            )
            stale_data_no_resource = self.daemon.db.query_col(
                'SELECT d.id FROM data d '
                'LEFT JOIN resource_data rd ON rd.data = d.id WHERE rd.data IS NULL'
            )

            bad_data = set(stale_data_no_files)
            self.log.debug('%d data have no files pointing', len(stale_data_no_files))
            bad_data.update(stale_data_no_resource)
            self.log.debug('%d data is not in any resource', len(stale_data_no_resource))

            if not full_resource_check:
                invalid_resources = self.daemon.db.query_col(
                    'SELECT derivied.resource FROM ( '
                    '    SELECT '
                    '        rd.resource, '
                    '        r.torrents_count - 1 as tcnt, '
                    '        (SELECT COUNT(DISTINCT data) '
                    '            FROM resource_data '
                    '            JOIN data d ON data = d.id '
                    '            WHERE d.size > 0 AND resource = rd.resource '
                    '        ) as cnt '
                    '    FROM resource_data rd '
                    '    JOIN resource r ON r.id = rd.resource '
                    '    WHERE '
                    '        rd.data IS NOT NULL '
                    '    GROUP BY rd.resource '
                    ') as derivied WHERE tcnt != cnt'
                )
                self.log.debug('%d resources are incomplete', len(invalid_resources))
            else:
                assert 0, 'not supported'

            if stale_data_no_files:
                incomplete_resources = self.daemon.db.query_col(
                    'SELECT DISTINCT resource FROM resource_data '
                    'WHERE data IS NOT NULL AND data IN (??)',
                    [stale_data_no_files]
                )
                self.log.debug(
                    '%d resources will be dropped because data without files will be dropped',
                    len(incomplete_resources)
                )
            else:
                incomplete_resources = []

            bad_resources = set(invalid_resources)
            bad_resources.update(incomplete_resources)

            if invalid_resources or incomplete_resources:
                extra_data_drop = self.daemon.db.query_col(
                    'SELECT DISTINCT data FROM resource_data WHERE data NOT IN ( '
                    '    SELECT data FROM resource_data WHERE resource NOT IN (??) '
                    ')',
                    [list(bad_resources)]
                )
                self.log.debug('%d data will be dropped because resources will be dropped', len(extra_data_drop))
            else:
                extra_data_drop = []

            bad_data.update(extra_data_drop)

            if stale_data_no_resource or extra_data_drop:
                extra_file_drop = self.daemon.db.query_col(
                    'SELECT DISTINCT path FROM file WHERE data IN (??)',
                    [list(bad_data)]
                )
                self.log.debug('%d files will be dropped because their data will be dropped', len(extra_file_drop))
            else:
                extra_file_drop = []

            ret = {
                'bad_files': len(extra_file_drop),
                'stale_datas': len(stale_data_no_files) + len(stale_data_no_resource) + len(extra_data_drop),
                'bad_resources': len(invalid_resources) + len(incomplete_resources)
            }

            if fix:
                self.resource_mngr.cache.delete_paths(extra_file_drop)

                [sched.wakeup() for sched in self.resource_mngr.announcer.scheduler_loops]

            return ret

    def file_move(self, job, pairs, after=False, quiet=False, lock=True):
        self.daemon.active.wait()

        if not after:
            disallow_other_containers(self.daemon.porto, job)

        results = {}
        for src, tgt in pairs:
            key = src
            src = Path(src).realpath()
            tgt = Path(tgt).realpath()

            if not after:
                if not src.check(exists=1):
                    results[key] = (False, '"%s" not exists' % (src, ))
                    continue

                if not src.check(file=1):
                    results[key] = (False, '"%s" is not a file' % (src, ))
                    continue

            if not tgt.dirpath().check(exists=1):
                results[key] = (False, 'target directory "%s" not exists' % (tgt.dirpath(), ))
                continue

            if tgt.dirpath().check(file=1):
                results[key] = (False, 'target directory "%s" is not a directory' % (tgt.dirpath(), ))
                continue

            if src.check(file=1) or (after and tgt.check(file=1)):
                if lock:
                    locker = self.daemon.resource_mngr.locker.paths
                else:
                    import contextlib

                    @contextlib.contextmanager
                    def _nolock(paths):
                        yield ((path for path in paths) for i in range(1))

                    locker = _nolock

                with locker([src.strpath]) as bulk_locker:
                    assert bulk_locker.next().next() == src.strpath

                    with locker([tgt.strpath]) as bulk_locker2:
                        assert bulk_locker2.next().next() == tgt.strpath

                        with self.daemon.db:
                            data = self.daemon.db.query_one_col(
                                'SELECT data FROM file WHERE path = ?', [src.strpath],
                                log=False
                            )
                            if data is None:
                                results[key] = (False, 'file "%s" is not shared by copier' % (src, ))
                                continue

                            if not after:
                                src.move(tgt)

                            self.daemon.db.query(
                                'DELETE FROM file WHERE path = ?', [tgt.strpath], log=False
                            )
                            self.daemon.db.query(
                                'UPDATE file SET path = ? WHERE path = ?',
                                [tgt.strpath, src.strpath]
                            )

                results[key] = (True, 'Moved "%s" => "%s"' % (src, tgt))

            elif after and tgt.check(dir=1):
                assert after, 'Moving directories allowed only with --after flag'

                targets = []
                for path in tgt.visit():
                    if path.check(file=1) and path.check(link=0):
                        targets.append(path.strpath)

                targets = sorted(targets)

                sources = []

                for path in targets:
                    src_path = src.strpath + path[len(tgt.strpath):]
                    sources.append(src_path)

                all_files = list(sources) + list(targets)

                not_locked = set(all_files)

                if all_files:
                    with self.daemon.resource_mngr.locker.paths(all_files) as bulk_locker:
                        while not_locked:
                            for locked in bulk_locker:
                                for locked_path in locked:
                                    assert locked_path in not_locked
                                    not_locked.discard(locked_path)

                            self.log.info(
                                'Locked %d source and target files out from %d total',
                                len(all_files) - len(not_locked),
                                len(all_files)
                            )

                        for idx, source in enumerate(sources):
                            source = source
                            target = targets[idx]

                            res = self.file_move(job, [(source, target)], after=True, lock=False)

                            failed = False

                            for k, v in res.items():
                                if not v[0]:
                                    failed = True
                                    results[k] = v
                                    break
                                else:
                                    if not quiet:
                                        results[k] = v

                            if failed:
                                break

        return results

    # SKYDEV-2322: this command is removed from skybone-ctl, can only be run via rpc
    def clean_resources(self, wait):
        self.log.warning('CLEAN EVERYTHING')

        with self.daemon.db(transaction=False):
            self.daemon.db.query('DELETE FROM resource')
            self.daemon.db.query('DELETE FROM resource_data')
            self.daemon.db.query('DELETE FROM announce')
            self.daemon.db.query('DELETE FROM announce_stop')
            self.daemon.db.query('DELETE FROM file')
            self.daemon.db.query('DELETE FROM data')

            self.daemon.db.query('PRAGMA FOREIGN_KEYS = NO')
            self.daemon.db.query(
                'INSERT INTO announce_stop '
                'SELECT ?, at.id, ? '
                'FROM announce_tracker at',
                ('CLEAN', time.time() + 86400 * 2)
            )
            self.daemon.db.query('PRAGMA FOREIGN_KEYS = YES')

        [sched.wakeup() for sched in self.daemon.resource_mngr.announcer.scheduler_loops]

        self.log.info('Waiting tracker clean responses (%d secs)', wait)

        deadline = time.time() + wait
        while deadline > time.time():
            left = self.daemon.db.query_one_col(
                'SELECT COUNT(*) FROM announce_stop WHERE hash = "CLEAN"', log=False
            )

            if left == 0:
                break
            else:
                self.log.debug('  still %d clean requests left...', left)

            gevent.sleep(0.5)
        else:
            self.log.info('  failed to wait for all clean responses')
            return 2

        self.log.info('  all trackers cleared us completely!')
        return 0
