from collections import defaultdict
import itertools
import random

HTTP_RANGES = '1'

CONNECT_TIMEOUT = 10
READ_TIMEOUT = 10


def build_data_idx(data):
    data_idx = defaultdict(list)
    for idx_offset, data in data:
        for idx, piece in enumerate(data.pieces):
            end_boundary = piece.length * (idx + 1)
            if end_boundary < data.size:
                piecelen = piece.length
            else:
                piecelen = data.size - (piece.length * idx)
                assert piecelen <= piece.length

            data_idx[data.md5hash].append((idx_offset + idx, piecelen))

    # Convert data_idx to list, so we can shuffle it
    data_idx = list(data_idx.items())
    random.shuffle(data_idx)

    # Little integrity checking
    for md5, idxs in data_idx:
        assert min(idx for idx, piece in idxs) == idxs[0][0]

    return data_idx

def convert_links(links, data_idx, log):
    for md5, md5_links in links.iteritems():
        # Three variants:
        # {md5: [link1, link2, ...]}  # v1
        # {md5: {link1: linkopts, link2: linkopts, ...} # v1 extended
        # {md5: [(start, size, (link1, link2, ...))]}  # v2 (yt chunked files)
        # {md5: [(start, size, {link1: linkopts, link2: linkopts, ...})]}  # v2 extended

        if md5_links and isinstance(md5_links, dict):
            log.info('[link format v1ext]  md5: %s, links: %s', md5, md5_links)
            links[md5] = [(0, None, md5_links)]
        elif md5_links and isinstance(md5_links[0], (list, tuple)):
            for linksinfo in md5_links:
                for partlink in linksinfo[2]:
                    log.info(
                        '[link format v2/v2ext]  md5: %s, start %d, size %d, link: %s',
                        md5, linksinfo[0], linksinfo[1], partlink
                    )
        else:
            log.info('[link format v1]  md5: %s, link: %s', md5, md5_links[0])
            links[md5] = [(0, None, md5_links)]

    # For oldstyle non-chunked http links, convert them to one big chunk
    # Also check we have each md5 chunk we need
    for md5, idxs in data_idx:
        if md5 not in links:
            raise RuntimeError('Unable to find http links for md5: %s, abort' % md5)

        for i, (start, length, md5_links) in enumerate(links[md5]):
            if length is None:
                assert i == 0
                links[md5][i] = (
                    0, sum(piecelen for _, piecelen in idxs),
                    md5_links
                )

def assign_pieces_to_chunks(chunk_links, idxs):
    # Assign indexes and piecelengths for each http chunk
    # (chunk_index, start, length, [(idx, piecelen), (idx2, piecelen2)], [links])

    # (start, length, [links])

    # Lengths of all http chunks should match sum of lengths of all skybit blocks in this chunk
    total_chunk_len = sum(size for _, size, _ in chunk_links)
    total_piece_len = sum(piecelen for _, piecelen in idxs)
    assert (total_chunk_len == total_piece_len), (
        'Sum of bytes in all http chunks (%d) do not match sum of all '
        'skybit blocks in this chunk (%d)' % (
            total_chunk_len, total_piece_len
        )
    )

    assigned_chunk_links = []

    for http_chunk_idx, (http_chunk_start, http_chunk_length, http_chunk_links) in enumerate(
        sorted(chunk_links)
    ):
        chunk_indexes = []  # [(idx, piecelen), (idx2, piecelen2)]

        current_start_byte = 0
        current_chunk_start = http_chunk_start
        current_chunk_length = http_chunk_length

        for idx, piecelen in idxs:
            if current_start_byte == current_chunk_start:
                chunk_indexes.append((idx, piecelen))
                current_chunk_start += piecelen
                current_chunk_length -= piecelen

                assert current_chunk_length >= 0
                if current_chunk_length == 0:
                    # We found all indexes for this http chunk
                    break
            else:
                # Piece do not belong to this chunk, just skip
                pass

            current_start_byte += piecelen

        assert sum([piecelen for _, piecelen in chunk_indexes]) == http_chunk_length

        assigned_chunk_links.append((
            http_chunk_idx, http_chunk_start, http_chunk_length, chunk_indexes, http_chunk_links
        ))

    random.shuffle(assigned_chunk_links)
    return assigned_chunk_links

def get_range_for_request(pieces_needed, idxs, chunk_length):
    """ Returns two inclusive ranges: bytes and piece indexes """

    piece_is_not_needed = lambda info: info[0] not in pieces_needed

    start_idx = sum(
        1 for _ in
        itertools.takewhile(piece_is_not_needed, idxs)
    )

    if start_idx == len(idxs):
        return None, None

    end_idx = len(idxs) - sum(
        1 for _ in 
        itertools.takewhile(piece_is_not_needed, reversed(idxs))
    )

    idx_offset = idxs[0][0]

    start_bytes = sum(piecelen for _, piecelen in idxs[:start_idx])
    end_bytes = chunk_length - sum(piecelen for _, piecelen in idxs[end_idx:])

    return (
        (start_bytes, end_bytes - 1),
        (idx_offset + start_idx, idx_offset + end_idx - 1)
    )

def _parse_http_range(http_range):
    prefix = 'bytes='
    if not http_range.startswith(prefix):
        raise ValueError('Invalid range specification: %s' % http_range)
    try:
        start, end = http_range[len(prefix):].split('-')
        start, end = int(start), int(end)
    except ValueError:
        raise ValueError('Invalid range specification: %s' % http_range)

    return start, end

def get_http_range_opt(linkopts):
    if linkopts is not None and HTTP_RANGES in linkopts:
        return _parse_http_range(linkopts[HTTP_RANGES])
    else:
        return None

def adjust_range(content_range, data_range):
    adjusted = data_range[0] + content_range[0], data_range[0] + content_range[1]
    assert adjusted[1] <= data_range[1], 'content range is wider than data range'
    return adjusted
