from __future__ import print_function, division

from .. import monkey_patch as skbn_monkey_patch

from .locality import LocalityPeerQueue

skbn_monkey_patch()

import argparse
import contextlib
import errno
import hashlib
import logging
import msgpack
import os
import random
import struct
import subprocess as subproc
import sys
import time
import urlparse
import yaml

try:
    import xxhash
except ImportError:
    pass

import gevent
import py

from gevent import socket

import api.copier.errors

try:
    from setproctitle import setproctitle
except ImportError:
    setproctitle = None

from ..resource.resource import Resource
from ..resource.hasher import Hasher
from ..hasher import PIECE_SIZE
from .world import World
from .shmem import SharedMemory
from ..diskio.io import IO
from ..diskio.iosem import IoSemaphore
from ..utils import Path, human_size, human_speed
from ..component import Component, ComponentLoggerAdapter
from ..resource.file import ResourceItem
from ..resource.announcer import Announcer, AnnounceResponseLeeching, ANNOUNCE_DOWNLOADING, ANNOUNCE_STOPPED
from ..logger import SmartLoggerAdapter
from ..metrics import Histogram, IO_HGRAM_INTERVALS, DL_HGRAM_INTERVALS
from ..fallocate import fallocate

from ...rpc.client import RPCClientGevent
from ...rpc.errors import ProtocolError, RPCError, Timeout as RPCTimeout, HandshakeTimeout

from . import encryption
from .cmd_utils import setup_logger, await_parallel_downloads, dl_helper, master_mon, MasterDisconnect
from .compression import parse_compression_mode
from .dc import DC, determine_dc
from .logger import ResourceIdLoggerAdapter


MSG_PACKER = msgpack.Packer()
MSS_MAGIC_BYTE = '\x1231835478432'

TRACKER_ANNOUNCE_INTERVAL = 120
TRACKER_ANNOUNCE_TIMEOUT = TRACKER_ANNOUNCE_INTERVAL
RESOURCE_HEAD_DL_TIMEOUT = 60
ZERO_DL_TIMEOUT = 300

# Those require randomization, because we will schedule connect to seeder after
# timeouts
HAPPYSEED_HEAD_DL_TIMEOUT = random.randint(20, 40)
HAPPYSEED_DL_TIMEOUT = random.randint(60, 120)

# Don't download local available piece via non-local connection with that probability
LOCAL_PROB = 0.8

assert HAPPYSEED_HEAD_DL_TIMEOUT < RESOURCE_HEAD_DL_TIMEOUT
assert HAPPYSEED_DL_TIMEOUT < ZERO_DL_TIMEOUT

# Force seeders connect in random interval
# This is "dumb" attempt to fix SKYDEV-1742.

SEEDERS_DISALLOW_SECONDS = random.randint(600, 1200)  # 10..20 mins

HTTP_ONLY_ANNOUNCE_RETRY_INTERVAL = 5


class IpLoggerAdapter(SmartLoggerAdapter):
    def process(self, msg, kwargs):
        return '[%s : %d seeder=%d dfs=%d]  %s' % (
            self.extra['ip'], self.extra['port'],
            self.extra['seeder'], self.extra['dfs'],
            msg
        ), kwargs


class MetricUpdater(Component):
    """
    Context manager that updates download counters on exit.
    """
    def __init__(self, parent, dl_helper, max_dl_speed):
        super(MetricUpdater, self).__init__(parent=parent, logname='metrics')

        self._dl_helper = dl_helper
        self._dfs_mode = None
        self._success_signal = None
        self._failure_signal = None
        if max_dl_speed >= 0:
            self._recv_speed_signal = 'dl_limited_crossdc_recv_speed_mb'
        else:
            self._recv_speed_signal = 'dl_nolimit_crossdc_recv_speed_mb'

        self._handle = None

        self._io_write_hgram = Histogram(IO_HGRAM_INTERVALS)
        self._io_read_hgram = Histogram(IO_HGRAM_INTERVALS)
        self._recv_speed_hgram = Histogram(DL_HGRAM_INTERVALS)

        self.set_dfs_mode(False)

        self.add_loop(self._push_to_daemon)

    def set_dfs_mode(self, mode=True):
        self._dfs_mode = mode
        if mode:
            self._success_signal = 'dfs_downloads_succeeded'
            self._failure_signal = 'dfs_downloads_failed'
        else:
            self._success_signal = 'p2p_downloads_succeeded'
            self._failure_signal = 'p2p_downloads_failed'

    def set_handle(self, handle):
        self._handle = handle

    def update_io_write_metric(self, value):
        self._io_write_hgram.update(value)

    def update_io_read_metric(self, value):
        self._io_read_hgram.update(value)

    def update_recv_speed_hgram(self, value):
        self._recv_speed_hgram.update(value)

    def _push_to_daemon(self):
        self.log.debug('Pushing metrics to daemon')
        if self._handle:
            self.update_recv_speed_hgram(self._handle.swarm.extract_recv_speed_hgram())

        for signal, hgram in (
            (self._recv_speed_signal, self._recv_speed_hgram),
            ('dl_io_write_ms', self._io_write_hgram),
            ('dl_io_read_ms', self._io_read_hgram),
        ):
            if not hgram:
                continue
            try:
                self._dl_helper.update_metric(signal, hgram.get_value())
                hgram.reset()
            except Exception:
                self.log.exception('Failed to update %s', signal)

        return 5

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_value, traceback):
        self.stop()
        self._push_to_daemon()

        try:
            if self._handle:
                self._dl_helper.update_metric('local_bytes', self._handle.dc_stats.local)
                self._dl_helper.update_metric('total_bytes', self._handle.dc_stats.total)

            no_peer_exc_types = (
                api.copier.errors.ResourceNotAvailable,
                api.copier.errors.ResourceNotAllowedByNetwork,
            )
            if exc_type is None:
                self._dl_helper.update_metric(self._success_signal, 1)
            elif issubclass(exc_type, no_peer_exc_types):
                self._dl_helper.update_metric('downloads_failed_no_peer', 1)
            elif (
                issubclass(exc_type, api.copier.errors.FilesystemError)
                and not exc_value.message.startswith('Unable to read data')
                and not exc_value.message.startswith('Unable to write data')
            ):
                # 'Unable to read/write data' may be caused by an error in copier.
                # Don't count it in fs errors caused by users.
                self._dl_helper.update_metric('downloads_failed_fs_error', 1)
            else:
                self._dl_helper.update_metric(self._failure_signal, 1)
        except Exception:
            self.log.exception('failed to update dl counters')


class AnnouncerLocalityPeerQueue(LocalityPeerQueue):

    def append(self, item):
        if item[0].get('is_local', False):
            self._loc.append(item)
        else:
            self._nloc.append(item)


class PreSortQueue(LocalityPeerQueue):

    def append(self, item):
        if item.get('is_local', False):
            self._loc.append(item)
        else:
            self._nloc.append(item)


class AnnouncerQueue(gevent.queue.Queue):

    def _create_queue(self, items=()):
        return AnnouncerLocalityPeerQueue()

    def _init(self, maxsize):
        self.queue = self._create_queue()

    def _put(self, item):
        self.queue.append(item)

    def _get(self):
        return self.queue.next()


class MockAnnouncer(object):

    class Announce(object):
        EV_FAIL = Announcer.Announce.EV_FAIL
        EV_OK = Announcer.Announce.EV_OK

        def __init__(self, response, parent):
            self._response = response
            self._parent = parent

        def wait(self):
            yield (self.EV_OK, self._parent._skybone_coord_client, self._response)

    class SkyboneCoordClient(object):
        def __init__(self, log):
            self.log = log

    class TrackerClient(object):
        def __init__(self, skybone_coord_client):
            self.clients = {'': (skybone_coord_client, [])}

    def __init__(self, peers, log):
        self._announce_response = AnnounceResponseLeeching(
            interval=86400,
            seeders=len(peers), leechers=0,
            peers=peers
        )

        self._skybone_coord_client = self.SkyboneCoordClient(log)
        self.client = self.TrackerClient(self._skybone_coord_client)

    @contextlib.contextmanager
    def announce(self, infohash, action, network, timeout=None, clients=None):
        yield self.Announce(self._announce_response, self)


class AnnounceManager(Component):  # {{{
    EV_COMPLETED = 1
    EV_FAIL = 2

    def __init__(self, parent=None):
        super(AnnounceManager, self).__init__(logname='anmgr', parent=parent)

        self.infohash = None
        self.network = None
        self.ann = None

        # client => [seeders, leechers, [peers], last_receive_ts, num_completed]
        self.announces = {}
        # [0, 0, [], 0, 0]

        # wait if all tracker requests will complete
        # unable to reach trackers
        # got 0 servers having resource

        self.added_peers = set()
        self.peers = gevent.queue.Queue()
        self.should_stop = False
        self.stop_ev = gevent.event.Event()

        self.seeders = 0   # total seeders
        self.leechers = 0  # total leechers

        self.peers_seeding = set()
        self.peers_seeding_nodfs = set()
        self.peers_leeching = set()
        self.peers_dfs = set()

        self.http_only_mode = False

    def set_announcer(self, ann, http_only_mode=False):
        self.ann = ann
        self.http_only_mode = http_only_mode
        if self.http_only_mode:
            self.log.warning('HTTP-only mode is enabled')

        for client, _ in self.ann.client.clients.values():
            self.announces[client] = {
                'seeders': 0,
                'leechers': 0,
                'peers_seeding': set(),
                'peers_leeching': set(),
                'peers_dfs': set(),
                'last_receive_ts': 0,
                'num_completed': 0,
            }

    def set_peers(self, local_prob):
        if local_prob > 0:
            self.peers = AnnouncerQueue()

    def announce(self, infohash, network):
        self.infohash = infohash
        self.network = network
        self.announce_loop.wakeup()

    def stop(self):
        self.should_stop = True
        self.announce_loop.wakeup()
        return self

    def join(self):
        self.stop_ev.wait(timeout=3)

    def has_responses(self):
        return any((info['num_completed'] for info in self.announces.values()))

    def all_done(self):
        return all((info['num_completed'] for info in self.announces.values()))

    def has_peers(self):
        return any((info['seeders'] for info in self.announces.values()))

    def _log_peers(self, log, peers, msg_prefix):
        if not peers:
            return

        peers_by_dc = {dc: [] for dc in DC}
        for ip, port in peers:
            peers_by_dc[determine_dc(ip, self.log)].append(ip)
        log.debug('%s: %s',
            msg_prefix,
            ', '.join(
                '{} in {} {}'.format(len(peers_by_dc[dc]), dc, peers_by_dc[dc])
                for dc in sorted(peers_by_dc.keys(), key=lambda x: x.value)
                if peers_by_dc[dc]
            )
        )

    def sort_peers_by_loc(self, peers):
        q = PreSortQueue()
        for p in peers:
            peerip = p['ips'][0]
            p['is_local'] = determine_dc(peerip) == self.parent.parent.world.current_dc
            q.append(p)
        return [q.next() for _ in peers]

    @Component.green_loop(logname='announcer')
    def announce_loop(self):
        if not self.infohash:
            return 10

        if not self.should_stop:
            self.added_peers.clear()

            with self.ann.announce(
                self.infohash, ANNOUNCE_DOWNLOADING, self.network,
                timeout=TRACKER_ANNOUNCE_TIMEOUT
            ) as ann:
                self.log.debug('peers is %r', self.peers)
                got_dfs_peers = False
                for ev, client, data in ann.wait():
                    if not data:
                        self.log.debug('Tracker %r timed out', client)
                    else:
                        leechers = data.leechers
                        seeders = data.seeders
                        peers = data.peers

                        peers_seeding, peers_leeching, peers_dfs = set(), set(), set()
                        for peer in peers:
                            if peer.get('dfs', False):
                                peers_dfs.add(peer['ips'])
                            elif peer.get('seeder', False):
                                peers_seeding.add(peer['ips'])
                            else:
                                peers_leeching.add(peer['ips'])

                        if peers_dfs:
                            got_dfs_peers = True
                        if self.http_only_mode and not peers_dfs:
                            continue

                        self.announces[client]['seeders'] = seeders
                        self.announces[client]['leechers'] = leechers
                        self.announces[client]['peers_seeding'].update(peers_seeding)
                        self.announces[client]['peers_leeching'].update(peers_leeching)
                        self.announces[client]['peers_dfs'].update(peers_dfs)
                        self.announces[client]['num_completed'] += 1
                        self.announces[client]['last_receive_ts'] = int(time.time())

                        self.seeders = max(self.seeders, seeders)
                        self.leechers = max(self.leechers, leechers)

                        self.peers_seeding_nodfs.update(peers_seeding)
                        self.peers_seeding.update(peers_seeding)
                        self.peers_seeding.update(peers_dfs)
                        self.peers_leeching.update(peers_leeching)
                        self.peers_dfs.update(peers_dfs)

                        client.log.debug(
                            'Announce completed (seeders %d, leechers %d (total: seeders %d, leechers %d))',
                            len(peers_seeding), len(peers_leeching),
                            seeders, leechers
                        )

                        self._log_peers(client.log, peers_seeding, 'Seeders')
                        self._log_peers(client.log, peers_leeching, 'Leechers')
                        self._log_peers(client.log, peers_dfs, 'DFS peers')

                        if isinstance(self.peers, AnnouncerQueue):
                            peers = self.sort_peers_by_loc(peers)

                        for peer in peers:
                            if not peer:
                                continue
                            if self.http_only_mode and not peer.get('dfs', False):
                                continue

                            peerep = peer['ips']
                            peerip, peerport = peerep

                            iplog = IpLoggerAdapter(
                                self.log, extra={
                                    'ip': peerip, 'port': peerport,
                                    'seeder': peer.get('seeder', False),
                                    'dfs': peer.get('dfs', False),
                                }
                            )

                            if peerip in ('127.0.0.1', '::1') and peerport == self.parent.parent.main_port:
                                iplog.debug('Ignoring peer: (matches us)')
                                continue

                            if peerep in self.added_peers:
                                continue
                            else:
                                self.peers.put((peer, iplog))
                                self.added_peers.add(peerep)

                if self.http_only_mode and not got_dfs_peers:
                    self.log.info('Got no DFS peers, retrying announce')
                    return HTTP_ONLY_ANNOUNCE_RETRY_INTERVAL
        else:
            clients = []
            for client, info in self.announces.iteritems():
                if info['num_completed']:
                    clients.append(client)

            with self.ann.announce(self.infohash, ANNOUNCE_STOPPED, self.network, timeout=10, clients=clients) as ann:
                for ev, client, data in ann.wait():
                    pass

            self.stop_ev.set()

        return TRACKER_ANNOUNCE_INTERVAL
# }}}


class PeerManager(Component):  # {{{
    class PeerInfo(object):
        __slots__ = 'ip port dfs seeder connected announced log'.split()

        def __init__(self, **kwargs):
            kwargs.setdefault('ip', None)
            kwargs.setdefault('port', None)
            kwargs.setdefault('dfs', None)
            kwargs.setdefault('seeder', None)
            kwargs.setdefault('connected', 0)
            kwargs.setdefault('announced', 0)
            kwargs.setdefault('log', logging.getLogger())

            for key, value in kwargs.iteritems():
                assert key in self.__slots__, 'Invalid key for PeerInfo: %r' % (key, )
                setattr(self, key, value)

    def __init__(self, parent=None):
        super(PeerManager, self).__init__(logname='prmgr', parent=parent)
        self.announce_mngr = AnnounceManager(parent=self)
        self.announced = False
        self.infohash = None
        self.network = None
        self.peers = {}

        self._new_peers_queue = gevent.queue.Queue()

    def set_network(self, network):
        self.network = network

    def set_infohash(self, infohash):
        self.infohash = infohash

    def search_peers(self):
        if not self.announced:
            self.announce_mngr.announce(self.infohash, self.network)
            self.getpeers_loop.wakeup()
        else:
            self.announced = True

    def reset(self):
        for peerep, peerinfo in self.peers.iteritems():
            self._new_peers_queue.put(peerinfo)
            peerinfo.connected = False

    @Component.green_loop(logname='peermanager')
    def getpeers_loop(self):
        if not self.infohash:
            return 10

        try:
            peer, peerlog = self.announce_mngr.peers.get(timeout=1)
        except gevent.queue.Empty:
            return 0

        peerep = peer['ips']
        if peerep not in self.peers:
            peerinfo = self.PeerInfo(
                ip=peerep[0],
                port=peerep[1],
                dfs=peer.get('dfs', False),
                seeder=peer.get('seeder', False),
                log=peerlog
            )
            peerinfo.log.debug('Got new peer')
            self._new_peers_queue.put(peerinfo)
            self.peers[peerep] = peerinfo
        else:
            peerinfo = self.peers[peerep]
            assert peerinfo.ip, peerinfo.port == peerep
            if time.time() - peerinfo.announced >= TRACKER_ANNOUNCE_INTERVAL:
                peerinfo.log.debug(
                    'Got new peer (last was %d seconds ago)',
                    time.time() - peerinfo.announced
                )
                self._new_peers_queue.put(peerinfo)

        peerinfo.announced = time.time()
        return 0

    def peer_connected(self, peerinfo):
        if (peerinfo.ip, peerinfo.port) not in self.peers:
            self.log.warning(
                'Set connected for peer, but we dont know him: %r',
                (peerinfo.ip, peerinfo.port)
            )
            return

        peerinfo.connected += 1

    def check(self, final=False):
        if self.announce_mngr.all_done():
            if self.announce_mngr.seeders == 0:
                raise api.copier.errors.ResourceNotAvailable(
                    'Got 0 peers having resource (connected to %d from %d possible trackers)' % (
                        len(self.announce_mngr.ann.client.clients),
                        len(self.announce_mngr.ann.client.clients)
                    )
                )
            elif len(self.announce_mngr.peers_seeding) == 0:
                raise api.copier.errors.ResourceNotAllowedByNetwork(
                    'Got 0 peers with %s network (%d total, connected to %d from %d possible trackers)' % (
                        'Backbone' if self.network == 'bb' else 'Fastbone',
                        self.announce_mngr.seeders,
                        len(self.announce_mngr.ann.client.clients),
                        len(self.announce_mngr.ann.client.clients)
                    )
                )

        if final:
            if not self.announce_mngr.has_responses():
                raise api.copier.errors.ResourceDownloadError(
                    'Unable to reach any of %d trackers' % (len(self.announce_mngr.ann.client.clients), )
                )

        return True

    def wait(self, timeout=None):
        try:
            peerinfo = self._new_peers_queue.get(timeout=timeout)
        except gevent.queue.Empty:
            return

        return peerinfo
# }}}


class Progress(object):  # {{{
    def __init__(self, log, resid, child):
        self.log = log
        self.resid = resid
        self.child = child
        self.last_log = 0
        self.last_typ = None

    def proctitle(self, title):
        if setproctitle:
            title = 'skybone-dl: %s' % (title, )
            setproctitle(title)

    def handle(self, typ, *args):
        if typ == 'dl_files':
            info = args[0]

            info_str = 'resid:%8s | au' % (self.resid[:8], )

            if info['stage'] in ('check', 'deduplicate', 'trycopy', 'download'):
                total_bytes = info['total_bytes']
                done_bytes = info['done_bytes']
                if total_bytes > 0:
                    percent = int(done_bytes / total_bytes * 100)
                else:
                    percent = 0
                percent_str = '%2d%%' % (percent, )
            else:
                total_bytes = done_bytes = -1
                percent_str = '---'

            if info['stage'] == 'download':
                dl_spd = info['dl_spd']
                ul_spd = info['ul_spd']
                ul_bytes_payload = info['ul_bytes_payload']
                info_str = '%s | dl %8s | ul %8s' % (info_str, human_speed(dl_spd), human_speed(ul_spd))
            else:
                ul_spd = dl_spd = 0
                ul_bytes_payload = 0

            self.proctitle(
                '%s %s [%s]' % (info['stage'], percent_str, info_str)
            )

            if time.time() - self.last_log > 5 or self.last_typ != typ:
                self.log.debug(
                    'progress: %s (%d bytes done from %d bytes total) [%s]',
                    percent_str, done_bytes, total_bytes, info_str
                )
                self.last_log = time.time()
                self.last_typ = typ

            if self.child:
                send(
                    'progress', {
                        'stage': 'get_resource',
                        'total_bytes': total_bytes,
                        'done_bytes': done_bytes,
                        'ul_bytes': ul_bytes_payload,
                        'state': {}
                    }
                )

        elif typ == 'connect_master':
            if not args[0]:
                self.log.debug('Connecting master...')
                self.proctitle('connecting master...')
            else:
                self.log.info('Connected to master')
                self.proctitle('preparing...')

            if self.child:
                send('progress', {'stage': 'subproc_connect_master', 'done': args[0]})

        elif typ == 'started':
            self.proctitle('initializing...')
            if self.child:
                send('progress', {'stage': 'subproc_started'})

    def log_hashed_bytes(self, hashed, total):
        if hashed == total or time.time() - self.last_log > 5:
            self.log.debug(
                'progress: [%2d%%] %s from %s',
                (hashed / float(total)) * 100,
                human_size(hashed), human_size(total)
            )
            self.last_log = time.time()
# }}}


def send(typ, data):
    data_packed = MSG_PACKER.pack((typ, data))
    datalen = len(data_packed)

    sys.stdout.write(struct.pack('!BI', 242, datalen))
    sys.stdout.write(data_packed)
    sys.stdout.flush()


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('resid')
    parser.add_argument('dest')
    parser.add_argument('--child', action='store_true')
    parser.add_argument('--master-uds', required=True)
    parser.add_argument('--head', action='store_true')
    parser.add_argument('--priority', type=int, required=True)
    parser.add_argument('--priority-hint', required=True)
    parser.add_argument('--network', choices=('Auto', 'Backbone', 'Fastbone'), required=True),
    parser.add_argument('--deduplicate', choices=('No', 'Symlink', 'Hardlink', 'HardlinkNocheck'), required=True),
    parser.add_argument('--max-dl-speed', type=int, required=True)
    parser.add_argument('--max-ul-speed', type=int, required=True)
    parser.add_argument('--tries', default=0, type=int)
    parser.add_argument('--extra')

    return parser.parse_args()


class Download(Component):
    def __init__(self, parent=None):
        super(Download, self).__init__(parent=parent)
        root_log = setup_logger('download')
        self.log = ComponentLoggerAdapter(root_log.getChild('m'), {'prefix': ''})

        self.peer_mngr = PeerManager(parent=self)

        self.master_cli = None
        self.progress = None

        self.uid = None
        self.uds = None
        self.world_desc = None
        self.ips = None
        self.trackers = None
        self.main_port = None
        self.proxy_uds = None

        self.fw_peers_grn = None
        self.allow_seeders = True  # HAPPYSEED LOGIC DISABLE AS OF NOW. SEE SKYDEV-1845
        self.seeders_disallow_deadline = time.time() + SEEDERS_DISALLOW_SECONDS

        self.world = None
        self.handle = None

        self.has_peers = False

        self.mask = os.umask(0o22)
        os.umask(self.mask)

        self.main_grn = None

    def _fw_peers(self):  # {{{
        seeders_waiting = set()

        while True:
            if self.allow_seeders and seeders_waiting:
                # If seeders allowed and we have some of them postponed -- connect them now
                for peerinfo in seeders_waiting:
                    peerinfo.log.debug('Connecting peer %s:%d (postponed seeder)', peerinfo.ip, peerinfo.port)
                    self.handle.add_peer(peerinfo.ip, peerinfo.port, weight=100, is_dfs=peerinfo.dfs)
                    self.peer_mngr.peer_connected(peerinfo)

                seeders_waiting.clear()

            peerinfo = self.peer_mngr.wait(timeout=1)

            self.has_peers = True

            while True:
                if not peerinfo:
                    peerinfo = self.peer_mngr.wait(timeout=0)

                if not peerinfo:
                    # We grabbed all peers for now
                    self.has_peers = False
                    break

                weight = 0

                if peerinfo.seeder:
                    if not self.allow_seeders:
                        if time.time() <= self.seeders_disallow_deadline:
                            if self.peer_mngr.announce_mngr.seeders * 2 < self.peer_mngr.announce_mngr.leechers:
                                # If leechers/4 > seeders, e.g. 2 seeders total and 8 leechers already
                                peerinfo.log.debug('Postponing seeder')
                                seeders_waiting.add(peerinfo)
                                peerinfo = None
                                continue
                            elif self.peer_mngr.announce_mngr.seeders >= self.peer_mngr.announce_mngr.leechers * 2:
                                # If we have too much seeders, force allow_seeders for now
                                self.allow_seeders = True

                                # Seeders come from tracker first, thus, we do not need to change their weight
                                # here
                        else:
                            # We hit deadline for disallowing seeders add them as is
                            # We could have this scenario here:
                            # - we have 4 outgoing conns (this is a max)
                            # - we have a lot (~1000) peers in next queue
                            # In this case we will not make any outgoing connection, but will wait for a slot. Once
                            # we will have slot, we will try 1st of already queued peers and seeder will become
                            # 1000st, which is too slow.
                            #
                            # To fix this instead of simple connect, we set allow_seeders and all seeders will be
                            # added with  greater weight. Thus, they will be tried first once outgoing slot will be
                            # available
                            #
                            # That WILL make a burst of connections to seeders after 10..20 mins of download
                            # progress, but we cant do anything with that as of now. Proper fix will require changes
                            # in tracker

                            self.allow_seeders = True

                            # force connect to this seeders first, other seeders will be scheduled in next loop cycle
                            weight = 100

                peerinfo.log.debug('Connecting peer %s:%d (weight=%d)', peerinfo.ip, peerinfo.port, weight)
                self.handle.add_peer(peerinfo.ip, peerinfo.port, weight=weight, is_dfs=peerinfo.dfs)
                self.peer_mngr.peer_connected(peerinfo)

                peerinfo = None
    # }}}

    def _ensure_directories(self, lst):  # {{{
        target_dir_mode = 0o777 & ~self.mask

        for missing_dir in lst:
            try:
                do_not_stat = False
                missing_dir.ensure(dir=1)
            except py.error.EEXIST:
                if missing_dir.check(exists=1):
                    do_not_stat = True
                    raise api.copier.errors.FilesystemError(
                        'Unable to create missing directory: %s, it is exists, '
                        'but not a directory' % (
                            missing_dir,
                        )
                    )
                else:
                    # probably created by somebody else
                    pass
            except:
                do_not_stat = True
                raise
            finally:
                if not do_not_stat:
                    try:
                        if missing_dir.stat().uid == os.getuid():
                            # Change chmod only if we own this dir
                            if missing_dir.stat().mode & 0o777 != target_dir_mode:
                                missing_dir.chmod(target_dir_mode)
                    except py.error.ENOENT:
                        raise api.copier.errors.FilesystemError(
                            'We created directory %s, but it was removed by somewhere else' % (
                                missing_dir
                            )
                        )
    # }}}

    def _precreate_fs_items(self, dest, structure, partial, use_fallocate):  # {{{
        _checked_directories = set()

        target_file_mode = 0o666 & ~self.mask
        target_file_mode_ex = 0o777 & ~self.mask

        created_paths = []
        fallocated_paths = []
        all_paths = []

        self.log.info(
            'Precreating all fs paths... (%s)',
            'will use fallocate' if (use_fallocate and fallocate) else 'will NOT use fallocate'
        )

        if len(structure) > 1000:
            do_not_log = True
        else:
            do_not_log = False

        dest_realpath = dest.realpath()

        for relative_path, info in structure.iteritems():
            if partial and relative_path not in partial:
                continue

            # This loop is blocking and may take a long time.
            # Allow other greenlets to run.
            gevent.sleep(0)

            pathobj = dest.join(relative_path)

            if pathobj.realpath().common(dest_realpath) != dest_realpath:
                raise api.copier.errors.FilesystemError(
                    'Path %s (realpath %s) is outside of destination %s' % (
                        pathobj, pathobj.realpath(), dest_realpath
                    )
                )

            # Create directories if needed
            ensure_dirs = []
            pathobj_tmp = pathobj

            while True:
                pathobj_tmp = pathobj_tmp.dirpath()

                if pathobj_tmp in _checked_directories:
                    break

                _checked_directories.add(pathobj_tmp)

                try:
                    if pathobj_tmp.check(dir=0):
                        ensure_dirs.append(pathobj_tmp)
                        continue
                    else:
                        break
                except py.error.EACCES as ex:
                    raise api.copier.errors.FilesystemError('Permission denied: %s (%s)' % (pathobj_tmp.strpath, ex))

            # If this is a directory itself -- add it to checked dirs
            if info['type'] == 'dir':
                ensure_dirs.append(pathobj)

            self._ensure_directories(reversed(ensure_dirs))

            # For some reason for resource-type 'symlink' we set
            # structure type 'file'
            if info['type'] not in ('dir', 'symlink') and info['resource']['type'] != 'symlink':
                executable = info['executable']

                try:
                    if pathobj.check(link=1):
                        try:
                            pathobj.remove()
                        except py.error.EACCES as ex:
                            raise api.copier.errors.FilesystemError(
                                'Unable to remove link: %s (%s)' % (pathobj.strpath, ex)
                            )

                    # Files which we want to touch should have write permission for us with no exceptions.
                    # Proper permissions will be set later.
                    if info['resource']['type'] == 'touch' and pathobj.check(file=1):
                        current_mode = pathobj.stat().mode & 0o777
                        needed_mode = current_mode | 0o600
                        self.log.debug(
                            'chmod %04o => %04o (pre-touch): %s',
                            current_mode, needed_mode, pathobj
                        )
                        pathobj.chmod(pathobj.stat().mode | needed_mode)
                except py.error.EACCES:
                    pass

                try:
                    flags = os.O_RDWR | os.O_APPEND

                    if info['resource']['type'] == 'touch':
                        flags |= os.O_TRUNC

                    def _open():
                        try:
                            fd = os.open(pathobj.strpath, flags)
                            created = False
                        except OSError as ex:
                            if ex.errno != errno.ENOENT:
                                raise

                            created = True
                            fd = os.open(pathobj.strpath, flags | os.O_CREAT)
                            pathobj.chmod(target_file_mode)

                        try:
                            if use_fallocate and fallocate and info['size'] > 0:
                                try:
                                    if not do_not_log:
                                        self.log.debug(
                                            'Preallocating %s: (%d bytes using fallocate call%s)',
                                            pathobj, info['size'], ', created' if created else ''
                                        )
                                    fallocate(fd, 0, 0, info['size'])
                                    if created:
                                        fallocated_paths.append(pathobj)
                                except IOError as ex:
                                    if ex.errno == errno.ENOSPC:
                                        self.log.warning(
                                            'Preallocating %s: error: No space left on device',
                                            pathobj
                                        )
                                        try:
                                            self.log.warning(
                                                '  removing fallocated path: %s (no space left on device)',
                                                pathobj
                                            )
                                            pathobj.remove()
                                            for fallocated_pathobj in fallocated_paths:
                                                try:
                                                    self.log.debug(
                                                        '  removing fallocated path: %s (cleaning)',
                                                        fallocated_pathobj
                                                    )
                                                    fallocated_pathobj.remove()
                                                except:
                                                    pass
                                        except:
                                            pass
                                        finally:
                                            raise api.copier.errors.FilesystemError('No space left on device')

                                    elif ex.errno == errno.ENOTSUP:
                                        self.log.warning(
                                            'Preallocating %s: error: Not supported by filesystem',
                                            pathobj
                                        )

                                    else:
                                        raise api.copier.errors.FilesystemError(
                                            'Fallocate failed: %s' % (
                                                errno.errorcode.get(
                                                    ex.errno,
                                                    'errno=%d' % (ex.errno, )
                                                ),
                                            )
                                        )

                                except Exception as ex:
                                    if not do_not_log:
                                        self.log.warning(
                                            'Preallocating %s: error: %s',
                                            pathobj, str(ex)
                                        )

                            if created or pathobj.stat().size != info['size']:
                                os.ftruncate(fd, info['size'])
                                st = os.fstat(fd)
                                self.log.debug(
                                    'Truncate %s done: (inode: %d, size: %d, blocks: %d, mtime: %d)',
                                    pathobj, st.st_ino, st.st_size, st.st_blocks, st.st_mtime
                                )
                        finally:
                            os.close(fd)

                        return created

                    try:
                        created = _open()
                    except OSError as ex:
                        if ex.errno in (getattr(errno, 'EISDIR', 21), ):
                            pathobj.remove()
                            created = _open()
                        else:
                            raise

                    if created:
                        created_paths.append(pathobj)

                except OSError as ex:
                    if ex.errno in (getattr(errno, 'ETXTBSY', 26), ):
                        # Seems file is executed by somebody.
                        # Ignore all errors here, we will check it and if we will need to download
                        # some pieces -- same error will appear
                        pass

                    elif ex.errno in (getattr(errno, 'EACCES', 13), ):
                        # Oops, we have no permission to open that file readwrite.
                        # But, do not raise any error here, unless we were just touching file.
                        # If this is regular file with data -- we will check it later (RO) and will pop same EACCES
                        # error if some data really should be downloaded.
                        # If this is empty file (0 byte) -- raise error here, coz we will not do any data checking
                        # later anyway
                        if info['resource']['type'] == 'touch' and not pathobj.check(file=1):
                            raise api.copier.errors.FilesystemError(
                                'Permission denied: %s' % (pathobj.strpath, )
                            )
                        elif info['type'] == 'file' and not pathobj.check(file=1):
                            raise api.copier.errors.FilesystemError(
                                'Permission denied: %s' % (pathobj.strpath, )
                            )
                    else:
                        raise

                pathobj_mode = pathobj.stat().mode & 0o777

                if pathobj.stat().uid == os.getuid():
                    target_chmod = target_file_mode_ex if executable else target_file_mode
                    if pathobj_mode != target_chmod:
                        if not do_not_log:
                            self.log.debug('chmod %04o => %04o (pre-dl   ): %s', pathobj_mode, target_chmod, pathobj)
                        # TODO: we reset possible extra bits here  # noqa
                        pathobj.chmod(target_chmod)
                else:
                    if executable:
                        if pathobj_mode & 0o111 != 0o111:  # not all bits is set
                            missing_bits = 0o111 ^ (pathobj_mode & 0o111)
                            raise api.copier.errors.FilesystemError(
                                'File %s mode is %04o, we want to set bits %04o, '
                                'because executable bit is set, but dont own file' % (
                                    pathobj, pathobj_mode, missing_bits
                                )
                            )
                    else:
                        if pathobj_mode & 0o111 != 0:  # some bits are set
                            extra_bits = pathobj_mode & 0o111
                            raise api.copier.errors.FilesystemError(
                                'File %s mode is %04o, we want to remove bits %04o, '
                                'because executable bit is not set, but dont own file' % (
                                    pathobj, pathobj_mode, extra_bits
                                )
                            )

                all_paths.append(pathobj.strpath)

        return created_paths, all_paths
    # }}}

    def _precreate_symlinks(self, dest, structure):  # {{{
        for relative_path, info in structure.iteritems():
            if info['resource']['type'] == 'symlink':
                path = dest.join(relative_path)
                oldlink = None
                newlink = info['resource']['data'].split(':', 1)[1]

                if path.check(exists=1, link=0):
                    path.remove()
                elif path.check(link=1):
                    oldlink = path.readlink()
                    if oldlink != newlink:
                        path.remove()

                if oldlink != newlink:
                    path.mksymlinkto(newlink)
    # }}}

    def connect_master(self, uds):  # {{{
        try:
            self.progress.handle('connect_master', False)
            master_cli = RPCClientGevent(uds, port=None)

            tries = 0
            deadline = time.time() + 300
            while True:
                tries += 1
                try:
                    master_cli.connect()
                    break
                except (socket.error, ProtocolError) as ex:
                    self.log.debug('master connect: error: %s: %s, tries: %d', type(ex).__name__, ex, tries)
                    if time.time() > deadline:
                        raise api.copier.errors.ApiConnectionError(
                            'Unable to connect skybone service: %s' % (ex, )
                        )

                    gevent.sleep(min(10, tries * 2))

            self.progress.handle('connect_master', True)

        except Exception as ex:
            self.log.error('master connect failed: %s', ex)
            raise

        self.master_cli = master_cli

        return master_cli
    # }}}

    def make_world(
        self,
        max_dl_speed,
        max_ul_speed,
        local_prob,
        rare_piece_prob,
        idle_conn_timeout,
        compression_params,
        encryption_config
    ):
        if max_dl_speed < 0:
            max_dl_speed = None

        if max_ul_speed < 0:
            max_ul_speed = None

        world = World(
            uid=self.uid, desc=self.world_desc,
            uds=self.uds,
            limit_dl=max_dl_speed, limit_ul=max_ul_speed,
            uds_proxy=self.proxy_uds,
            parent=self,
            ips=self.ips,
            local_prob=local_prob,
            rare_piece_prob=rare_piece_prob,
            idle_conn_timeout=idle_conn_timeout,
            compression_params=compression_params,
            encryption_config=encryption_config
        )
        world.start()
        self.world = world
        self.peer_mngr.announce_mngr.set_peers(local_prob)

    def make_announcer(self, http_only_mode=False, mock_peer_hostnames=None):
        if mock_peer_hostnames is None:
            ann = Announcer(
                self.uid, None, self.trackers, self.ips, 0, self.main_port,
                db_scheduler=False, parent=self
            )
            for address, (worker, _) in ann.client.clients.iteritems():
                worker.set_db_id(-1)

            if self.announcer_state:
                ann.set_state(self.announcer_state)
            ann.start()
        else:
            if not isinstance(mock_peer_hostnames, (list, tuple)):
                mock_peer_hostnames = [mock_peer_hostnames]
            mock_peers = []
            for hostname in mock_peer_hostnames:
                parsed = urlparse.urlparse('//' + hostname)
                addrinfo = socket.getaddrinfo(parsed.hostname, parsed.port or 6881, socket.AF_INET6)

                for info in addrinfo:
                    ip = info[4][0]
                    port = info[4][1]
                    mock_peers.append({
                        'ips': (ip, port), 'dfs': True, 'seeder': True
                    })
            ann = MockAnnouncer(mock_peers, self.log)

        self.peer_mngr.announce_mngr.set_announcer(ann, http_only_mode)

    def set_props(self, props):
        self.uid = props['uid']
        self.world_desc = props['desc']
        self.ips = props['ips']
        self.trackers = props['trackers']
        self.main_port = props['main_port']
        self.proxy_uds = props['proxy_uds']
        self.announcer_state = props.get('announcer_state', None)
        self.loose_path_checking = props.get('loose_path_checking', False)
        self.max_write_chunk = props.get('max_write_chunk', None)
        self.allowed_compression_codecs = props.get('allowed_compression_codecs', [])
        self.allowed_encryption_modes = props.get('allowed_encryption_modes', [encryption.PLAIN])

        # Grab net priorities from master
        self.net_priorities = {}
        for prio, prio_opts in props.get('priorities', {}).items():
            if 'tos' in prio_opts:
                self.net_priorities[prio] = prio_opts['tos'] if prio_opts['tos'] is not None else 0
            else:
                self.net_priorities[prio] = 0

        # Also set numeric aliases
        for name, num in (
            ('Idle', -3),
            ('Low', -2),
            ('BelowNormal', -1),
            ('Normal', 0),
            ('AboveNormal', 1),
            ('High', 2),
            ('RealTime', 3)
        ):
            self.net_priorities[num] = self.net_priorities.get(name, 0)

    def main(self):
        args = parse_args()

        log = ResourceIdLoggerAdapter(self.log.logger, {'prefix': '', 'uid': args.resid})
        self.log = log

        from library.config import query
        try:
            base_extra = query('skynet.services.copier').opts.download
        except (RuntimeError, AttributeError):
            base_extra = {}

        if args.extra:
            args.extra = yaml.safe_load(args.extra)
        else:
            args.extra = {}

        args_from_base = set(base_extra.keys())
        args_from_args = set(args.extra.keys())

        base_extra.update(args.extra)
        args.extra = base_extra

        if args.network == 'Auto':
            network = 'auto'
        elif args.network == 'Fastbone':
            network = 'fb'
        elif args.network == 'Backbone':
            network = 'bb'

        log.debug('Started with args:')
        for arg in sorted(dir(args)):
            if arg.startswith('_'):
                continue
            if arg == 'resid':
                resid = getattr(args, arg)
                log.debug('  %s: %s%s', arg, resid[:8], '?' * 32)
            elif arg == 'extra':
                log.debug('  extra:')
                for exarg, exvalue in sorted(args.extra.iteritems()):
                    if exarg in args_from_args:
                        origin = 'CMD'
                    elif exarg in args_from_base:
                        origin = 'CFG'
                    else:
                        origin = '???'
                    log.debug('    %s: %s: %r', origin, exarg, exvalue)
            else:
                log.debug('  %s: %r', arg, getattr(args, arg))

        self.peer_mngr.set_network(network)
        self.peer_mngr.set_infohash(args.resid)

        self.progress = Progress(log, args.resid, args.child)
        self.progress.handle('started')

        log.info('[resid:%s] Started downloader', args.resid[:8])

        try:
            return self.real_main(args, log)
        except (MasterDisconnect, ProtocolError, RPCError, RPCTimeout, HandshakeTimeout, socket.error) as ex:
            # MasterDisconnect: comes from master_mon coroutine
            # ProtocolError: maybe come from any other place, but we use RPC cli only to talk with master, so this
            # usually means master has gone away
            # RPCTimeout: we usually want to catch "Job registration timed out" error here.

            # RPCError we only grab "gone away" errors here
            if type(ex) is RPCError:
                if 'Server has gone away' not in str(ex):
                    raise

            # Catch only EBADF and ECONNREFUSED here, sometimes RPC can spill that out
            if isinstance(ex, socket.error):
                if ex.errno not in (errno.EBADF, errno.ECONNREFUSED):
                    raise

            tries = args.tries + 1
            if tries == 10:
                raise api.copier.errors.ApiConnectionError('Unable to reconnect master skybone service')

            log.warning('Master disconnect: %s: %s, will retry...', type(ex).__name__, ex)
            args = [sys.executable] + sys.argv

            for idx, arg in enumerate(args):
                if arg == '--tries':
                    args[idx + 1] = str(tries)

            os.closerange(3, subproc.MAXFD)
            os.execve(sys.executable, args, os.environ)

    def real_main(self, args, log):
        opt_compress_data = args.extra.get('compress_data', None)
        opt_direct_read = args.extra.get('direct_read', False)
        opt_direct_write = args.extra.get('direct_write', False)
        opt_force_disable_directio = args.extra.get('force_disable_directio', False)
        opt_sync_writes_period = args.extra.get('sync_writes_period', 0)
        opt_subproc = args.extra.get('subproc', False)
        opt_fallocate = args.extra.get('fallocate', True)
        opt_verify_before_write = args.extra.get('verify_before_write', False)
        opt_quick_check_before_write = args.extra.get('quick_check_before_write', True)
        opt_max_write_chunk = args.extra.get('max_write_chunk', None)
        opt_local_prob = args.extra.get('local_prob', LOCAL_PROB)
        opt_rare_piece_prob = args.extra.get('rare_piece_prob', 0)
        opt_http_only = args.extra.get('http_only', False)
        opt_http_peers = args.extra.get('http_peers', None)
        opt_idle_conn_timeout = args.extra.get('idle_conn_timeout', None)
        opt_trycopy_wait_duration = args.extra.get('trycopy_wait_duration', 0)

        setmtime = args.extra.get('mtime', None)
        gentle_io = args.extra.get('gentle_io', False)

        if opt_force_disable_directio:
            opt_direct_read = False
            opt_direct_write = False

        if opt_quick_check_before_write and 'xxhash' not in sys.modules:
            log.warning('xxhash module not found, disabling quick_check_before_write')
            opt_quick_check_before_write = False

        if opt_direct_write and opt_sync_writes_period:
            log.warning('direct_write is enabled, disabling sync_writes')
            opt_sync_writes_period = 0

        dest = Path(args.dest)

        self.connect_master(args.master_uds)

        partial = args.extra.get('partial', None)

        with master_mon(self.master_cli, opt_subproc):
            log.info('Locking resource...')

            with dl_helper(self.master_cli, args.resid, dest, self.main_grn) as dl, \
                 MetricUpdater(self, dl, args.max_dl_speed) as metrics_updater:

                metrics_updater.start()
                self.set_props(dl.get_props())

                if opt_max_write_chunk:
                    self.max_write_chunk = opt_max_write_chunk

                exc_info = None

                for i in range(100):
                    if os.uname()[0].lower() == 'linux':
                        if i == 0:
                            self.uds = '\0skbn_dl_%d_%d' % (os.getuid(), os.getpid())
                        else:
                            self.uds = '\0skbn_dl_%d_%d_%d' % (os.getuid(), os.getpid(), i)
                    else:
                        self.uds = None

                    compression_params = {'reply': self.allowed_compression_codecs}
                    if opt_compress_data:
                        codec, level = parse_compression_mode(opt_compress_data)
                        if codec not in self.allowed_compression_codecs:
                            raise Exception('compression codec {} is not allowed in config'.format(codec))
                        compression_params['request'] = opt_compress_data

                    encryption_config = self.allowed_encryption_modes

                    try:
                        self.make_world(
                            args.max_dl_speed,
                            args.max_ul_speed,
                            opt_local_prob,
                            opt_rare_piece_prob,
                            opt_idle_conn_timeout,
                            compression_params,
                            encryption_config
                        )
                    except socket.error as ex:
                        exc_info = sys.exc_info()

                        if ex.errno in (getattr(errno, 'EADDRINUSE', 98), ):
                            continue
                        raise
                    else:
                        break
                else:
                    if exc_info:
                        raise exc_info[0], exc_info[1], exc_info[2]
                    else:
                        raise Exception('Unable to create world')

                dl.set_world_sock(self.uds or self.world.sock.getsockname()[:2])

                log.debug('Locking resource in skybone daemon...')

                if not args.head:
                    dl.lock_resource()
                    resource_head = None
                    log.debug('Resource locked')
                else:
                    # First attempt to grab head from cache, if resource already downloaded
                    resource_head = dl.get_head_from_cache(args.resid)
                    if not resource_head:
                        resource_head = dl.lock_resource_or_get_head()
                    if resource_head:
                        log.debug('Resource not locked, but we received head in other downloader')
                    else:
                        log.debug('Resource locked')

                if not args.head:
                    try:
                        dest_in_dom0 = dl.check_dest(dest)
                    except Exception as ex:
                        if self.loose_path_checking:
                            dest_in_dom0 = '(failed to convert)'
                            log.warning(
                                'Destination path check failed: %s: %s, but we continue', type(ex).__name__, ex
                            )
                        else:
                            raise

                    if dest_in_dom0 != dest:
                        log.info('Destination path checked in skybone daemon and converted')
                        log.info('  old: %s', dest)
                        log.info('  new: %s', dest_in_dom0)
                    else:
                        log.info('Destination path checked in skybone daemon')

                if not resource_head:
                    resource_head = dl.get_head_from_cache(args.resid)

                if args.head and resource_head:
                    send('result', resource_head['structure'])
                    return 0
                else:
                    # Grab priority by number or by hint or use default
                    # FYI: we use only net_priority for now
                    priority = self.net_priorities.get(
                        args.priority,
                        self.net_priorities.get(args.priority_hint, None)
                    )
                    self.handle = self.world.handle(args.resid, priority=priority)
                    metrics_updater.set_handle(self.handle)

                self.make_announcer(opt_http_only, mock_peer_hostnames=opt_http_peers)

                if resource_head:
                    self.handle.set_head(resource_head)

                self.handle.get_head()

                if self.handle.head_res.ready() and args.head:
                    send('result', self.handle.head_res.get()['structure'])
                    return 0

                # We want peers and announce if:
                # 1) we have no head
                # 2) we will download/check data (so other peers will be able to connect us)
                # 3) the resource isn't currently announced (so other peers discover us as early as possible)

                self.fw_peers_grn = gevent.spawn(self._fw_peers)
                if not self.handle.head_res.ready() or not dl.check_announces():
                    self.peer_mngr.search_peers()

                if not self.handle.head_res.ready():
                    deadline = time.time() + RESOURCE_HEAD_DL_TIMEOUT
                    # HAPPYSEED LOGIC DISABLE AS OF NOW. SEE SKYDEV-1845
                    # happyseed_deadline = time.time() + HAPPYSEED_HEAD_DL_TIMEOUT

                    while time.time() < deadline:
                        self.handle.head_res.wait(timeout=1)
                        if self.handle.head_res.ready():
                            break
                        else:
                            self.peer_mngr.check()

                        # HAPPYSEED LOGIC DISABLED AS OF NOW. SEE SKYDEV-1845
                        # if not self.allow_seeders and time.time() >= happyseed_deadline:
                        #     log.info('Allow seeders to connect')
                        #     self.allow_seeders = True

                    else:
                        # break was not called above
                        self.peer_mngr.check(final=True)

                if not self.handle.head_res.ready():
                    raise api.copier.errors.ResourceDownloadError(
                        'Unable to receive resource info in %d seconds' % (RESOURCE_HEAD_DL_TIMEOUT, )
                    )

                resource_head = self.handle.head_res.get()

                if args.head:
                    send('result', resource_head['structure'])
                    return 0

                # If we allowed seeders while grabbing head -- disable them again
                # And reenable if we receive 0 blocks in 2 mins
                # HAPPYSEED LOGIC DISABLED AS OF NOW. SEE SKYDEV-1845
                # if self.peer_mngr.announce_mngr.seeders < self.peer_mngr.announce_mngr.leechers * 2:
                #     log.info('Too much leechers -- do not allow seeders for data')
                #     self.allow_seeders = False

                dl.set_head(resource_head)
                alternatives = dl.get_alternatives()
                dl.lock_files()

                created_paths, all_paths = self._precreate_fs_items(
                    dest, resource_head['structure'], partial, opt_fallocate
                )
                self._precreate_symlinks(dest, resource_head['structure'])

                try:
                    dl.check_paths(all_paths)
                except api.copier.errors.FilesystemError as ex:
                    if self.loose_path_checking:
                        log.warning('Paths check failed: %s: %s, but we continue', type(ex).__name__, ex)
                    else:
                        raise

                if gentle_io:
                    sem = IoSemaphore(self.master_cli)
                else:
                    sem = None

                io = IO(1, sem=sem, sync_writes_every=opt_sync_writes_period, parent=None)
                if self.max_write_chunk:
                    io.set_max_write_chunk(self.max_write_chunk)

                shmem = SharedMemory()
                shmem.create_mmap(32 * 1024 * 1024)
                self.handle.set_shared_memory(shmem)

                def _check_io_error(ex, fn, op):
                    if ex.errno in (
                        666,
                        errno.ENOENT,
                        errno.EPERM,
                        errno.EACCES
                    ):
                        log.info('%s: bad file (%s)', fn, str(ex))
                    else:
                        log.warning('%s: unexpected error (%s)', fn, str(ex))
                        raise api.copier.errors.FilesystemError('Unable to %s data: %s (at %s)' % (
                            op, str(ex), fn
                        ))

                def _put_block(fn, start, mem, sha1, quickhash=None):
                    data = mem.peek()
                    if opt_verify_before_write and sha1:
                        data_sha1 = hashlib.sha1(data).hexdigest()
                        assert data_sha1 == sha1, (
                            'SKYDEV-1004, SKYDEV-1366: %s: block (start at %r) checksum mismatch %s != %s' % (
                                fn, start, data_sha1, sha1
                            )
                        )
                    elif opt_quick_check_before_write and quickhash:
                        data_hash = xxhash.xxh3_64_hexdigest(data)
                        if data_hash != quickhash:
                            dl.update_metric('quick_check_failed', 1)
                            assert False, '%s: block (start at %r) hash mismatch %s != %s' % (
                                fn, start, data_hash, quickhash
                            )

                    try:
                        time_start = time.time()
                        _, warn_msg = io.write_block(None, fn, start, data, sha1=sha1, direct=opt_direct_write)
                        if warn_msg:
                            log.warning('write_block returned a warning: %s', warn_msg)
                        if len(data) == PIECE_SIZE:
                            metrics_updater.update_io_write_metric((time.time() - time_start) * 1000)
                    except (OSError, IOError) as ex:
                        _check_io_error(ex, fn, 'write')
                    else:
                        return True

                def _get_block(fn, start, length, mem, sha1):
                    try:
                        time_start = time.time()
                        data = io.read_block(None, fn, start, length, direct=opt_direct_read)
                        if length == PIECE_SIZE:
                            metrics_updater.update_io_read_metric((time.time() - time_start) * 1000)
                    except (OSError, IOError) as ex:
                        _check_io_error(ex, fn, 'read')
                    else:
                        assert mem.offset == mem.size == 0, 'Memory segment already used'
                        mem.write(data)
                        mem.rewind()
                        return True

                self.handle.get_data(
                    dest, _get_block, _put_block,
                    dedup=args.deduplicate,
                    alternatives=alternatives, partial=partial, nocheck=created_paths
                )

                last_pieces_check = 0
                last_announce = 0
                pieces_left_at = [time.time(), -1]  # time, left pieces

                removed_from_trycopy_waiters = False

                while True:
                    self.handle.dl_res.wait(timeout=0.1)

                    if self.handle.stage == 'prepare':
                        self.progress.handle('dl_files', {'stage': 'prepare'})

                    elif self.handle.stage in ('check', 'deduplicate', 'trycopy'):
                        self.progress.handle(
                            'dl_files', {
                                'stage': self.handle.stage,
                                'done_bytes': self.handle.io.stats['done_bytes'],
                                'total_bytes': self.handle.io.stats['total_bytes'],
                            }
                        )

                    else:
                        self.progress.handle(
                            'dl_files', {
                                'stage': 'download',
                                'done_bytes': self.handle.io.stats['done_bytes'],
                                'total_bytes': self.handle.io.stats['total_bytes'],
                                'dl_spd': self.handle.swarm.get_dl_speed(),
                                'ul_spd': self.handle.swarm.get_ul_speed(),
                                'ul_bytes_payload': self.handle.swarm.get_ul_bytes_payload(),
                            }
                        )

                    if self.handle.dl_res.ready():
                        break

                    if self.handle.state == 'get_data':
                        if time.time() - last_announce > 20:
                            self.peer_mngr.search_peers()
                            last_announce = time.time()

                    if self.handle.stage in ('download', 'download_dfs'):
                        if not removed_from_trycopy_waiters:
                            log.debug('removing self from trycopy waiters')
                            dl.remove_trycopy_waiter()
                            removed_from_trycopy_waiters = True

                        if self.handle.stage == 'download_dfs':
                            metrics_updater.set_dfs_mode()

                        if opt_http_only:
                            self.handle.set_allow_dfs(True)

                        if self.peer_mngr.announce_mngr.all_done():
                            # Once we complete all announces check if we have only dfs seeders and 0 regular
                            # seeders. Enable DFS mode ASAP in that case.
                            if (
                                not self.peer_mngr.announce_mngr.peers_seeding_nodfs and
                                self.peer_mngr.announce_mngr.peers_dfs
                            ):
                                self.handle.set_allow_dfs(True)

                        if time.time() - last_pieces_check >= 5:
                            pieces_left = self.handle.cnt_pieces_left
                            log.info('pieces left %r', pieces_left)
                            if pieces_left < 10:
                                log.info("needed pieces: %s", ', '.join(map(str, self.handle.piece_pkr.pieces_needed)))
                                log.info("requested pieces: %s", ', '.join(map(str, self.handle.piece_pkr.pieces_requested)))
                                log.info("checked pieces: %s", ', '.join(map(str, self.handle.piece_pkr.pieces_checked)))

                            if pieces_left_at[1] == -1 or pieces_left < pieces_left_at[1]:
                                pieces_left_at[0] = time.time()
                                pieces_left_at[1] = pieces_left

                            if time.time() - pieces_left_at[0] >= 30:
                                self.handle.set_allow_dfs(True)
                            else:
                                self.handle.set_allow_dfs(False)

                            # HAPPYSEED LOGIC DISABLE AS OF NOW. SEE SKYDEV-1845
                            # if time.time() - pieces_left_at[0] >= HAPPYSEED_DL_TIMEOUT:
                            #     log.info('Allow seeders to connect')
                            #     self.allow_seeders = True

                            if time.time() - pieces_left_at[0] >= ZERO_DL_TIMEOUT:
                                raise api.copier.errors.ResourceDownloadError(
                                    'Received no data in %d seconds, stalled (total done %d pieces from %d)' % (
                                        time.time() - pieces_left_at[0],
                                        self.handle.cnt_pieces_done,
                                        self.handle.cnt_pieces_total
                                    )
                                )

                            last_pieces_check = time.time()

                log.info('possible ready stage -- waiting dl_res')
                self.handle.dl_res.get()

                log.info('dl_res ready -- final steps...')
                log.info("Download stats for rbtorrent:%s: %s", args.resid, self.handle.dc_stats)

                # Fsync all files with data written
                fsync_time = 0
                for fn in io.iter_written_files():
                    ts = time.time()
                    io.fsync(fn)
                    te = time.time()
                    fsync_time += (te - ts)
                    log.debug('fsync [%0.2fs] %s: done', te - ts, fn)

                log.info('fsync [%0.2fs]: all done', fsync_time)

                dl_stats = {}
                items = {}

                if partial:
                    all_files = set([dest.join(p) for p in partial])
                else:
                    all_files = []

                for path, pathinfo in self.handle.io.pathmap.iteritems():
                    if all_files and path not in all_files:
                        continue

                    dl_stats[path.strpath] = (
                        pathinfo[5],  # mtime
                        pathinfo[7],  # inode
                        pathinfo[6],  # size
                        pathinfo[8] & 0o777,  # mode
                    )

                for relative_path, info in resource_head['structure'].iteritems():
                    if partial and relative_path not in partial:
                        continue

                    if info['type'] == 'file':
                        if info['resource']['type'] == 'symlink':
                            items[relative_path] = ResourceItem.symlink(
                                path=dest.join(relative_path),
                                symlink=info['resource']['data'].split(':', 1)[1]
                            )
                        else:
                            path = dest.join(relative_path)

                            if setmtime:
                                if path.check(file=1):
                                    os.utime(path.strpath, (path.stat().atime, setmtime))
                                    if path.strpath in dl_stats:
                                        file_stat = dl_stats[path.strpath]
                                        dl_stats[path.strpath] = (
                                            setmtime, file_stat[1], file_stat[2], file_stat[3]
                                        )

                            if info['resource']['type'] == 'touch':
                                stat = path.stat()

                                mtime, inode, size, perms = stat.mtime, stat.ino, stat.size, stat.mode & 0o777
                                assert size == 0, (
                                    '%s: invalid size after download (%d but should be 0)' % (
                                        path, size
                                    )
                                )
                                sha1_blocks_lt = hashlib.sha1('').digest()
                                sha1_blocks = [1, (4096, sha1_blocks_lt)]  # fake one
                            else:
                                mtime, inode, size, perms = dl_stats[path]
                                tinfo = resource_head['torrents'][info['resource']['id']]['info']
                                assert tinfo['length'] == size, (
                                    '%s: downloaded size dont match (md5: %s): want %d, got %d' % (
                                        path, info.get('md5sum', '\0').encode('hex'),
                                        tinfo['length'], size
                                    )
                                )

                                sha1_blocks = Hasher.convert_sha1_blocks_from_lt_form(
                                    tinfo['pieces'], tinfo['piece length']
                                )

                            items[relative_path] = ResourceItem.file(
                                path=path, size=size, inode=inode, mtime=mtime, perms=perms,
                                md5=info.get('md5sum', None),
                                sha1_blocks=sha1_blocks,
                                chktime=int(time.time())
                            )
                    elif info['type'] == 'dir':
                        items[relative_path] = ResourceItem.directory()

                if not partial:
                    resource = Resource(args.resid, items, data=resource_head)
                    dl.store_resource(resource.to_dict())

                result = [
                    {
                        'name': name,
                        'path': str(m.path),
                        'type': 'file',
                        'size': m.size,
                        'executable': (m.perms & 0o111) > 0,
                        'md5sum': m.md5.encode('hex'),
                    }
                    for name, m in items.items()
                    if isinstance(m, ResourceItem.file)
                ]

                log.info('sending result...')
                send('result', sorted(result, key=lambda x: x['name']))

                if opt_trycopy_wait_duration:
                    # Close connections to peers before releasing the resource lock.
                    # Otherwise connections from other processes on this host
                    # will be considered duplicate and refused by peers.
                    self.handle.stop()

            # dl_helper exited, resource lock is released
            if opt_trycopy_wait_duration:
                await_parallel_downloads(self.master_cli, args.resid, opt_trycopy_wait_duration, log)


def main():
    c = None

    try:
        assert sys.stdin.read() == '', 'invalid stdin'
        sys.stdin.close()

        c = Download()
        c.start()

        return gevent.spawn(c.main).get()

    except Exception as ex:
        info = ' {%d:%d}' % (os.getppid(), os.getpid())

        try:
            if c and c.log:
                c.log.exception(ex)
        except:
            pass

        try:
            send('error', (ex.__module__, ex.__class__.__name__, str(ex) + info))
        except:
            send('error', str(ex) + info)
