import cEcdsa
import contextlib
import gevent
try:
    import gevent.coros as coros
except ImportError:
    import gevent.lock as coros
import random
import time
import urllib2

from . import dfs
from .dc_stats import DCStats
from .logger import HandleLoggerAdapter
from .interrupts import Interrupts
from .peer import Peer, PeerAddress
from .peer_connection import PeerConnection
from .swarm import Swarm
from .piece import PieceMap, PiecePicker
from .io import DlIO, SeedIO

from ..greenlet import LinkedExited
from ..resource.resource import Resource
from ..utils import gevent_urlopen


class HandleHeadReady(Exception):
    pass


class StateChanged(Exception):
    pass


class Disconnect(Exception):
    pass


@contextlib.contextmanager
def dfs_conn_limiter(handle, conn):
    if handle.connected_to_dfs_peer:
        # keeping connections to multiple DFS peers at once is pointless,
        # disconnect immediately
        raise Disconnect('Redundant DFS peer')

    handle.connected_to_dfs_peer = True
    conn.log.debug('DFS conn limiter acquired')
    try:
        yield
    finally:
        handle.connected_to_dfs_peer = False
        conn.log.debug('DFS conn limiter released')


class HandleComp(object):
    def __init__(self, log):
        self.log = log
        self.loops = {}

    def add_child(self, component):
        return


class Handle(object):
    STATE_STOP = 'stop'             # initial state, we dont want anything
    STATE_GET_HEAD = 'get_head'     # we want to grab head somewhere
    STATE_GET_DATA = 'get_data'     # we will want to grab some data
    STATE_READY = 'ready'           # we dont want init connections anymore, but has something to send out

    valid_state_transitions = {
        STATE_STOP: (STATE_GET_HEAD, STATE_READY),
        STATE_GET_HEAD: (STATE_STOP, STATE_READY),
        STATE_GET_DATA: (STATE_STOP, STATE_READY),
        STATE_READY: (STATE_STOP, STATE_GET_DATA)
    }

    CONNECTOR_STATES = (STATE_GET_HEAD, STATE_GET_DATA)

    STATE_DESC = {
        STATE_STOP: {
            'conn': {'accept': False, 'make': False},
            'head': {'want': False},
            'data': {'want': False},
        },
        STATE_GET_HEAD: {
            'conn': {'accept': True, 'make': True},
            'head': {'want': True},
            'data': {'want': False},
        },
        STATE_GET_DATA: {
            'conn': {'accept': True, 'make': True},
            'head': {'want': False},
            'data': {'want': True},
        },
        STATE_READY: {
            'conn': {'accept': True, 'make': False},
            'head': {'want': False},
            'data': {'want': False},
        },
    }

    class GiveUp(Exception):
        pass

    class NoHead(Exception):
        pass

    def __init__(
        self,
        world,
        uid,
        log,
        on_stop=None,
        net_priority=None,
        local_prob=0,
        rare_piece_prob=0,
        idle_conn_timeout=60,
        in_conn_limit=4,
        compression_params=None,
        encryption_config=None
    ):  # {{{
        self.log = HandleLoggerAdapter(log, {'uid': uid})
        self._comp = HandleComp(self.log)

        self.world = world
        self.uid = uid
        self.net_priority = net_priority
        self.idle_conn_timeout = idle_conn_timeout

        self.log.debug('Created new handle [prio=%s]', net_priority)

        self.swarm = Swarm(
            self, uid,
            parent=self._comp,
            local_prob=local_prob,
            compression_params=compression_params,
            encryption_config=encryption_config
        )

        # Semaphores and limits

        # Max number of outgoing connections
        self.conns_ou = coros.Semaphore(4)

        # Max number of incoming connections
        self.conns_in_soft = coros.Semaphore(in_conn_limit)  # max active connections
        self.conns_in_hard = coros.Semaphore(max(in_conn_limit, 16))  # max accepts, tell these peers that we have no slots

        # Ask max 3 active connections for head in parallel
        self.conns_ask_head = coros.Semaphore(3)  # may ask head on 3 connections simultaneously

        self.sign_key = None
        self.sign_key_private = None

        self.head_res = gevent.event.AsyncResult()
        self.dl_res = gevent.event.AsyncResult()
        self.head_extra = None

        self.state = self.STATE_STOP
        self.stage = None

        self.peer_state = {
            'want_head': False,
            'has_head': False,
        }

        self.interrupts = Interrupts()

        self.connector_grn = None
        self.swarm_started = False
        self.worker_grns = set()

        self.last_conn_ts = time.time()

        self.piece_map = PieceMap()
        self.piece_pkr = PiecePicker(local_prob=local_prob, rare_piece_prob=rare_piece_prob)

        self.io = None
        self.io_ready = gevent.event.Event()

        self.shmem = None
        self.shmem_segments_by_piece = {}

        self.partial = None
        self.partial_abs = None

        self.evwaiters = set()

        self.dfs = False
        self.dfs_lock = None
        self.dfs_getlink = None
        self.dfs_allow = gevent.event.Event()
        self.dfs_lock = coros.Semaphore(1)

        self.connected_to_dfs_peer = False

        self._on_stop = on_stop

        self.used_memory = set()

        self.dc_stats = DCStats()
    # }}}

    # Utility meths {{{
    def _set_state(self, state):
        assert state in self.valid_state_transitions

        old_state = self.state

        if state not in self.valid_state_transitions[old_state]:
            self.log.error('Invalid state transition %s => %s', old_state, state)
            raise Exception('Invalid state transition %s => %s' % (old_state, state))

        self.log.info('New state: %s <= from %s', state.upper(), old_state.upper())
        self.state = state

        self.interrupts.interrupt(StateChanged)

        if self._on_stop and state == self.STATE_STOP:
            self._on_stop(self)

    def _check_head(self, head):
        sign_key = head.get('sign_key', None)
        if sign_key:
            sign_key = cEcdsa.Key.from_raw(sign_key)

        rb1 = Resource._generate_rbtorrent1(head['structure'], head['torrents'], sign_key)
        return rb1.infohash() == self.uid

    def _spawn_connector(self):
        if not self.connector_grn:
            self.connector_grn = gevent.spawn(self._connector)
            return True
        return False
    # }}}

    def _connector(self):  # {{{
        self.log.info('Started connector')

        if not self.swarm_started:
            self.swarm.start()
            self.log.info('Started swarm')
            self.swarm_started = True

        while self.STATE_DESC[self.state]['conn']['make']:
            self.conns_ou.acquire()

            # This will block for 3s if there is no peer available
            addr = self.swarm.get_connect_candidate(allow_dfs=not self.connected_to_dfs_peer)

            if addr:
                self.log.debug('New connect candidate %s, initializing conn worker', addr)

                grn = gevent.spawn(self._conn_worker, addr)
                self.worker_grns.add(grn)
                grn.rawlink(lambda grn: self.conns_ou.release())
                grn.rawlink(lambda grn: self.worker_grns.discard(grn))

                self.last_conn_ts = time.time()
            else:
                self.conns_ou.release()
    # }}}

    def handle_incoming_connection(self, sock, payload, ip, port):  # {{{
        self.conns_in_hard.acquire()

        addr = PeerAddress(ip, port)

        if self.conns_in_soft.acquire(blocking=False):
            self.log.debug('[in %-40s]  New Incoming connection', '%s : %d' % (addr.ip, addr.port))
            release_soft = True
            no_slot = False
        else:
            self.log.debug('[in %-40s]  New Incoming connection (NOSLOT)', '%s : %d' % (addr.ip, addr.port))
            release_soft = False
            no_slot = True

        grn = gevent.spawn(self._conn_worker, addr, sock, payload=payload, no_slot=no_slot)
        self.worker_grns.add(grn)
        grn.rawlink(lambda grn: self.conns_in_hard.release())
        grn.rawlink(lambda grn: self.worker_grns.discard(grn))

        if release_soft:
            grn.rawlink(lambda grn: self.conns_in_soft.release())

        self.last_conn_ts = time.time()
    # }}}

    # Connection worker {{{
    def _conn_worker_get_head(self, conn, peer_head_has):  # {{{
        """
        Returns True if head has been downloaded here, False if it has
        been downloaded by other connection.
        """

        log = conn.log

        try:
            with self.interrupts.interruptable([HandleHeadReady]) as interrupts:
                # Wait untill peer says he has head for us
                peer_head_has.wait()

                # Acquire parallel asking semaphore and ask head
                with self.conns_ask_head:
                    with interrupts.suspend():
                        conn.msg_head_want()

                    self.head_res.wait()

        except HandleHeadReady:
            log.debug('stage: get_head cancelled (got head in other conns)')
            return False
        else:
            log.debug('stage: get_head received')
            self.interrupts.interrupt(HandleHeadReady)
            return True

    def _conn_worker_wait_head_want(self, conn, has_ev, want_ev):
        wait_evs = gevent.queue.Queue()

        has_ev.rawlink(wait_evs.put)
        want_ev.rawlink(wait_evs.put)

        ev = wait_evs.get()

        if ev == has_ev:
            # Well, we has head and peer has head. We actually cant do anything else yet
            #
            # This could happen only if peer without HEAD connects to us, and then
            # he grab HEAD somewhere.
            return False

        assert ev == want_ev
        return True
    # }}}

    def _conn_worker_data_loop(self, conn, events):
        log = conn.log

        last_data_transfer = time.time()

        while True:
            if self.state not in (self.STATE_GET_DATA, self.STATE_READY):
                conn.msg_stop('NO_WANT', 'Handle is not in active state anymore')
                return

            if self.state == self.STATE_GET_DATA:
                while 1:
                    piece_memory = self.get_memory(block=False, log=False)
                    if not piece_memory:
                        break

                    piece_idx = self.piece_pkr.get_piece_for_request(conn)
                    if piece_idx is None:
                        self.release_memory(piece_memory, log=False)
                        break

                    self.memory_log('GET', piece_memory.idx)
                    self.shmem_segments_by_piece[piece_idx] = piece_memory

                    log.debug('Request: #%d', piece_idx)
                    conn.msg_piece_req(piece_idx, piece_memory)

            try:
                with self.interrupts.interruptable([StateChanged]) as interrupts:  # noqa
                    while True:
                        try:
                            ev = events.get(timeout=5)
                            break
                        except gevent.queue.Empty:
                            if time.time() - last_data_transfer >= self.idle_conn_timeout:
                                log.warning('Connection IDLE for %d secs -- closing in favour of other conns',
                                            self.idle_conn_timeout)
                                conn.msg_stop_abort('IDLE', 'Connection idle for {} seconds'.format(self.idle_conn_timeout))
                                return
                            continue
            except StateChanged:
                continue

            if ev[0] == 'BAD_PIECE':
                # This means we received piece, but with hash mismatch
                idx = ev[1]
                log.warning('Got BAD PIECE %d', idx)
                conn.msg_stop_abort('BAD_PIECE', 'Got invalid piece #%d' % (idx, ))
                return

            elif ev[0] == 'READ_PIECE':
                # This means we have some requested data to send out
                if self.sign_key:
                    if not conn.verified.wait(timeout=10):
                        return

                    if not conn.verified.get():
                        return

                conn.msg_piece(ev[1], ev[2])

            elif ev[0] == 'HAVE':
                # Peer said he has some new piece
                idx = ev[1]
                self.piece_pkr.peer_have(conn, idx)

            elif ev[0] == 'PIECE_DONE':
                # We received piece (either in this conn, or in another)
                idx = ev[1]
                conn.msg_have(idx)
                last_data_transfer = time.time()

            elif ev[0] == 'RETRY_PIECES':
                # Some pieces are now needed again, try to request them
                # in case this conn hasn't reached the limit
                pass

            elif ev[0] == 'ERROR':
                then = ev[1]
                exc = ev[2]
                tb = ev[3]

                self.log.critical('Error at %r: %s', then, tb)
                self.dl_res.set_exception(exc)
                self._set_state(self.STATE_STOP)

    def _conn_worker_piece_request(self, idx, conn, request_queue):
        # This one is called during receiving peer messages, thus
        # it will block peer messages!
        request_queue.put(idx)

    def _conn_worker_piece_reader_loop(self, request_queue, events):
        active_requests = {}  # idx => memory_segment

        def _on_read(idx):
            shmem = active_requests.pop(idx)
            try:
                events.put(('READ_PIECE', idx, shmem.peek()))
            finally:
                self.release_memory(shmem)

        def _on_read_err(idx, exc, tb):
            shmem = active_requests.pop(idx)
            try:
                events.put(('ERROR', 'READ', exc, tb))
            finally:
                self.release_memory(shmem)

        while True:
            piece_idx = request_queue.get()
            memory_segment = self.get_memory()

            active_requests[piece_idx] = memory_segment
            self.io.async_read(piece_idx, memory_segment, _on_read, _on_read_err)

    def _conn_worker_dfs_seed(self, conn, peer_piecemap_want):
        if 'http' not in conn.capabilities:
            conn.log.warning('Peer not support HTTP capability, aborting connection')
            return

        with self.dfs_lock:
            md5_links = {}
            for md5 in self.piece_map.data:
                links = self.dfs_getlink(self.uid, md5, self.head_res.get())
                if not links:
                    self.log.critical('Unable to get links for md5: %s', md5)
                    return

                if not isinstance(links, (list, tuple)):
                    links = [links]

                md5_links[md5] = links

            conn.msg_file_links(md5_links)

            # Peer should never ask piecemap for us
            # If it did -- that means it does not support LINK messages
            while True:
                peer_piecemap_want.wait(timeout=30)
                if peer_piecemap_want.isSet():
                    conn.log.critical('Peer requested piecemap, aborting connection')
                    return
                else:
                    conn.msg_ping()

    def _conn_worker_dfs_dl(self, conn, dfs_links, on_piece):
        links = dfs_links.get()
        conn.log.info('Http download mode')

        def _pinger():
            while True:
                gevent.sleep(30)
                conn.msg_ping()

        # Build global md5 => [(idx, piecelen)] map, convert to list and shuffle
        #   [md5sum1 [(0, piecelen), (1, piecelen), (N, piecelen)]]
        #   [md5sum2 [(10, piecelen),(11, piecelen),(M, piecelen)]]
        data_idx = dfs.build_data_idx(self.piece_pkr.pmap.data.values())

        dfs.convert_links(links, data_idx, conn.log)
        # now links := {md5: [(start, size, linkspec), ...]}
        # where linkspec is either (link1, ...) or {link1: linkopts, ...}

        piece_memory = self.get_memory()

        try:
            pinger = gevent.spawn(_pinger)

            with self.dfs_lock:
                self.dfs_allow.wait()

                if not self.piece_pkr.pieces_needed:
                    return

                conn.log.info('We allowed to use DFS http download -- do it!')
                self.stage = 'download_dfs'

                for md5, idxs in data_idx:
                    # idxs: [(global_idx1, piecelen), (global_idx2, piecelen), ...]
                    # these are for the whole file, we may have multiple http chunks for this

                    conn.log.debug(
                        'dfs: md5: %s: indexes %d-%d (%d total)',
                        md5, idxs[0][0], idxs[-1][0],
                        idxs[-1][0] - idxs[0][0] + 1
                    )
                    for idx, (start, length, md5_links) in enumerate(links[md5]):
                        if isinstance(md5_links, dict):
                            conn.log.debug('dfs: md5: %s:   [%d-%d] %r', md5, start, start + length, md5_links.keys())
                        else: 
                            conn.log.debug('dfs: md5: %s:   [%d-%d] %r', md5, start, start + length, md5_links)

                    have_missing_pieces_in_file = any(
                        idx in self.piece_pkr.pieces_needed
                        for idx, _ in idxs
                    )

                    if not have_missing_pieces_in_file:
                        # We have all pieces from this file, skip it completely
                        conn.log.debug('dfs: md5: %s: skip all chunks for this file - already done', md5)
                        continue

                    assigned_chunk_links = dfs.assign_pieces_to_chunks(links[md5], idxs)

                    for (
                        http_chunk_idx, http_chunk_start, http_chunk_length, http_chunk_idxs, http_chunk_links
                    ) in assigned_chunk_links:

                        self._dfs_download_chunk(
                            conn, md5, piece_memory, on_piece,
                            http_chunk_idx, http_chunk_start, http_chunk_length,
                            http_chunk_idxs, http_chunk_links,
                        )

        except Exception as ex:
            conn.log.warning('Failed to download file from http: %s', ex)
            raise

        finally:
            self.release_memory(piece_memory)
            pinger.kill()

    def _dfs_download_chunk(self, conn, md5, piece_memory, on_piece, chunk_idx, chunk_start, chunk_length, chunk_idxs, chunk_links):
        if isinstance(chunk_links, dict):
            shuffled_links = random.sample(chunk_links.keys(), len(chunk_links))
            linkopts = chunk_links
        else:
            shuffled_links = random.sample(chunk_links, len(chunk_links))
            linkopts = {}

        for link_idx, link in enumerate(shuffled_links):
            if 'dfs_http_range' in conn.capabilities:
                content_range, idx_range = dfs.get_range_for_request(self.piece_pkr.pieces_needed, chunk_idxs, chunk_length)
                if not content_range:
                    return
            else:
                content_range = 0, chunk_length
                idx_range = chunk_idxs[0][0], chunk_idxs[-1][0]

            last_attempt = (link_idx == len(shuffled_links) - 1)
            timeout = None if last_attempt else dfs.CONNECT_TIMEOUT
            try:
                with gevent.Timeout(timeout) as tout:
                    # we will have None in part start if this is just plain links for whole files
                    conn.log.debug(
                        'dfs: md5: %s: #%d: (start %d, size %d, range %d-%d) request [%s]',
                        md5, chunk_idx, chunk_start, chunk_length, content_range[0], content_range[1], link
                    )

                    req = self._dfs_connect_to_node(conn, link, content_range, linkopts.get(link))

                with contextlib.closing(req):
                    chunk_idxs_iter = (
                        (idx, piecelen)
                        for idx, piecelen in chunk_idxs
                        if idx >= idx_range[0] and idx <= idx_range[1]
                    )

                    self._dfs_receive_data(
                        conn, req, md5, chunk_idx, chunk_idxs_iter, piece_memory, on_piece
                    )

                return
            except (gevent.Timeout, Exception) as ex:
                if isinstance(ex, gevent.Timeout) and ex != tout:
                    raise
                elif last_attempt:
                    raise
                else:
                    conn.log.info('Request failed: %s: %s, will retry', type(ex).__name__, ex)

    def _dfs_connect_to_node(self, conn, link, content_range, linkopts=None):
        http_range = dfs.get_http_range_opt(linkopts)
        if http_range is not None:
            assert 'dfs_http_range' in conn.capabilities
            conn.log.debug('Range header found: %s-%s', *http_range)
            content_range = dfs.adjust_range(content_range, http_range)

        hdrs = {}
        if 'dfs_http_range' in conn.capabilities:
            hdrs['Range'] = 'bytes={}-{}'.format(*content_range)
            conn.log.debug('Requesting range %s-%s', *content_range)

        return gevent_urlopen(link, headers=hdrs)

    def _dfs_receive_data(self, conn, req, md5, http_chunk_idx, chunk_idxs_iter, piece_memory, on_piece):
        # Go thru all pieces assigned to this http chunk and download them
        for idx, piecelen in chunk_idxs_iter:
            conn.log.debug(
                'dfs: md5: %s: #%d: idx %r reading %d bytes',
                md5, http_chunk_idx, idx, piecelen
            )

            with gevent.Timeout(dfs.READ_TIMEOUT, RuntimeError('Read timeout')):
                bucket = self.world.bucket_in
                if bucket:
                    left = piecelen
                    buff = []
                    while left:
                        bucket.leak(8192)
                        data = req.read(8192)
                        if not data:
                            break
                        left -= len(data)
                        buff.append(data)
                    data = ''.join(buff)
                else:
                    data = req.read(piecelen)

            if not data:
                conn.log.info('dfs: %s: md5: #%d: Got EOF', md5, http_chunk_idx)
                raise RuntimeError('Got EOF')

            data_len = len(data)
            self.dc_stats.update(data_len, conn.is_local)
            assert data_len == piecelen, 'Received size %d does not match requested %d' % (
                data_len, piecelen
            )

            if idx not in self.piece_pkr.pieces_needed:
                # If we dont need this piece -- just skip
                conn.log.debug(
                    'dfs: md5: %s: #%d: idx %r -- skip, not needed',
                    md5, http_chunk_idx, idx
                )
                # Prevent starvation of other receivers using bucket_in
                # (we just released the lock on bucket, give other greenlets
                # an opportunity to acquire it)
                gevent.sleep(0)
                continue

            piece_memory.size = 0
            piece_memory.write(data)
            piece_memory.rewind()

            self.shmem_segments_by_piece[idx] = piece_memory

            self.piece_pkr.requests[conn] = set([]), set([idx]), set([])
            self.piece_pkr.register_requested_piece(conn, idx)
            on_piece(idx, 0, block=True, shmem=piece_memory)

    def _conn_worker_on_piece(self, conn, events, idx, num_bytes, block=False, shmem=None):
        # This called once a piece was received
        # If shmem is given -- grab from it, or search shmem in self.shmem_segments_by_piece

        # zero is passed on dfs downloads
        if num_bytes != 0:
            self.dc_stats.update(num_bytes, conn.is_local)

        try:
            data_obj = self.piece_pkr.pmap.data_by_idx[idx]
            if shmem is None:
                piece_memory_segment = self.shmem_segments_by_piece[idx]
            else:
                piece_memory_segment = shmem

            if not data_obj.check(idx - data_obj.offset, piece_memory_segment):
                events.put(('BAD_PIECE', idx))
                if block:
                    raise Exception('Bad piece %d' % (idx, ))

            else:
                self.piece_pkr.piece_checked(conn, idx)

                if block:
                    wrote_ev = gevent.event.AsyncResult()

                def _wrote_piece(idx):
                    # This means we received some piece
                    if shmem is None:
                        shmem_segment = self.shmem_segments_by_piece.pop(idx, None)
                        if shmem_segment:
                            self.release_memory(shmem_segment)

                    self.piece_pkr.piece_done(conn, idx)

                    for evs in self.evwaiters:
                        evs.put(('PIECE_DONE', idx))

                    if (
                        not self.piece_pkr.pieces_needed and
                        not self.piece_pkr.pieces_requested and
                        not self.piece_pkr.pieces_checked
                    ):
                        self.dl_res.set(True)

                        if self.state == self.STATE_GET_DATA:
                            self._set_state(self.STATE_READY)

                    if block:
                        wrote_ev.set(True)

                def _wrote_err(idx, exc, tb):
                    if shmem is None:
                        shmem_segment = self.shmem_segments_by_piece.pop(idx, None)
                        if shmem_segment:
                            self.release_memory(shmem_segment)

                    self.piece_pkr.piece_unchecked(conn, idx)
                    events.put(('ERROR', 'WRITE', exc, tb))

                    if block:
                        wrote_ev.set_exception(exc)

                self.io.async_write(
                    idx, piece_memory_segment,
                    _wrote_piece,
                    _wrote_err
                )

                if block:
                    wrote_ev.get()

        except Exception as ex:
            import traceback
            events.put(('ERROR', 'WRITE', ex, traceback.format_exc()))

            if block:
                raise

    def _conn_worker_data(self, conn, peer_head_has, peer_head_want):
        assert self.state in (self.STATE_GET_DATA, self.STATE_READY)

        events = gevent.queue.Queue()

        peer_piecemap_want = gevent.event.Event()
        peer_piecemap_res = gevent.event.AsyncResult()

        dfs_links = gevent.event.AsyncResult()

        conn.register_cb('piecemapreq', peer_piecemap_want.set)
        conn.register_cb('piecemap', lambda pieces: peer_piecemap_res.set(pieces))
        conn.register_cb('links', dfs_links.set)

        try:
            conn.msg_head_has()

            if not peer_head_has.ready():
                if self._conn_worker_wait_head_want(conn, peer_head_has, peer_head_want):
                    # If we want to auth peers -- wait verification
                    if self.sign_key:
                        if not conn.verified.wait(timeout=10):
                            return

                        if not conn.verified.get():
                            return

                    # Peer said he want head from us! Give it!
                    conn.msg_head(self.head_res.get())

                    # Now wait until it reports he got head
                    peer_head_has.wait()

            while True:
                try:
                    with self.interrupts.interruptable([StateChanged]):
                        self.piece_map.loaded.wait()
                        break
                except StateChanged:
                    if self.state == self.STATE_STOP:
                        return

            if self.piece_pkr.pmap is None:
                self.piece_pkr.load_pmap(self.piece_map, self.partial)

            if self.dfs:
                # We are dfs peer
                # So, our only job is to send http links to 1 peer at a time
                self._conn_worker_dfs_seed(conn, peer_piecemap_want)
            else:
                self.io_ready.wait()

                if self.io.piecemap is None:
                    # We do not expect peer to request something before we send our piecemap, so it is
                    # safe to map piecemap to io right now
                    self.io.set_piecemap(self.piece_map)
                    self.io.start()

                def _on_piece(idx, num_bytes, block=False, shmem=None):
                    return self._conn_worker_on_piece(conn, events, idx, num_bytes, block, shmem)

                if 'dfs' in conn.capabilities:
                    # This is DFS peer, so we should expect only link messages from him
                    with dfs_conn_limiter(self, conn):
                        self._conn_worker_dfs_dl(conn, dfs_links, _on_piece)

                else:
                    requests_queue = gevent.queue.Queue()
                    conn_worker_piece_reader_grn = gevent.spawn(
                        self._conn_worker_piece_reader_loop, requests_queue, events
                    )

                    try:
                        conn.register_cb('piecereq', lambda idx: (
                            self._conn_worker_piece_request(idx, conn, requests_queue)
                        ))

                        conn.register_cb('piece', _on_piece)

                        conn.register_cb('have', lambda idx: events.put(('HAVE', idx)))

                        conn.msg_piece_map_req()

                        peer_piecemap_want.wait()

                        # After we send piecemap -- we will receive requests from peer if it wants something
                        conn.msg_piece_map(self.piece_map.get_map().pieces)

                        # First of all -- wait for other's piecemap
                        peer_piecemap_res.wait()
                        self.piece_pkr.set_peer_pieces(conn, peer_piecemap_res.get())

                        self.evwaiters.add(events)

                        try:
                            self._conn_worker_data_loop(conn, events)
                        finally:
                            self.evwaiters.discard(events)
                    finally:
                        conn_worker_piece_reader_grn.kill()

        except LinkedExited:
            # Oops, something happened
            # If this is our receiver greenlet died -- popup error
            raise

        return

    def _randchr(self):
        while True:
            x = random.randint(0, 58)
            if x <= 25 or x > 31:
                return chr(65 + x)

    def _conn_worker(self, addr, sock=None, payload=None, no_slot=False):  # {{{
        # Main busyness logic unit
        # Here we determine what to do on each peer connection

        # First step, we need to connect to specified address and initialize
        # peer object. If connect fails -- we push address back to connect
        # candidates.
        #
        # Reconnect logic:
        # - if we were unable to connect peer (so, we dont know it's uid), just put
        #   plain address back to connect queue
        # - if we were unable to connect peer earlier, but cant connect here -- push back
        #   plain address. PeerCollection will search for existing peer by address and will put it
        #   to queue instead of plain address if found.
        # - if we were able to connect peer, and we know it's uid -- push back peer to connect queue.
        #   this is done in swarm.collect_peer() logic

        try:
            addr.connect_attempts += 1
            peer = self.swarm.connect_addr(addr, sock, payload=payload)
            addr.connect_succeeded = True

        except self.swarm.ConnectFailed as ex:
            if sock is not None:
                # Address is not connectable, probably that was an incoming
                # connection. We should not add it to the pool of connection
                # tries in any case
                return

            if isinstance(ex, (self.swarm.NoSkybit, self.swarm.ConnectedToSelf)):
                self.swarm.peers.addrs_next.discard(addr)

            elif isinstance(ex, self.swarm.HandshakeFailed):
                self.swarm.add_candidate_address(addr, 1)

            elif isinstance(ex, self.swarm.ConnectFailed):
                self.swarm.add_candidate_address(addr, 3)

            return

        except PeerConnection.EPIPE:
            # This was probably a libtorrent which closed connection
            # even before we were able to send handshake. But this could be
            # also a copier which has been restarted
            #
            # But we add it as candidate only if it was not incoming connection.
            if sock is not None:
                self.swarm.add_candidate_address(addr, 120)

            return

        except self.swarm.Deactivate as ex:
            # Since this means peer already connected to us somehow -- remove it's address
            # from theoretical candidates
            self.swarm.peers.addrs_next.discard(addr)

            ex.conn.log.debug('Connection was not activated: %s', str(ex))
            try:
                ex.conn.msg_stop_abort('NO_ACTIVATE', 'Not activated')
            except ex.conn.EPIPE:
                pass
            ex.conn.close()

            if sock is not None:
                # Do not ever add incoming connections as candidates -- or we will possibly try to
                # connect non-6881 port
                self.swarm.add_candidate_address(addr, 120)

            return
        else:
            # If handshake was completed, we should not consider this address as
            # "new" or "unkown" anymore
            self.swarm.peers.addrs_next.discard(addr)

        assert peer.state in (
            Peer.CONNECTED, Peer.CAP_MISMATCH,
            Peer.NO_RESOURCE, Peer.NO_SLOT,
        ), 'Unexpected peer state: %r' % (peer.state, )

        if peer.state != Peer.CONNECTED:
            return

        conn = peer.conn
        log = conn.log

        if no_slot:
            try:
                try:
                    conn.msg_stop('NO_SLOT', 'No more slots')
                except conn.EPIPE:
                    pass
            except self.swarm.Deactivate:
                try:
                    conn.msg_stop_abort('DE_ACTIVATE', 'Deactivated')
                except conn.EPIPE:
                    pass
            finally:
                conn.close()
                peer.conn = None
                return

        if self.state not in (self.STATE_GET_HEAD, self.STATE_GET_DATA, self.STATE_READY):
            try:
                try:
                    conn.msg_stop_abort('STOPPED', 'Handle is stopped')
                except conn.EPIPE:
                    pass
            except self.swarm.Deactivate:
                try:
                    conn.msg_stop_abort('DE_ACTIVATE', 'Deactivated')
                except conn.EPIPE:
                    pass
            finally:
                conn.close()
                peer.conn = None
                return

        try:
            self.swarm.conns_active[conn] = peer

            peer_head_want = gevent.event.Event()
            peer_head_has = gevent.event.Event()

            conn.register_cb('head_want', peer_head_want.set)
            conn.register_cb('head_has', peer_head_has.set)
            conn.register_cb('head', self._on_head)

            # If we have public key to check auth, register verify callback and ask
            if self.sign_key:
                data = ''.join([self._randchr() for _ in range(8)])

                def _verify(sign):
                    verify_result = self.sign_key.verify(data, sign)
                    conn.log.info('Connection verification result: %s' % ('yes' if verify_result else 'no', ))
                    conn.verified.set(verify_result)

                conn.register_cb('verify', _verify)
                conn.msg_auth(data)

            def _auth(data):
                if not self.sign_key_private:
                    if not self.head_res.ready():
                        self.head_res.set_exception(Exception('Authorization required to download this resource'))

                    if not self.dl_res.ready():
                        self.dl_res.set_exception(Exception('Authorization required to download this resource'))

                    conn.log.warning('Auth asked, but we have no private key for that')
                else:
                    conn.msg_verify(self.sign_key_private.sign(data))

            conn.register_cb('auth', _auth)

            messages_async_grn = self._start_async_messages_grn(conn.process_messages)

            try:
                try:
                    if self.state == self.STATE_GET_HEAD:
                        log.debug('stage: get_head')

                        if self._conn_worker_get_head(conn, peer_head_has):
                            # We received head from this peer. Now we both have heads.
                            pass
                        else:
                            # This means we did received head in other conns
                            # Since we probably need to pass head to this peer -- do not disconnect yet
                            pass

                    # ... and also we must have head
                    assert self.head_res.ready(), 'head_res is not ready after conn_worker_get_head'

                    if self.state in (self.STATE_GET_DATA, self.STATE_READY):
                        self._conn_worker_data(conn, peer_head_has, peer_head_want)

                except conn.EPIPE:
                    log.debug('worker died: broken pipe')

                    peer.set_state(Peer.DISCONNECTED)

            except LinkedExited as ex:
                if messages_async_grn.ready():
                    # Oops receiver loop exited accidentially!
                    try:
                        result = messages_async_grn.get()
                        if result is not None:
                            result_type, result_text = result

                            if result_type == 'CLOSED_REMOTE':
                                log_severity = log.warning
                            else:
                                log_severity = log.debug

                            log_severity('process_messages: exited (%s)', result_text)

                            # If receiver just died without no reason, this could happen if
                            # another connection with higher votenum was created to same peer.
                            # So, we are setting DISCONNECTED state only if peer's connection match
                            # this one

                            if result_type in (
                                'CLOSED_REMOTE', 'CLOSED_BOTH',
                                'CLOSED_NOACT', 'CLOSED_DEACT',
                            ):
                                peer.set_state(Peer.DISCONNECTED)
                            elif result_type == 'NO_SLOT':
                                peer.set_state(Peer.NO_SLOT)
                            elif result_type == 'NO_RESOURCE':
                                peer.set_state(Peer.NO_RESOURCE)
                            elif result_type == 'NO_NEED':
                                peer.set_state(Peer.NO_NEED)
                            elif result_type == 'STOPPED':
                                log.info('Peer is not ready to accept us (STOPPED)')
                                peer.set_state(Peer.DISCONNECTED)
                            elif result_type == 'IDLE':
                                log.info('Peer closed connection because it is IDLE')
                                peer.set_state(Peer.DISCONNECTED)
                            else:
                                log.error('process_messages: exited with unknown result type: %r', result_type)
                                peer.set_state(Peer.DISCONNECTED)
                        else:
                            log.debug('process_messages: exited (ok)')
                            if peer.conn == conn:
                                peer.set_state(Peer.DISCONNECTED)

                    except Exception as ex:
                        log.warning('process_messages: died (%s: %s)', type(ex).__name__, ex)
                        peer.set_state(Peer.DISCONNECTED)  # disconnected, because messages send/receive loop died
                else:
                    # Another link exited, we dont know what to do with that now
                    raise

            except self.swarm.Deactivate as ex:
                conn.log.debug('Connection was deactivated: %s', str(ex))
                try:
                    conn.msg_stop_abort('DE_ACTIVATE', 'Deactivated')
                except conn.EPIPE:
                    pass

            except Disconnect as ex:
                conn.log.debug('Dropping connection: %s', ex)
                try:
                    conn.msg_stop_abort('DROP', str(ex))
                except conn.EPIPE:
                    pass

            except Exception as ex:
                log.error('worker died with error: %s: %s', type(ex).__name__, ex)
                import traceback
                log.error(traceback.format_exc())
                peer.set_state(Peer.DISCONNECTED)

            finally:
                self._stop_async_messages_grn(messages_async_grn, conn.close, conn.log)

        finally:
            # Final worker part.
            conn.close()

            # Remove any pending piece requests in this connection and clean shmem blocks
            # This MUST be done after full connection close
            idxs = self.piece_pkr.discard_requests(conn)

            for idx in idxs:
                shmem_segment = self.shmem_segments_by_piece.pop(idx, None)
                if shmem_segment:
                    self.release_memory(shmem_segment)

            peer.conn = None

            if peer.state == Peer.CONNECTED:
                peer.set_state(Peer.DISCONNECTED)

            self.swarm.conns_active.pop(conn)

            if self.piece_pkr.pieces_needed:
                log.debug('Peer disconnected, waking up all connections')
                for evs in self.evwaiters:
                    evs.put(('RETRY_PIECES', ))

            # This is called once in handler at the end of worker logic.
            # Depending on peer state -- it will or will not be rescheduled to reconnect.
            self.swarm.collect_peer(peer)
    # }}}
    # }}}

    # Async incoming messages handler {{{
    def _start_async_messages_grn(self, meth):
        async_grn = gevent.spawn(meth, self.shmem_segments_by_piece)

        # raise LinkedExited here once it dies
        self._async_messages_grn_exit_notifier = lambda grn, current=gevent.getcurrent(): current.throw(LinkedExited) if not current.ready() else None
        async_grn.link(self._async_messages_grn_exit_notifier)

        return async_grn

    def _stop_async_messages_grn(self, async_grn, stopper, log):
        # Unlink async grn
        async_grn.unlink(self._async_messages_grn_exit_notifier)

        if not async_grn.ready():
            ts = time.time()
            # Allow connection to close properly
            try:
                async_grn.join(timeout=1)
            except LinkedExited:
                pass

            if async_grn.ready():
                log.debug('process_messages: exited normally (we wait %0.4fs)', time.time() - ts)
            else:
                log.info('process_messages: didnt exited, aborting (we wait %0.4fs)', time.time() - ts)

        stopper()

        if not async_grn.ready():
            ts = time.time()

            # Attempt to read final portion of data
            try:
                async_grn.join(timeout=1)
            except LinkedExited:
                pass

            if async_grn.ready():
                log.debug(
                    'process_messages: exited after conn stop (we wait %0.4fs)', time.time() - ts
                )
            else:
                log.info(
                    'process_messages: didnt exited after stopping (we wait %0.4fs)', time.time() - ts
                )

                ts = time.time()
                # Kill it!
                async_grn.kill()
                log.info('process_messages: killed in %0.4fs', time.time() - ts)
    # }}}

    # Asunc events {{{
    def _on_head(self, head):
        """ This event called once head is received from network. """

        if not self.head_res.ready():
            if self._check_head(head):
                if self.state == self.STATE_GET_HEAD:
                    self._set_state(self.STATE_READY)

                if 'sign_key' in head:
                    self.sign_key = cEcdsa.Key.from_raw(head['sign_key'])

                self.head_res.set(head)
            else:
                raise Exception('Head hash mismatch')

    def _on_head_extra(self, extra):
        self.head_extra = extra
    # }}}

    # Public interface {{{
    def idle_time(self):
        if self.worker_grns:
            return 0

        return time.time() - self.last_conn_ts

    def add_peer(self, ip, port, weight=0, is_dfs=False):
        """ Add new peer from outside. """
        self.swarm.add_candidate(ip, port, weight, is_dfs)

    def enable_dfs_mode(self, getlink):
        self.dfs = True
        self.dfs_lock = coros.Semaphore(1)
        self.dfs_getlink = getlink

    def set_shared_memory(self, mem):
        self.shmem = mem

    def _memory_info_txt(self):
        result = []
        used_here = set([mem.idx for mem in self.used_memory])

        for i in range(self.shmem.pieces):
            result.append('X' if i in used_here else 'x' if i not in self.shmem.free_set else ' ')

        return ''.join(result)

    def get_memory(self, block=True, log=True):
        mem = self.shmem.get_segment(block=block)

        if not mem:
            return

        self.used_memory.add(mem)

        if log:
            self.memory_log('GET', mem.idx)

        return mem

    def release_memory(self, mem, log=True):
        self.shmem.put_segment(mem)
        self.used_memory.discard(mem)

        if log:
            self.memory_log('FREE', mem.idx)

    def memory_log(self, op, idx):
        self.log.debug('Memory: %4s  %r  [%s]', op, idx, self._memory_info_txt())

    def set_head(self, head, extra=None):
        """ Set head from outside. It will check hash and interrupt all
        pending downloads if any. """

        if not self.head_res.ready():
            try:
                self._on_head(head)
                if extra:
                    self._on_head_extra(extra)
                self.interrupts.interrupt(HandleHeadReady)
            except Exception as ex:
                self.log.debug('Unable to set head: %s', ex)
                return False
            else:
                self.log.info('Set already downloaded head')
                return True
        else:
            self.log.warning('Set head which already downloaded, ignoring')
            return True

    def set_allow_dfs(self, flag):
        if flag:
            self.dfs_allow.set()
        else:
            self.dfs_allow.clear()

    def set_sign(self, sign):
        self.sign_key = sign.public_key()
        self.sign_key_private = sign

    def get_head(self):
        self.log.info('Get head')

        if not self.head_res.ready():
            self._set_state(self.STATE_GET_HEAD)

            self._spawn_connector()

        elif self.state == self.STATE_STOP:
            # If head_res already set -- update our state if needed
            if self.state == self.STATE_STOP:
                self._set_state(self.STATE_READY)

        return self.head_res

    def _get_data_worker(self, dedup, alternatives, nocheck):
        def _piece_done(idx):
            self.piece_pkr.piece_done(None, idx)

            for evs in self.evwaiters:
                evs.put(('PIECE_DONE', idx))

        if dedup:
            self.stage = 'deduplicate'
            self.log.info('Stage: deduplicating (mode %r)', dedup)
            self.io.set_deduplicate(dedup)

            if alternatives:
                # Phase 1: attempt to deduplicate files if we have something on host sys
                self.io.deduplicate(alternatives, _piece_done, self.shmem, partial=self.partial)

        self.stage = 'check'
        self.log.info('Stage: checking existing files')
        # Phase 2: check files which already exist, and was not deduplicated
        self.io.check(_piece_done, self.shmem, self.partial_abs, nocheck=nocheck)

        if alternatives and not self.partial:
            self.stage = 'trycopy'
            self.log.info('Stage: trycopy alternatives')
            # Phase 3: attempt to find any missing pieces and copy them
            self.io.trycopy(alternatives, _piece_done, self.shmem)

        if self.piece_pkr.pieces_needed:
            self.stage = 'download'
            self.log.info('Stage: download missing pieces')
            self._set_state(self.STATE_GET_DATA)

            self._spawn_connector()
        else:
            self.dl_res.set(True)

    def get_data(
        self, path, get_block, put_block,
        dedup=None, alternatives=None,
        partial=None, nocheck=None
    ):
        assert self.head_res.ready(), 'Head was not received yet'
        head = self.head_res.get()

        assert self.shmem, 'No shared memory has been set!'

        if partial:
            self.partial = set(partial)
            self.partial_abs = set(path.join(p) for p in self.partial)

        if not self.dl_res.ready():
            assert dedup in (None, 'No', 'Hardlink', 'HardlinkNocheck', 'Symlink')

            if dedup and dedup != 'No':
                dedup = dedup.lower()
            else:
                dedup = False

            # After this call incoming connections which are waiting on piecemap will send it out
            self.piece_map.load_head(head)

            if self.piece_pkr.pmap is None:
                # After we loaded head pmap actually may be loaded by some active connections
                self.piece_pkr.load_pmap(self.piece_map, self.partial)

            if self.io is None:
                self.io = DlIO(log=self.log.getChild('io'))
                self.io.set_dl_cbs(path, get_block, put_block)
                self.io.set_piecemap(self.piece_map)
                self.io.start()

                self.io_ready.set()

            if self.partial:
                self.io.set_partial(self.partial)

            if alternatives:
                # Filter out alternatives which match our own targets
                for md5hash, paths in alternatives.iteritems():
                    filtered = tuple([pair for pair in paths if pair[0] not in self.io.pathmap])
                    alternatives[md5hash] = filtered

            self.stage = None
            gevent.spawn(self._get_data_worker, dedup, alternatives, nocheck)

        else:
            # If dl_res already set -- switch state to READY if we are stopped
            if self.state == self.STATE_STOP:
                self._set_state(self.STATE_READY)

        return self.dl_res

    @property
    def cnt_pieces_done(self):
        return self.cnt_pieces_total - self.cnt_pieces_left

    @property
    def cnt_pieces_total(self):
        return len(self.piece_pkr.pmap.data_by_idx)

    @property
    def cnt_pieces_left(self):
        pkr = self.piece_pkr
        return len(pkr.pieces_needed) + len(pkr.pieces_requested) + len(pkr.pieces_checked)

    def seed(self, get_block):
        if not self.head_res.ready():
            raise self.NoHead('Unable to seed -- head not ready yet')

        if self.io is None:
            self.io = SeedIO(log=self.log.getChild('io'))
            self.io.start()
            self.io_ready.set()

        if get_block is not None:
            assert self.shmem, 'No shared memory has been set!'
            self.io.set_seed_cb(get_block)

        self.dl_res.set(True)
        self.piece_map.load_head(self.head_res.get())
        self.piece_map.all_done()

        self.log.info('Seeding')
        self._set_state(self.STATE_READY)

    def stop(self):
        self.log.debug('Stopping handle')

        if self.state != self.STATE_STOP:
            self._set_state(self.STATE_STOP)

        if self.io and self.io.workers:
            self.io.stop()

        if self.connector_grn:
            self.connector_grn.kill()
            self.connector_grn = None

        self.swarm.stop()

        reraise = False
        for grn in list(self.worker_grns):
            try:
                grn.kill(block=True)

            except gevent.GreenletExit:
                # Since stop may be initiated inside this greenlet, record that case
                # and reraise at the end.
                reraise = True

        self.worker_grns.clear()

        if self.shmem:
            for segment in list(self.used_memory):
                self.release_memory(segment)

            self.used_memory.clear()

        self.log.info('Handle stopped (reraise: %r)', reraise)

        if reraise:
            raise gevent.GreenletExit('reraise')
    # }}}


class NoResourceHandle(object):  # {{{
    def __init__(self, world, log):
        self.world = world
        self.log = log
        self.payload = None

        self.max_parallel_handles = coros.Semaphore(32)

    def handle_incoming_connection(self, sock, payload, ip, port):
        self.max_parallel_handles.acquire()

        self.payload = payload

        try:
            swrm = Swarm(self, None, None)
            conn = PeerConnection(swrm, (ip, port), sock, connected=True, log=self.log.getChild('conn'))
            conn.start()

            grn = gevent.spawn(self.worker, conn)
            grn.rawlink(lambda grn: self.max_parallel_handles.release())
            grn.rawlink(lambda grn: conn.close())
            grn.rawlink(lambda grn: sock.close())
        except Exception:
            self.max_parallel_handles.release()
            raise

    def worker(self, conn):
        # Since initiator of this types of connections (to NoResource handler)
        # is always other-initiated, we allow remote peer to sendout data to us
        # and close connection by itself for 1 second more.
        with gevent.Timeout(1) as tout:
            infohash, world_uid, world_desc = conn.read_handshake(self.payload)
            conn.write_handshake(infohash, self.world.uid, self.world.desc, dfs=False)
            conn.msg_stop('NO_RESOURCE', 'No such resource here')

            try:
                while True:
                    msgtype, payload = conn.get_message()
                    if not msgtype:
                        break
                    else:
                        conn.log.debug('RCV: %s', msgtype)
            except gevent.Timeout as ex:
                if ex != tout:
                    raise
            finally:
                conn.close()


class InfoHashDetectHandle(object):
    def __init__(self, world, log):
        self.world = world
        self.log = log

    def handle_incoming_connection(self, sock, payload, ip, port):
        swrm = Swarm(self, None, None)
        conn = PeerConnection(
            swrm, (ip, port), sock, connected=True, log=self.log.getChild('conn')
        )
        conn.store_payload = True
        conn.start()

        infohash, world_uid, world_desc = conn.read_handshake()
        return infohash, ''.join(conn.payload)
# }}}
