import contextlib

import gevent
try:
    import gevent.coros as coros
except ImportError:
    import gevent.lock as coros


class InterruptsProxy(object):
    def __init__(self, parent, current):
        self.parent = parent
        self.current = current

    @contextlib.contextmanager
    def suspend(self):
        self.parent.suspended[self.current] = None
        try:
            yield
        finally:
            ex = self.parent.suspended.pop(self.current)
            if ex:
                raise ex


class Interrupts(object):
    def __init__(self):
        self.waiters = {}
        self.interrupts = {}
        self.suspended = {}
        self.lock = coros.Semaphore(1)

    @contextlib.contextmanager
    def interruptable(self, events):
        import sys
        current = gevent.getcurrent()

        proxy = InterruptsProxy(self, current)

        try:
            locked = False
            with self.lock:
                for event in events:
                    waiters = self.waiters.setdefault(event, set())
                    waiters.add(current)

            yield proxy

            if not self.lock.locked():
                # Dont allow to interrupt us anymore
                self.lock.acquire()  # this can be interrupted as well
                locked = True

            exc_info = None

        except BaseException as ex:
            exc_info = sys.exc_info()
            raise
        finally:
            if not self.lock.locked():
                self.lock.acquire()
                locked = True

            try:
                if exc_info:
                    ex = exc_info[1]
                else:
                    ex = None

                if current in self.interrupts:
                    # We have pending interruptions for this greenlet
                    interrupt, ev = self.interrupts.pop(current)

                    while ex != interrupt:
                        # Wait for interrupt to happen
                        try:
                            gevent.sleep(1000)
                        except BaseException as ex:
                            continue

                else:
                    ev = None

                if ev:
                    ev.set()

                # Interrupt happend, signaling event
                for event in events:
                    self.waiters[event].discard(current)

            finally:
                if locked:
                    self.lock.release()

                if exc_info:
                    raise exc_info[0], exc_info[1], exc_info[2]

    def interrupt(self, event):
        import sys

        with self.lock:
            if event not in self.waiters:
                return

            wait_evs = []
            for waiter in list(self.waiters[event]):
                ev = gevent.event.Event()
                interrupt = event('killing %r' % (waiter, ))

                assert waiter not in self.interrupts

                if waiter in self.suspended:
                    self.suspended[waiter] = interrupt
                else:
                    self.interrupts[waiter] = (interrupt, ev)
                    wait_evs.append(ev)
                    waiter.kill(interrupt, block=0)

            ei = None
            for ev in wait_evs:
                while True:
                    try:
                        ev.wait()
                    except BaseException:
                        if not ei:
                            ei = sys.exc_info()
                        continue
                    else:
                        break

            if ei:
                raise ei[0], ei[1], ei[2]
