import time


class PeerAddress(object):
    __slots__ = ('ip', 'port', 'weight', 'connect_attempts', 'connect_succeeded', 'is_dfs', 'is_local')

    def __init__(self, ip, port):
        self.ip = ip
        self.port = port
        self.weight = 0
        self.connect_attempts = 0
        self.connect_succeeded = False
        self.is_dfs = False
        self.is_local = False

    def __repr__(self):
        return '<PeerAddress ip=%s port=%s is_local=%r>' % (self.ip, self.port, self.is_local)


class Peer(object):
    """
    Peer object
    """

    __slots__ = (
        'uid',          # unique id
        'desc',         # description from peer
        'state',        # peer status
        'state_ts',     # peer status change timestamp
        'worker',
        'conn',         # active peer connection (PeerConnection)
        'weight',       # ???
        'ou_address',   # outgoing address
        'is_local',
    )

    # Generic states
    CONNECTING = 'connecting'
    CONNECTED = 'connected'
    DISCONNECTED = 'disconnected'

    # Behaviour
    CAP_MISMATCH = 'cap_mismatch'  # either peer doesn't support a capability we require,
                                   # or we don't support some capability required by the peer
    NO_SKYBIT_DATA = 'no_skybit_data'
    NO_RESOURCE = 'no_resource'

    # Temporary
    NO_NEED = 'no_need'  # this one should be marked on seeder-seeder connection
    NO_WANT = 'no_want'  # this mark is the same as disconnected, without forced retry
    NO_SLOT = 'no_slot'

    # Difference between NO_NEED and NO_WANT is quite simple: NO_NEED is a bidirectional. So if we
    # receive NO_NEED message, we will mark peer as NO_NEED. But if we receive NO_WANT message, we will
    # mark peer as disconnected.

    def __init__(self, **kwargs):
        defaults = {
            'state': self.DISCONNECTED,
            'state_ts': 0,
            'worker': None,
            'uid': None,
            'desc': None,
            'conn': None,
            'weight': 0,
            'is_local': False,
        }
        defaults.update(kwargs)

        for key, value in defaults.iteritems():
            setattr(self, key, value)

    def set_state(self, state):
        if state == self.CONNECTED:
            assert self.uid is not None
            assert self.desc is not None

        prev_state = self.state
        if prev_state != state:
            self.state = state
            self.state_ts = int(time.time())

    def __repr__(self):
        return '<Peer [%s] [%-40s] %s>' % (
            (self.uid or 'unknown ')[:8], self.desc or '', self.state
        )
