from .. import monkey_patch as skbn_monkey_patch

skbn_monkey_patch()

import argparse
import msgpack
import os
import struct
import sys
import time
import yaml
import subprocess as subproc

import gevent
from gevent import socket

import api.copier.errors

try:
    from setproctitle import setproctitle
except ImportError:
    setproctitle = None

from ..resource.resource import Resource
from ..resource.hasher import Hasher
from ..utils import Path, human_size, human_time

from ...rpc.client import RPCClientGevent
from ...rpc.errors import ProtocolError

from .cmd_utils import setup_logger, path_locker, master_mon, MasterDisconnect


MSG_PACKER = msgpack.Packer()
MSS_MAGIC_BYTE = '\x1231835478432'

FORKDATA = None


def send(typ, data):
    data_packed = MSG_PACKER.pack((typ, data))
    datalen = len(data_packed)

    sys.stdout.write(struct.pack('!BI', 242, datalen))
    sys.stdout.write(data_packed)
    sys.stdout.flush()


class Progress(object):
    def __init__(self, log, child):
        self.log = log
        self.child = child
        self.last_log = 0

    def proctitle(self, title):
        if setproctitle:
            title = 'skybone-sh [ %s ]' % (title, )
            setproctitle(title)

    def handle(self, typ, *args):
        if typ == 'hash_files':
            if args[1] > 0:
                self.log_hashed_bytes(args[0], args[1])
                self.proctitle(
                    '%2d%% hashing %s (out of %s)' % (
                        (args[0] / float(args[1])) * 100, human_size(args[0]), human_size(args[1])
                    )
                )

                if self.child:
                    send('progress', {'stage': 'hashing', 'hashed_bytes': args[0], 'total_bytes': args[1]})

        elif typ == 'connect_master':
            if not args[0]:
                self.log.debug('Connecting master...')
                self.proctitle('connecting master...')
            else:
                self.log.info('Connected to master')
                self.proctitle('preparing to hash...')

            if self.child:
                send('progress', {'stage': 'subproc_connect_master', 'done': args[0]})

        elif typ == 'started':
            self.proctitle('initializing...')
            if self.child:
                send('progress', {'stage': 'subproc_started'})

    def log_hashed_bytes(self, hashed, total):
        if hashed == total or time.time() - self.last_log > 5:
            self.log.debug(
                'progress: [%2d%%] %s from %s',
                (hashed / float(total)) * 100,
                human_size(hashed), human_size(total)
            )
            self.last_log = time.time()


def create_resource(master_cli, files, log, progress):
    log = log.getChild('create')

    files = [(Path(a), b) for (a, b) in files]

    class RemoteCache(object):
        def store_resource(self, resource):
            job = master_cli.call('cache_store_resource', resource.to_dict())
            job.wait()

        def load_info_from_db(self, items, cache_dict=None):
            from ..resource.file import ResourceItem

            clean_items = []

            for item in items:
                dct = item.to_dict()
                clean_items.append(dct)

            job = master_cli.call('cache_load_info_from_db', clean_items)
            clean_items = job.wait()

            for idx, item in enumerate(clean_items):
                item = ResourceItem.from_dict(item)

                former_item = items[idx]
                if isinstance(item, (ResourceItem.file, ResourceItem.symlink)):
                    assert item.path == former_item.path

                    for key in former_item.__slots__:
                        setattr(former_item, key, getattr(item, key))

    class PathLocker(object):
        def paths(self, paths):
            paths_dict = {}
            for path in paths:
                path_stat = path.stat()
                paths_dict[path.strpath] = (path_stat.ino, path_stat.dev)

            return path_locker(master_cli, paths_dict)

    remote_cache = RemoteCache()

    hasher = Hasher(
        None,
        max_parallel=4
    )

    time_start = time.time()
    # 2nd arg -- db -- is not used
    resource, stats = Resource.create(
        files, None, remote_cache, hasher, PathLocker(), log, progress=progress.handle
    )
    time_spent = time.time() - time_start

    log.info(
        'Generated resource with %s data in %s '
        '(files: %d, dirs: %d, symlinks: %d)' % (
            human_size(stats['counts']['size']),
            human_time(time_spent),
            stats['counts']['file'],
            stats['counts']['directory'],
            stats['counts']['symlink'],
        )
    )

    return resource


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('--child', action='store_true')
    parser.add_argument('--master-uds', required=True)
    parser.add_argument('--tries', default=0, type=int)
    parser.add_argument('--extra')

    return parser.parse_args()


def _main(args):
    global FORKDATA

    args = parse_args()

    from library.config import query
    try:
        base_extra = query('skynet.services.copier').opts.create
    except (RuntimeError, AttributeError):
        base_extra = {}

    if args.extra:
        args.extra = yaml.safe_load(args.extra)
    else:
        args.extra = {}

    args_from_base = set(base_extra.keys())
    args_from_args = set(args.extra.keys())

    base_extra.update(args.extra)
    args.extra = base_extra

    opt_subproc = args.extra.get('subproc', False)

    root_log = setup_logger('create')
    log = root_log.getChild('main')

    log.debug('Started with args:')
    for arg in sorted(dir(args)):
        if arg.startswith('_'):
            continue
        if arg == 'resid':
            resid = getattr(args, arg)
            log.debug('  %s: %s%s', arg, resid[:8], '?' * 32)
        elif arg == 'extra':
            log.debug('  extra:')
            for exarg, exvalue in sorted(args.extra.iteritems()):
                if exarg in args_from_args:
                    origin = 'CMD'
                elif exarg in args_from_base:
                    origin = 'CFG'
                else:
                    origin = '???'
                log.debug('    %s: %s: %r', origin, exarg, exvalue)
        else:
            log.debug('  %s: %r', arg, getattr(args, arg))

    progress = Progress(log, args.child)
    progress.handle('started')

    data = FORKDATA = sys.stdin.read()

    params = msgpack.loads(data)

    assert 'files' in params, 'no "files" given'
    assert isinstance(params['files'], tuple), 'invalid "files" type: %r' % (params['files'], )

    files = params['files']

    for fns in files:
        for fn in fns:
            try:
                str(fn)
                unicode(fn)
            except UnicodeDecodeError:
                raise api.copier.errors.CopierError(
                    'Copier does not support files with unicode names! Got %s' % (fn, )
                )

    log.info('Started (%d files), connecting master...', len(files, ))

    try:
        progress.handle('connect_master', False)
        master_cli = RPCClientGevent(args.master_uds, port=None)

        tries = 0
        deadline = time.time() + 60
        while True:
            tries += 1
            try:
                master_cli.connect()
                break
            except (socket.error, ProtocolError) as ex:
                log.debug('master connect: error: %s, tries: %d', ex, tries)
                if time.time() > deadline:
                    raise api.copier.errors.ApiConnectionError(
                        'Unable to connect skybone service: %s' % (ex, )
                    )

                gevent.sleep(min(10, tries * 2))

        progress.handle('connect_master', True)

    except Exception as ex:
        log.error('master connect failed: %s', ex)
        raise

    with master_mon(master_cli, opt_subproc):
        try:
            resource = create_resource(master_cli, params['files'], log, progress)
            send('result', resource.uid)
        except Exception as ex:
            log.exception(ex)
            raise


def retryable_main():
    args = parse_args()

    try:
        return _main(args)
    except MasterDisconnect:
        tries = args.tries + 1
        if tries == 10:
            raise api.copier.errors.ApiConnectionError('Unable to reconnect master skybone service')

        args = [sys.executable] + sys.argv

        found_tries = False
        for idx, arg in enumerate(args):
            if arg == '--tries':
                args[idx + 1] = str(tries)
                found_tries = True

        if not found_tries:
            args.append('--tries')
            args.append(str(tries))

        if FORKDATA:
            rpipe, wpipe = os.pipe()
            os.dup2(rpipe, sys.stdin.fileno())

            forkpid = os.fork()
            if not forkpid:
                os.write(wpipe, FORKDATA)
                os._exit(0)

            os.close(wpipe)

        os.closerange(3, subproc.MAXFD)
        os.execve(sys.executable, args, os.environ)


def main():
    try:
        # We need to make main greenlet gevent.Greenlet class (so, master_mon is able to kill it)
        return gevent.spawn(retryable_main).get()
    except Exception as ex:
        info = ' {%d:%d}' % (os.getppid(), os.getpid())

        try:
            send('error', (ex.__module__, ex.__class__.__name__, str(ex) + info))
        except:
            send('error', str(ex) + info)
