from __future__ import absolute_import, division

import contextlib
import gevent
import gevent.socket
try:
    import gevent.coros as coros
except ImportError:
    import gevent.lock as coros
import os
import py

import httplib
import urllib2

from py import std

from . import PROGNAME
from ..kernel_util.console import setProcTitle

from ..common import Path


def getaddrinfo(*args, **kwargs):
    this = getaddrinfo

    meth = kwargs.pop('getaddrinfo', std.socket.getaddrinfo)

    ips = meth(*args, **kwargs)
    anyips, gentime = getattr(this, 'anyips', (None, 0))
    now = std.time.time()

    if now - gentime > 3600:
        try:
            anyips = set([ip[4][0] for ip in meth('any.yandex.ru', 0)])
        except std.socket.gaierror as ex:
            if ex.errno in (
                getattr(std.socket, 'EAI_NONAME', None),
                getattr(std.socket, 'EAI_NODATA', None),
            ):
                # any.yandex.ru was not found
                anyips = set()
            else:
                raise
        this.anyips = (anyips, now)

    for ipinfo in ips:
        ip = ipinfo[4][0]
        if ip in anyips:
            raise std.socket.gaierror(std.socket.EAI_NONAME, 'Name or service not known (resolved to any.yandex.ru)')

    return ips


getaddrinfo_g_resolve_cache = {}  # host args => (result, result_complete, ts)
getaddrinfo_g_resolve_lock = {}   # host args => lock


def getaddrinfo_g(*args, **kwargs):
    lock = getaddrinfo_g_resolve_lock.get(args, None)

    if not lock:
        lock = getaddrinfo_g_resolve_lock[args] = coros.Semaphore()

    with lock:
        try:
            rpipe, wpipe = std.os.pipe()

            result_q = []

            def _runner(*args, **kwargs):
                try:
                    result_q.append((True, getaddrinfo(*args, **kwargs)))
                except:
                    result_q.append((False, std.sys.exc_info()))
                finally:
                    std.os.write(wpipe, '1')

            cache, cache_complete, cache_ts = getaddrinfo_g_resolve_cache.get(args, (None, None, None))

            if not cache or std.time.time() - cache_ts > 600:
                thr = std.threading.Thread(target=_runner, args=args, kwargs=kwargs)
                thr.daemon = True
                thr.start()

                gevent.socket.wait_read(rpipe)
                assert std.os.read(rpipe, 1) == '1'

                complete, result = result_q[0]

                getaddrinfo_g_resolve_cache[args] = result, complete, int(std.time.time())

                # Cleanup old items in cache
                for args_key in getaddrinfo_g_resolve_cache.keys():
                    if std.time.time() - getaddrinfo_g_resolve_cache[args_key][2] > 600:
                        getaddrinfo_g_resolve_cache.pop(args_key, None)
            else:
                result, complete = cache, cache_complete

            if complete:
                return result
            else:
                raise result[0], result[1], result[2]
        finally:
            std.os.close(rpipe)
            std.os.close(wpipe)


def gethostbyaddr_g(*args, **kwargs):
    meth = kwargs.pop('gethostbyaddr', std.socket.gethostbyaddr)

    try:
        rpipe, wpipe = std.os.pipe()

        result_q = []

        def _runner(*args, **kwargs):
            try:
                result_q.append((True, meth(*args, **kwargs)))
            except:
                result_q.append((False, std.sys.exc_info()))
            finally:
                std.os.write(wpipe, '1')

        thr = std.threading.Thread(target=_runner, args=args, kwargs=kwargs, name='gethostbyaddr')
        thr.daemon = True
        thr.start()

        gevent.socket.wait_read(rpipe)
        assert std.os.read(rpipe, 1) == '1'

        complete, result = result_q[0]

        if complete:
            return result
        else:
            raise result[0], result[1], result[2]
    finally:
        std.os.close(rpipe)
        std.os.close(wpipe)


def fastbonize_ip(ip, log):
    """
    Returns: hostname, fastbone hostname, [[ipv4 fb ips], [ipv6 fb ips]]
    """

    import _socket

    try:
        hostname = gethostbyaddr_g(ip, gethostbyaddr=_socket.gethostbyaddr)[0]
    except _socket.error as ex:
        log.warning('Unable to grab hostname: %s', ex)
        return None, None, None

    log.debug('Resolved hostname %s', hostname)

    fb_hostname, fb_ips = fastbonize_hostname(hostname, log)
    return hostname, fb_hostname, fb_ips


def fastbonize_hostname(hostname, log):
    import _socket

    fb_ips = [[], []]  # ipv4, ipv6

    log.debug('fastbonize_hostname: %s', hostname)

    for variant in 1, 2, 3:
        if variant == 1:
            fb_hostname = 'fb-' + hostname
        elif variant == 2:
            fb_hostname = 'fastbone.' + hostname
        else:
            if hostname.endswith('.yandex.ru'):
                fb_hostname = hostname[:-len('.yandex.ru')] + '.fb.yandex.ru'
            elif '.' not in hostname:
                fb_hostname = hostname + '.fb.yandex.ru'
            else:
                continue

        try:
            log.debug('  resolving %s...', fb_hostname)
            ips = getaddrinfo_g(fb_hostname, 0, 0, gevent.socket.SOCK_STREAM, getaddrinfo=_socket.getaddrinfo)

            for family, _, _, _, ipinfo in ips:
                if family == gevent.socket.AF_INET:
                    fb_ips[0].append(ipinfo[0])
                    log.debug('    detected ipv4 fastbone: %s', ipinfo[0])
                elif family == gevent.socket.AF_INET6:
                    fb_ips[1].append(ipinfo[0])
                    log.debug('    detected ipv6 fastbone: %s', ipinfo[0])

            if variant == 2:
                # fastbone.* may be the same as backbone address, so we resolve
                # backbone here and recheck if they are equal. See SKYDEV-459.
                bb_ips = []
                try:
                    log.debug('    resolving bb ips %s...', hostname)
                    bb_ips_raw = getaddrinfo_g(
                        hostname, 0, 0, gevent.socket.SOCK_STREAM, getaddrinfo=_socket.getaddrinfo
                    )

                    for family, _, _, _, ipinfo in bb_ips_raw:
                        log.debug('      detected ip: %s', ipinfo[0])
                        bb_ips.append(ipinfo[0])
                except gevent.socket.gaierror as ex:
                    if ex.errno in (
                        getattr(gevent.socket, 'EAI_NONAME', None),
                        getattr(gevent.socket, 'EAI_NODATA', None)
                    ):
                        log.debug('      %s not found (errno: %d)', hostname, ex.errno)

                bb_same_as_fb = False
                for ip_by_family in fb_ips:
                    for ip in ip_by_family:
                        if ip in bb_ips:
                            bb_same_as_fb = True
                            break
                    if bb_same_as_fb:
                        break

                if bb_same_as_fb:
                    log.debug('    detected fastbone address matches backbone address, ignoring')
                    continue

            log.info(
                '  detected fastbone address %s with ipv4 ips %s and ipv6 ips %s',
                fb_hostname, fb_ips[0], fb_ips[1]
            )
            return fb_hostname, fb_ips

        except gevent.socket.gaierror as ex:
            if ex.errno in (
                getattr(gevent.socket, 'EAI_NONAME', None),
                getattr(gevent.socket, 'EAI_NODATA', None),
            ):
                log.debug('    %s not found (errno: %d)', fb_hostname, ex.errno)
                continue
            else:
                raise

    log.info('  fastbone address not found')
    return None, None


class SlottedDict(dict):
    def __getattr__(self, key):
        value = self[key]
        if isinstance(value, dict):
            value = self[key] = type(self)(value)
        self.__dict__[key] = value
        return value

    def __setattr__(self, key, value):
        self[key] = self.__dict__[key] = value


@contextlib.contextmanager
def timer():
    class TimerResult(object):
        __slots__ = 'spent',

    ts = std.time.time()
    result = TimerResult()

    try:
        yield result
    finally:
        result.spent = std.time.time() - ts


@contextlib.contextmanager
def dummy_timer():
    class DummyResult(object):
        __slots__ = 'spent',
        spent = 0

    yield DummyResult()


def human_size_sep(b):
    kb = 1024
    mb = kb * 1024
    gb = mb * 1024
    tb = gb * 1024

    def mkfl(fl, k):
        return ('%%.%df' % (k, )) % (fl, )

    if b >= tb * 100:  # 123 TiB
        return str(int(b / tb)), 'TiB'
    if b >= tb * 10:   # 12.23 TiB
        return mkfl(b / tb, 2), 'TiB'
    if b >= tb:        # 1.12 TiB
        return mkfl(b / tb, 2), 'TiB'
    if b >= gb * 100:  # 123 GiB
        return str(int(b / gb)), 'GiB'
    if b >= gb * 10:   # 12.23 GiB
        return mkfl(b / gb, 2), 'GiB'
    if b >= gb:        # 1.123 GiB
        return mkfl(b / gb, 2), 'GiB'
    if b >= mb:        # 123.23 MiB
        return mkfl(b / mb, 2), 'MiB'
    if b >= kb:        # 123.23 KiB
        return mkfl(b / kb, 2), 'KiB'
    if b > 0:          # 123 B
        return str(b), 'B'
    else:
        return '0', ''


def human_size(b):
    return ''.join(human_size_sep(b))


def human_time(s):
    if s < 1:
        return '%dms' % (s * 1000, )

    days, hours = divmod(s, 86400)
    hours, minutes = divmod(hours, 3600)
    minutes, seconds = divmod(minutes, 60)

    if not days and not hours and not minutes:
        return '%ds' % (seconds, )
    elif not days and not hours:
        return '%dm%.2ds' % (minutes, seconds)
    elif not days:
        return '%dh%.2dm%.2ds' % (hours, minutes, seconds)
    else:
        return '%dd%.2dh%.2dm%.2ds' % (days, hours, minutes, seconds)


def human_speed_sep(b):
    bps = b * 8
    mib = 10 ** 6
    kib = 10 ** 3

    if bps >= mib:
        return str(int(bps // mib)), 'Mbps'
    if bps >= kib:
        return str(int(bps // kib)), 'kbps'
    if bps >= 10:
        return '%.2f' % (bps / kib, ), 'kbps'
    else:
        return '0', ''


def human_speed(b):
    return ''.join(human_speed_sep(b))


def has_root():
    if std.os.uname()[0].lower() == 'darwin' or std.os.uname()[0].lower().startswith('cygwin'):
        return std.os.getuid() == 0
    else:
        return std.os.getresuid()[2] == 0


def update_proc_title(cfg, ul, dl):
    proctitle = '%s [%d/%d] [ul:%0.2fmbps dl:%0.2fmbps]'
    setProcTitle(proctitle % (
        PROGNAME, cfg.rpc_port, cfg.proxy.port,
        ul, dl
    ))


def gen_uds(name, fname, workdir, allow_abstract=True):
    """
    name -- name system wide
    fname -- name inside workdir
    """

    if os.uname()[0].lower() == 'linux' and allow_abstract:
        return name, True
    else:
        uds = workdir.join(fname + '.sock')
        if len(uds.strpath) >= 108:
            uds = Path('/tmp').join(name + '.sock')

        try:
            uds.remove()
        except py.error.ENOENT:
            pass

        return uds.strpath, False


class GeventHTTPConnection(httplib.HTTPConnection):
    def connect(self):
        from gevent import socket
        self.sock = socket.create_connection((self.host, self.port), self.timeout)


class GeventHTTPHandler(urllib2.HTTPHandler):
    def http_open(self, request):
        return self.do_open(GeventHTTPConnection, request)


def gevent_urlopen(url, data=None, headers={}, timeout=None):
    opener = urllib2.build_opener(GeventHTTPHandler)
    # headers:
    # primarily used for Range headers as of 22 Oct 2021, this may be used for any request header in the future
    # as an example, {'Range': 'bytes=-100,300-599,700'} 
    # it is up to caller to format headers in a right way - no verification is done
    req = urllib2.Request(url, data=data, headers=headers)
    return opener.open(req, timeout=timeout)
