from __future__ import absolute_import

import json
import functools
import socket
import os

import flask
from pymongo import MongoClient

from skybone_yt.model import MongoStorage
from skybone_yt import cypress
import skybone_yt.args
import skybone_yt.share


def nodeinfo_headers_wrapper(node_info, wsgi_app):
    node_info = [
        ('X-SkyboneYt-{}'.format(key.replace('_', '-').title()), str(value))
        for key, value in node_info.items()
    ]

    @functools.wraps(wsgi_app)
    def wrapped(environ, start_response):
        def injecting_headers_start_response(status, response_headers,
                                             exc_info=None):
            response_headers.extend(node_info)
            return start_response(status, response_headers, exc_info)
        return wsgi_app(environ, injecting_headers_start_response)
    return wrapped


def make_app(config):
    app = flask.Flask('skybone_yt.api')

    app.config.from_object('skybone_yt.config')
    app.config.update(config)

    mongoclient = MongoClient(app.config['MONGODB_URI'])
    app.storage = MongoStorage(mongoclient, app.config['MONGODB_DB_NAME'])

    # health check
    app.add_url_rule('/ping', view_func=ping, methods=['GET'])

    app.add_url_rule('/share/<proxy>', view_func=share, methods=['POST'])
    app.add_url_rule('/resolve/<file_id>.json',
                     view_func=resolve, methods=['GET'])
    app.add_url_rule('/get/<file_id>', view_func=get, methods=['GET'])

    node_info = {'hostname': socket.gethostname(),
                 'pid': os.getpid(),
                 'address': app.config['WSGI_LISTEN_ADDRESS']}
    node_info_str = json.dumps(node_info, sort_keys=True)

    @app.errorhandler(Exception)
    def server_error(error):
        app.logger.exception(error)
        text = "Internal server error: %s\nNode info: %s" % (error,
                                                             node_info_str)
        return text, 500

    app.wsgi_app = nodeinfo_headers_wrapper(node_info, app.wsgi_app)

    return app


def ping():
    return flask.current_app.response_class(status=200)


def share(proxy):
    paths = flask.request.values.getlist('path')
    paths = [path.encode('ascii') for path in paths]
    rbtorrent_id = skybone_yt.share.share(
        flask.current_app.config, flask.current_app.storage, proxy, paths
    )
    return rbtorrent_id


def get(file_id):
    storage = flask.current_app.storage
    file_info = storage.get_file_info(file_id)
    if file_info is None:
        flask.abort(404)
    client = cypress.make_client(file_info['proxy'],
                                 flask.current_app.config['YT_ACCESS_TOKEN'])
    file_exists = cypress.check_file_exists(client, file_info['path'])
    if not file_exists:
        skybone_yt.share.remove(flask.current_app.config, storage,
                                file_info['proxy'], file_info['path'])
        flask.abort(404)
    storage.update_file_last_checked(file_info['proxy'], file_info['path'])
    chunks = cypress.read_file_by_chunks(client, file_info['path'])
    return flask.Response(chunks)


def resolve(file_id):
    return json.dumps([flask.url_for('get', file_id=file_id, _external=True)])


def main():
    parser = skybone_yt.args.make_parser()
    parser.add_argument('-H', '--listen-address', default='0.0.0.0',
                        help='listen address (defaults to 0.0.0.0)')
    parser.add_argument('-P', '--listen-port', default=7654, type=int,
                        help='listen port (defaults to 7654)')
    config, args = skybone_yt.args.parse_args(parser)
    config['WSGI_LISTEN_ADDRESS'] = (args.listen_address, args.listen_port)
    app = make_app(config)
    app.run(*app.config['WSGI_LISTEN_ADDRESS'], threaded=True)


if __name__ == '__main__':
    main()
