import os
import hashlib

import skybone_yt.bencode as bencode


def make_torrent(data, sha1hashes):
    return {
        'name': 'data',
        'piece length': 4 * 1024 * 1024,
        'pieces': sha1hashes,
        'length': data['size']
    }


def sanitize_structure(files, checksums):
    directories = {}
    structure = {}
    torrents = {}

    for name, data in sorted(files.iteritems()):
        if os.sep in name:
            dirs, filename = name.rsplit(os.sep, 1)
            dirs = dirs.split('/')
        else:
            dirs = []

        current_dir = None
        for dirname in dirs:
            if current_dir:
                current_dir = os.path.join(current_dir, dirname)
            else:
                current_dir = dirname

            if current_dir not in directories:
                dirinfo = {
                    'type': 'dir',
                    'path': current_dir,
                    'size': -1,
                    'executable': 1,
                    'mode': 'rwxr-xr-x',
                    'resource': {
                        'type': 'mkdir',
                        'data': 'mkdir',
                        'id': 'mkdir:%s' % (current_dir, )
                    }
                }

                directories[current_dir] = dirinfo
                structure[current_dir] = dirinfo

        structure[name] = {
            'type': 'file',
            'path': name,
            'md5sum': data['md5'].decode('hex'),
            'size': data['size'],
            'executable': 1 if data['executable'] else 0,
            'mode': 'rwxr-xr-x' if data['executable'] else 'rw-r--r--'
        }

        if data['size'] > 0:
            torrent_info = make_torrent(data, checksums[data['md5']])
            infohash = hashlib.sha1(bencode.bencode(torrent_info)).digest()
            structure[name]['resource'] = {'type': 'torrent',
                                           'id': infohash.encode('hex')}
            torrents[infohash.encode('hex')] = {'info': torrent_info}
        else:
            structure[name]['resource'] = {'type': 'touch'}

    return structure, torrents


def generate_resource(files, hashes):
    """
    files: dictionary with files, names and md5 checksums

    {
        'pathA/file': {
            'md5': <32-byte hex md5 checksum>,
            'executable': <True|False>,
            'size': <size in bytes>,
        }
    }

    hashes: dictonary with sha1 hashsums by md5
    {
        <32-bytes hex md5 checksum>: <concantenated sha1 RAW hashsums
                                      (20 bytes * (size / 4MB))>
    }
    """

    structure, torrents = sanitize_structure(files, hashes)
    head = {
        'structure': structure,
        'torrents': torrents,
    }

    head_binary = bencode.bencode(head)

    piece_size = 4 * 1024 * 1024
    pieces = []

    num_pieces = len(head_binary) // piece_size
    if len(head_binary) % piece_size:
        num_pieces += 1

    pieces = []
    for idx in range(num_pieces):
        start = idx * piece_size
        end = start + piece_size

        data = head_binary[start:end]
        pieces.append(hashlib.sha1(data).digest())

    head_ti = {
        'name': 'metadata',
        'piece length': piece_size,
        'pieces': ''.join(pieces),
        'length': len(head_binary)
    }

    return (
        'rbtorrent:%s' % hashlib.sha1(bencode.bencode(head_ti)).hexdigest(),
        head_binary
    )
