import hashlib
import json
import os
import random
import socket
import sys

import requests

from .bencode import bencode


PY3 = sys.version_info.major > 2


class ServiceUnavailable(RuntimeError):
    pass


def _make_torrent(data, sha1hashes):
    return {
        'name': 'data',
        'piece length': 4 * 1024 * 1024,
        'pieces': sha1hashes,
        'length': data['size']
    }


def _sanitize_structure(items, checksums):
    directories = {}
    structure = {}
    torrents = {}

    for name, data in sorted(items.items()):
        datatype = data.get('type', 'file')

        if os.sep in name:
            dirs, _ = name.rsplit(os.sep, 1)
            dirs = dirs.split('/')
        else:
            dirs = []

        if datatype == 'dir':
            # If this is empty dir -- add it itself
            dirs.append(name.rsplit(os.sep, 1)[1] if os.sep in name else name)

        current_dir = None
        for dirname in dirs:
            if current_dir:
                current_dir = os.path.join(current_dir, dirname)
            else:
                current_dir = dirname

            if current_dir not in directories:
                dirinfo = {
                    'type': 'dir',
                    'path': current_dir,
                    'size': -1,
                    'executable': 1,
                    'mode': 'rwxr-xr-x',
                    'resource': {
                        'type': 'mkdir',
                        'data': 'mkdir',
                        'id': 'mkdir:%s' % (current_dir, )
                    }
                }

                directories[current_dir] = dirinfo
                structure[current_dir] = dirinfo

        if datatype == 'dir':
            continue

        if datatype == 'file':
            if PY3:
                md5sum_bytes = bytes.fromhex(data['md5'])
            else:
                md5sum_bytes = data['md5'].decode('hex')

            structure[name] = {
                'type': 'file',
                'path': name,
                'md5sum': md5sum_bytes,
                'size': data['size'],
                'executable': 1 if data['executable'] else 0,
                'mode': 'rwxr-xr-x' if data['executable'] else 'rw-r--r--'
            }

            if data['size'] > 0:
                torrent_info = _make_torrent(data, checksums[data['md5']])
                infohash = hashlib.sha1(bencode(torrent_info)).digest()
                if PY3:
                    infohash_hex = infohash.hex()
                else:
                    infohash_hex = infohash.encode('hex')
                structure[name]['resource'] = {'type': 'torrent', 'id': infohash_hex}
                torrents[infohash_hex] = {'info': torrent_info}
            else:
                structure[name]['resource'] = {'type': 'touch'}

        if datatype == 'symlink':
            structure[name] = {
                'type': 'file',
                'path': name,
                'size': 0,
                'mode': '',
                'executable': 1,
                'resource': {
                    'type': 'symlink',
                    'data': 'symlink:%s' % (data['target'], )
                }
            }

    return structure, torrents


def skyboned_generate_resource(items, hashes):
    structure, torrents = _sanitize_structure(items, hashes)
    head = {
        'structure': structure,
        'torrents': torrents,
    }

    head_binary = bencode(head)

    piece_size = 4 * 1024 * 1024
    pieces = []

    num_pieces = len(head_binary) // piece_size
    if len(head_binary) % piece_size:
        num_pieces += 1

    pieces = []
    for idx in range(num_pieces):
        start = idx * piece_size
        end = start + piece_size

        data = head_binary[start:end]
        pieces.append(hashlib.sha1(data).digest())

    head_ti = {
        'name': 'metadata',
        'piece length': piece_size,
        'pieces': b''.join(pieces),
        'length': len(head_binary)
    }

    return (
        hashlib.sha1(bencode(head_ti)).hexdigest(),  # rbtorrent:
        head_binary
    )


def skyboned_add_resource(items, hashes, links, servers, tvm_ticket, source_id=None, no_wait=False):
    """
    items: dictionary with files/symlinks/empty_dirs and md5 checksums
    {
        'pathA/file': {
            'type': 'file',        # not required, each item treated as file by default
            'md5': <32-byte hex md5 checksum>,
            'executable': <True|False>,
            'size': <size in bytes>,
        },
        'pathB/symlink': {
            'type': 'symlink',
            'target': '../symlink/target'
        },
        'pathX/empty/directory': { # not required for directories with links/files -- they will be added automatically
            'type': 'dir'
        }
    }

    hashes: dictonary with sha1 hashsums by md5
    {
        <32-bytes hex md5 checksum>: <concantenated sha1 RAW hashsums (20 bytes * (size / 4MB))>
    }

    links: file links by md5 checksum
    {
        '<32-byte hex md5 checksum>': [http://link1, ...]
    }

    servers: list of skyboned api servers to use
    """
    assert isinstance(servers, (list, tuple))
    assert len(servers) >= 1

    uid, head = skyboned_generate_resource(items, hashes)

    # Sanity check
    for path, pathinfo in items.items():
        if pathinfo.get('type', 'file') == 'file' and pathinfo.get('size', 0) > 0:
            assert pathinfo['md5'] in links, 'Some md5 file links are missing'

    if PY3:
        head_hex = head.hex()
    else:
        head_hex = head.encode('hex')

    payload = {
        'uid': uid,
        'head': head_hex,
        'info': links,
        'mode': 'plain'
    }
    if source_id:
        payload['source_id'] = source_id
    payload['no_wait'] = no_wait

    data = json.dumps(payload)

    for idx, server in enumerate(random.sample(servers, len(servers))):
        final = idx == len(servers) - 1

        orig_gai = requests.packages.urllib3.util.connection.allowed_gai_family

        try:
            # Force requests to use only IPv6
            requests.packages.urllib3.util.connection.allowed_gai_family = lambda: socket.AF_INET6
            response = requests.post(
                'http://%s/add_resource' % (server, ),
                data=data,
                headers={
                    'X-Ya-Service-Ticket': tvm_ticket,
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                }
            )

            if response.status_code == 503:
                # If we get 503, raise error immidiately without retrying other servers
                # It is adviced to retry in a big interval (like 30s) again.
                raise ServiceUnavailable('503 Service Unavailable')

            if response.status_code in (400, 403):
                final = True
                response.raise_for_status()

            assert response.status_code == 200
            return 'rbtorrent:{}'.format(uid), True
        except Exception:
            if final:
                raise
        finally:
            requests.packages.urllib3.util.connection.allowed_gai_family = orig_gai


def skyboned_remove_resource(uid, servers, tvm_ticket, source_id=None):
    assert isinstance(servers, (list, tuple))
    assert len(servers) >= 1

    for idx, server in enumerate(random.sample(servers, len(servers))):
        final = idx == len(servers) - 1

        payload = {'uid': uid}
        if source_id:
            payload['source_id'] = source_id

        data = json.dumps(payload)

        orig_gai = requests.packages.urllib3.util.connection.allowed_gai_family

        try:
            # Force requests to use only IPv6
            requests.packages.urllib3.util.connection.allowed_gai_family = lambda: socket.AF_INET6
            response = requests.post(
                'http://%s/remove_resource' % (server, ),
                data=data,
                headers={
                    'X-Ya-Service-Ticket': tvm_ticket,
                    'Content-Type': 'application/json',
                    'Accept': 'application/json'
                }
            )

            if response.status_code == 503:
                # If we get 503, raise error immidiately without retrying other servers
                # It is adviced to retry in a big interval (like 30s) again.
                raise ServiceUnavailable('503 Service Unavailable')

            if response.status_code in (400, 403):
                final = True
                response.raise_for_status()

            assert response.status_code == 200
            return True
        except Exception:
            if final:
                raise
        finally:
            requests.packages.urllib3.util.connection.allowed_gai_family = orig_gai

    return False
