import os
import hashlib
import argparse
import shutil

from infra.skyboned.api import skyboned_add_resource, skyboned_remove_resource


def share_resource():
    # example usage
    dirname = os.path.realpath('example_data')

    if os.path.exists(dirname):
        shutil.rmtree(dirname)
    os.makedirs(dirname)

    try:
        f1 = os.path.join(dirname, 'file1')
        open(f1, 'w').write('some content')

        f2 = os.path.join(dirname, 'path', 'to', 'deep', 'file2')
        os.makedirs(f2.rsplit(os.sep, 1)[0])

        open(f2, 'w').write('some big and nice content' * 1024 * 1024)  # about 26mb of data

        f1md5 = hashlib.md5(open(f1, 'rb').read())
        f2md5 = hashlib.md5(open(f2, 'rb').read())

        f1sha1 = hashlib.sha1(open(f1, 'rb').read()).digest()

        f2sha1 = b''

        piece_size = 4 * 1024 * 1024

        os.makedirs(os.path.join(dirname, 'empty/dir'))
        os.symlink('path/to/deep/file2', os.path.join(dirname, 'symlink'))

        with open(f2, 'rb') as fp:
            while True:
                data = fp.read(piece_size)
                if len(data) == 0:
                    break
                f2sha1 += hashlib.sha1(data).digest()
                if len(data) < piece_size:
                    break

        resource_id, added = skyboned_add_resource(
            {
                'file1': {
                    'md5': f1md5.hexdigest(),
                    'executable': False,
                    'size': os.path.getsize(f1),
                },
                'path/to/deep/file2': {
                    'md5': f2md5.hexdigest(),
                    'executable': False,
                    'size': os.path.getsize(f2),
                },
                'file3': {
                    'md5': hashlib.md5(b'').hexdigest(),
                    'executable': True,
                    'size': 0
                },
                'empty/dir': {
                    'type': 'dir'
                },
                'symlink': {
                    'type': 'symlink',
                    'target': 'path/to/deep/file2'
                }
            },
            {
                f1md5.hexdigest(): f1sha1,
                f2md5.hexdigest(): f2sha1,
            },
            {
                f1md5.hexdigest(): ['http://[2a02:6b8:b000:66a:225:90ff:fe95:8362]:8080/file1'],
                f2md5.hexdigest(): ['http://[2a02:6b8:b000:66a:225:90ff:fe95:8362]:8080/path/to/deep/file2']
            },
            (
                'sas1-skybonecoord1.search.yandex.net:80',
                'vla1-skybonecoord1.search.yandex.net:80',
                'man1-skybonecoord1.search.yandex.net:80'
            ),
            '<tvmticket>'
        )

        assert resource_id == 'rbtorrent:0b6289a0125e9a21a82188adede39568f625528a', (
            'Got %s which is invalid' % (resource_id, )
        )
        assert added

        print('added resource', resource_id)
    finally:
        pass  # shutil.rmtree(dirname)


def remove_resource():
    removed = skyboned_remove_resource(
        'rbtorrent:0b6289a0125e9a21a82188adede39568f625528a',
        (
            'sas1-skybonecoord1.search.yandex.net:80',
            'vla1-skybonecoord1.search.yandex.net:80',
            'man1-skybonecoord1.search.yandex.net:80'
        ),
        '<tvmticket>'
    )

    if removed:
        print('resource removed completely')
    else:
        print('resource was not removed, something went wrong')


if __name__ == '__main__':
    parser = argparse.ArgumentParser()
    parser.add_argument('op', choices=('add', 'remove'))

    args = parser.parse_args()

    if args.op == 'add':
        share_resource()
    elif args.op == 'remove':
        remove_resource()
