#!/usr/bin/env python3

import argparse
import logging
import os
import sys

import gevent
import yaml

from infra.skyboned.src.announcer import Announcer
from infra.skyboned.src.db import Database, add_db_arguments
from infra.skyboned.src.deblock import Deblock


def parse_args():
    parser = argparse.ArgumentParser()
    add_db_arguments(parser)
    parser.add_argument('--udp-port', type=int, required=True)
    parser.add_argument('--data-port', type=int, required=True)
    parser.add_argument('--data-hostnames')
    parser.add_argument('-c', '--config')
    parser.add_argument('--cfmt', choices=('skycore', ), default='skycore')

    args = parser.parse_args()

    for attr in dir(args):
        if attr.startswith('_'):
            continue

        value = getattr(args, attr)

        if isinstance(value, str) and value.startswith('env:'):
            setattr(args, attr, os.environ.get(value.split(':', 1)[1]))

        if isinstance(value, str) and value.startswith('file:'):
            setattr(args, attr, open(value.split(':', 1)[1], 'r').read().strip())

    return args


def get_config(cfgfile, cfgfmt):
    assert cfgfmt == 'skycore', 'Only skycore-like config files supported as of now'

    cfg = yaml.load(open(cfgfile, 'r'), Loader=yaml.FullLoader)

    skey = 'subsections'

    return cfg[skey]['skynet'][skey]['services'][skey]['skyboned'][skey]['announcer']['config']


def get_data_ip(hostname):
    info = gevent.socket.getaddrinfo(hostname, 0, gevent.socket.AF_INET6, gevent.socket.SOCK_STREAM)
    ip = info[0][4][0]
    return ip


def data_ip_watchdog(hostname, ip, interval):
    while True:
        try:
            new_ip = get_data_ip(hostname)
            if new_ip != ip:
                os._exit(1)
        except Exception:
            pass

        gevent.sleep(interval)


def main():
    assert gevent.spawn(lambda: True).get(), 'Greenlets are not working correctly!'

    args = parse_args()
    cfg = get_config(args.config, args.cfmt)

    logging.basicConfig(
        level=logging.DEBUG,
        format='%(asctime)s %(levelname)-7s [%(name)-24s]  %(message)s'
    )

    log = logging.getLogger('')

    db_host = None
    if 'db_host' in cfg:
        db_host = cfg['db_host']
    elif args.db_host:
        db_host = args.db_host
    assert db_host, 'Db host is not set!'

    dbobj = Database(
        host=db_host,
        port=args.db_port,
        user=args.db_user,
        password=args.db_password,
        dbname='skyboned'
    )
    dbnewobj = Database(
        host=db_host,
        port=args.db_port,
        user=args.db_user,
        password=args.db_password,
        dbname='skyboned'
    )
    dbtobj = Database(
        host=db_host,
        port=args.db_port,
        user=args.db_user,
        password=args.db_password,
        dbname='skyboned'
    )

    db_deblock = Deblock(keepalive=None, logger=dbobj.log.getChild('deblock'))
    db = db_deblock.make_proxy(dbobj, put_deblock='deblock')

    dbnew_deblock = Deblock(keepalive=None, logger=dbnewobj.log.getChild('deblock'))
    dbnew = dbnew_deblock.make_proxy(dbnewobj, put_deblock='deblock')

    dbt_deblock = Deblock(keepalive=None, logger=dbtobj.log.getChild('deblock'))
    dbt = dbt_deblock.make_proxy(dbtobj, put_deblock='deblock')

    dbt.connect(autocommit=False)
    # dbt.migrate()  run migrations only in skybone-skybit service

    db.connect(autocommit=True)
    dbnew.connect(autocommit=True)

    uid = db.query_one_col('SELECT value_text FROM dbmaintain WHERE key = %s', ('uid', ))
    log.info('Using UID: %r', uid)

    tracker_hostname = cfg['tracker_hostname']
    tracker_ip = get_data_ip(tracker_hostname)
    gevent.spawn(data_ip_watchdog, tracker_hostname, tracker_ip, 1800)

    data_hostnames = None
    if 'data_hostnames' in cfg:
        data_hostnames = cfg['data_hostnames']
    elif args.data_hostnames:
        data_hostnames = args.data_hostnames
    assert data_hostnames, 'Data hostnames are not set!'

    data_ips = []
    data_fb_ips = []

    for data_hostname in data_hostnames.split(','):
        data_ip = get_data_ip(data_hostname)
        data_ips.append(data_ip)
        gevent.spawn(data_ip_watchdog, data_hostname, data_ip, 60)

        data_fb_hostname = 'fb-' + data_hostname
        data_fb_ip = get_data_ip(data_fb_hostname)
        data_fb_ips.append(data_fb_ip)
        gevent.spawn(data_ip_watchdog, data_fb_hostname, data_fb_ip, 60)

    announcer = Announcer(
        uid=uid,
        db=db, dbnew=dbnew, dbt=dbt,
        trackers={
            'coord': {
                tracker_hostname: ((tracker_ip, 2399), )
            },
        },
        ips={
            'bb': None, 'bb6': tuple(data_ips),
            'fb': None, 'fb6': tuple(data_fb_ips)
        },
        port=args.udp_port,
        data_port=args.data_port,
        dfs=True,
        parent=None
    )

    announcer.start()

    # web = Web(db, 'hostname', resource_mngr)
    # web.listen()
    # web.start()

    log.info('running forever')

    try:
        while True:
            gevent.sleep(1)
    except KeyboardInterrupt:
        print('Got KeyboardInterrupt', file=sys.stderr)
        pass


if __name__ == '__main__':
    main()
