#!/usr/bin/env python3

import argparse
import logging
import logging.handlers
import os
import socket
import time

import gevent

from infra.skyboned.src.deblock import Deblock
from infra.skyboned.src.db import Database, add_db_arguments
from infra.skyboned.src.utils import TvmClient
from infra.skyboned.src.web.api import Web


def parse_args():
    parser = argparse.ArgumentParser()
    add_db_arguments(parser)
    parser.add_argument('--http-port', type=int, required=True)
    parser.add_argument('--https-port', type=int, required=True)
    parser.add_argument('--tvm-port', type=int, default=18080)
    parser.add_argument('--tvm-token', required=True)
    parser.add_argument('--log-file', type=str)
    parser.add_argument('--enable-ssl', action='store_true')
    parser.add_argument('--ssl-key')
    parser.add_argument('--ssl-cert')
    parser.add_argument('--sem-cap', type=int, default=50)

    args = parser.parse_args()

    for attr in dir(args):
        if attr.startswith('_'):
            continue

        value = getattr(args, attr)

        if isinstance(value, str) and value.startswith('env:'):
            setattr(args, attr, os.environ.get(value.split(':', 1)[1]))

        if isinstance(value, str) and value.startswith('file:'):
            setattr(args, attr, open(value.split(':', 1)[1], 'r').read().strip())

    return args


def main():
    assert gevent.spawn(lambda: True).get(), 'Greenlets are not working correctly!'

    args = parse_args()
    hostname = socket.gethostname()

    log = logging.getLogger('')
    log.setLevel(logging.DEBUG)

    if args.log_file:
        log_file_size = 1024 * 1024 * 1024
        handler = logging.handlers.RotatingFileHandler(args.log_file, maxBytes=log_file_size, backupCount=9)
    else:
        handler = logging.StreamHandler()
    formatter = logging.Formatter('%(asctime)s %(levelname)-7s [%(name)-24s]  %(message)s')
    handler.setFormatter(formatter)
    log.addHandler(handler)

    dbobj = Database(
        host=args.db_host,
        port=args.db_port,
        user=args.db_user,
        password=args.db_password,
        dbname='skyboned'
    )
    dbroobj = Database(
        host=args.db_host,
        port=args.db_port,
        user=args.db_user,
        password=args.db_password,
        dbname='skyboned'
    )
    dbtobj = Database(
        host=args.db_host,
        port=args.db_port,
        user=args.db_user,
        password=args.db_password,
        dbname='skyboned'
    )
    tracked_db = None
    db_tracked_host = None
    dc = hostname.split('.')[1]
    if dc in ('vla', 'sas', 'man', 'iva', 'myt'):
        for host in args.db_host.split(','):
            if host.startswith(dc):
                db_tracked_host = host
                break
    else:
        log.warning('We are in unknown DC!')

    if args.db_tracked:
        db_tracked_host = args.db_tracked
    if db_tracked_host:
        tracked_dbobj = Database(
            host=db_tracked_host,
            port=args.db_port,
            user=args.db_user,
            password=args.db_password,
            dbname='skyboned'
        )
        tracked_db_deblock = Deblock(keepalive=None, logger=tracked_dbobj.log.getChild('deblock'))
        tracked_db = tracked_db_deblock.make_proxy(tracked_dbobj, put_deblock='deblock')
        tracked_db.connect(autocommit=True, read_write=False)
        log.info("Tracking dbhost: %s in dc: %s", db_tracked_host, dc)

    db_deblock = Deblock(keepalive=None, logger=dbobj.log.getChild('deblock'))
    db = db_deblock.make_proxy(dbobj, put_deblock='deblock')

    db_ro_deblock = Deblock(keepalive=None, logger=dbroobj.log.getChild('deblock'))
    db_ro = db_ro_deblock.make_proxy(dbroobj, put_deblock='deblock')

    dbt_deblock = Deblock(keepalive=None, logger=dbtobj.log.getChild('deblock'))
    dbt = dbt_deblock.make_proxy(dbtobj, put_deblock='deblock')

    dbt.connect(autocommit=False)
    dbt.migrate()

    db_ro.connect(autocommit=True, read_write=False)
    db.connect(autocommit=True)

    tvm_client = TvmClient(args.tvm_port, args.tvm_token)

    web = Web(db, db_ro, dbt, tracked_db, tvm_client, hostname, args.sem_cap)
    web.listen(args.http_port, args.https_port)
    # HACK: wait start of other skyboned-api instances before accepting requests
    # otherwise REUSEPORT doesn't work ¯\_(ツ)_/¯
    time.sleep(5)
    web.start()

    log.info('running forever')

    try:
        while True:
            gevent.sleep(1)
    except KeyboardInterrupt:
        log.info('Got KeyboardInterrupt')
        pass


if __name__ == '__main__':
    main()
