#!/usr/bin/env python3

import argparse
import logging
import logging.handlers
import os
import socket
import gevent
import json

from infra.skyboned.src.deblock import Deblock
from infra.skyboned.src.db import Database, add_db_arguments
from infra.skyboned.src.lookup import Lookup
from infra.skybit.src.world import World

from infra.yasm.unistat import global_unistat
from flask import Flask, jsonify
from gevent import pywsgi


def parse_args():
    parser = argparse.ArgumentParser()
    add_db_arguments(parser)
    parser.add_argument('--data-port', type=int, required=True)
    parser.add_argument('--log-file', type=str)
    parser.add_argument('--unistat-host', type=str, default="::")
    parser.add_argument('--unistat-port', type=int, default=55365)

    args = parser.parse_args()

    for attr in dir(args):
        if attr.startswith('_'):
            continue

        value = getattr(args, attr)

        if isinstance(value, str) and value.startswith('env:'):
            setattr(args, attr, os.environ.get(value.split(':', 1)[1]))

    return args


def main():
    assert gevent.spawn(lambda: True).get(), 'Greenlets are not working correctly!'

    args = parse_args()

    log = logging.getLogger('')
    log.setLevel(logging.DEBUG)

    if args.log_file:
        log_file_size = 512 * 1024 * 1024
        handler = logging.handlers.RotatingFileHandler(args.log_file, maxBytes=log_file_size, backupCount=5)
    else:
        handler = logging.StreamHandler()
    formatter = logging.Formatter('%(asctime)s %(levelname)-7s [%(name)-24s]  %(message)s')
    handler.setFormatter(formatter)
    log.addHandler(handler)

    dbobj = Database(
        host=args.db_host,
        port=args.db_port,
        user=args.db_user,
        password=args.db_password,
        dbname='skyboned'
    )
    dbroobj = Database(
        host=args.db_host,
        port=args.db_port,
        user=args.db_user,
        password=args.db_password,
        dbname='skyboned'
    )
    dbtobj = Database(
        host=args.db_host,
        port=args.db_port,
        user=args.db_user,
        password=args.db_password,
        dbname='skyboned'
    )

    db_deblock = Deblock(keepalive=None, logger=dbobj.log.getChild('deblock'))
    db = db_deblock.make_proxy(dbobj, put_deblock='deblock')

    db_ro_deblock = Deblock(keepalive=None, logger=dbroobj.log.getChild('deblock'))
    db_ro = db_ro_deblock.make_proxy(dbroobj, put_deblock='deblock')

    dbt_deblock = Deblock(keepalive=None, logger=dbtobj.log.getChild('deblock'))
    dbt = dbt_deblock.make_proxy(dbtobj, put_deblock='deblock')

    dbt.connect(autocommit=False)
    dbt.migrate()

    db_ro.connect(autocommit=True, read_write=False)
    db.connect(autocommit=True)

    uid = socket.gethostname()
    log.info('Using UID: %r', uid)

    resource_lookupper = Lookup(db, db_ro, dbt)

    world = World(
        uid=uid,
        desc=uid,
        port=args.data_port,
    )
    world.enable_daemon_mode()
    world.set_lookupper(resource_lookupper)
    world.set_handles_clean_watchdog(600)
    world.start()

    unistat_app = Flask('unistat')
    @unistat_app.route('/unistat', methods=['GET'])
    def unistat():
        return jsonify(json.loads(global_unistat.to_json()))

    server = pywsgi.WSGIServer((args.unistat_host, args.unistat_port), unistat_app)
    log.info('running forever')
    server.serve_forever()


if __name__ == '__main__':
    main()
