package main

import (
	"flag"
	"fmt"
	"log"
	"net"
	"net/url"
	"os"

	"a.yandex-team.ru/library/go/maxprocs"

	"a.yandex-team.ru/infra/skyboned/go/src/auth"
	"a.yandex-team.ru/infra/skyboned/go/src/database"
	"a.yandex-team.ru/infra/skyboned/go/src/skyboned"
	"a.yandex-team.ru/infra/skyboned/go/src/util"

	"go.uber.org/zap"
	"go.uber.org/zap/zapcore"
	"golang.org/x/crypto/ssh/terminal"
	"golang.org/x/net/netutil"
	"gopkg.in/natefinch/lumberjack.v2"
)

var (
	LogPath string
	Debug   bool
)

type lumberjackSink struct{ *lumberjack.Logger }

func (*lumberjackSink) Sync() error { return nil } //zap.Sink requires Sync method to be implemented

func makeZapLogger(debug bool) (*zap.Logger, error) {
	u, err := url.ParseRequestURI(LogPath)
	if err != nil {
		return nil, err
	}

	lumbersink := &lumberjackSink{
		Logger: &lumberjack.Logger{
			Filename:   u.String(),
			MaxSize:    1024, //MB
			MaxBackups: 20,
			LocalTime:  true,
			Compress:   true,
		}}

	encoderCfg := zap.NewProductionEncoderConfig()
	encoderCfg.EncodeTime = zapcore.ISO8601TimeEncoder
	encoderCfg.EncodeLevel = zapcore.CapitalLevelEncoder
	encoderCfg.EncodeCaller = zapcore.ShortCallerEncoder
	encoder := zapcore.NewConsoleEncoder(encoderCfg)
	al := zap.NewAtomicLevelAt(zapcore.InfoLevel)
	if debug {
		al = zap.NewAtomicLevelAt(zapcore.DebugLevel)
	}
	core := zapcore.NewCore(encoder, lumbersink, al)
	if terminal.IsTerminal(int(os.Stdout.Fd())) {
		core = zapcore.NewTee(
			core,
			zapcore.NewCore(encoder, zapcore.Lock(os.Stdout), al))
	}
	return zap.New(core, zap.AddCaller()), nil
}

func main() {
	maxprocs.AdjustAuto()
	flag.StringVar(&LogPath, "log-file", "", "logfile path")

	database.DBFlags()
	auth.TVMFlags()
	skyboned.ServerFlags()
	flag.Parse()

	auth.TVMToken = util.EnvVar(auth.TVMToken)
	database.DBConfGlobal.DBpassword = util.EnvVar(database.DBConfGlobal.DBpassword)

	logger, err := makeZapLogger(Debug)
	if err != nil {
		log.Fatalf("failed to init zap logger: %v", err)
	}
	_ = zap.ReplaceGlobals(logger)

	s, err := skyboned.SetupServer()
	if err == nil {
		l, err := net.Listen("tcp", fmt.Sprintf(":%v", skyboned.HTTPPort))
		if err != nil {
			zap.S().Fatal(err.Error())
		}
		l = netutil.LimitListener(l, skyboned.MaxConnections)
		zap.S().Infof("serving...")
		log.Fatal(s.Serve(l))
	}
}
