package api

import (
	"errors"
	"fmt"
	"regexp"
	"strings"

	"a.yandex-team.ru/infra/skyboned/go/src/util"
	"github.com/vmihailenco/msgpack/v4"
)

type (
	LinkOpts    = map[string]string
	Link        = string
	Hash        = string
	LinkVersion int
	Setter      = func(LinkOpts, string) error
)

type Links interface {
	Parse(interface{}) error
	SetOpts(LinkOpts) error
	Update(Links) error
	Delete(string) (bool, error)
	Pack() ([]byte, error)
}

const (
	MaxLinksCapacity             = 25
	OptSourceID                  = "0"
	OptHTTPRange                 = "1"
	LinkV1           LinkVersion = iota
	LinkV2
)

// translates human-readable option keys to pre-defined in copier
// sets corresponding options
var setters map[string]Setter = map[string]Setter{
	"SourceId": func(info LinkOpts, value string) error {
		optkey := OptSourceID
		delete(info, "SourceId") //just in case
		info[optkey] = value
		return nil
	},
	"Range": func(info LinkOpts, _ string) error {
		optkey := OptHTTPRange
		optval, ok := info["Range"]
		if !ok {
			return nil
		}
		// format validation:
		// should staisfy "bytes=(\d+)-(\d+)", where ranges are first < second
		ok, err := regexp.MatchString("bytes=(\\d+)-(\\d+)", optval)
		if err != nil {
			return err
		}
		if !ok {
			return fmt.Errorf("malformed HTTP Range: %s", optval)
		}
		ranges := strings.Split(strings.TrimPrefix(optval, "bytes="), "-")
		if util.CheckRange(ranges[0], ranges[1]) {
			delete(info, "Range")
			info[optkey] = optval
			return nil
		} else {
			return fmt.Errorf("malformed HTTP Range: %s", optval)
		}
	},
}

// new links format, extened with options
// each link has a LinkOpts assigned to it, where option keys are matched with corresponding option values
// options can be either global (applied to all links, mandatory), or selective (optional)
// SOURCE_ID is mandatory for V2, but HTTP_RANGE is optional, since links may not be ranged
type Info struct {
	Type  LinkVersion
	Links map[Hash]map[Link]LinkOpts
}

func ParseInfo(packedInfo interface{}) (info Info, err error) {
	err = info.Parse(packedInfo)
	return
}

func (l *Info) Parse(packedInfo interface{}) error {
	var info interface{}
	if unpackedInfo, ok := packedInfo.([]byte); ok {
		err := msgpack.Unmarshal(unpackedInfo, &info)
		if err != nil {
			return err
		}
	} else {
		info = packedInfo
	}
	switch info := info.(type) {
	case map[Hash]interface{}:
		l.Links = make(map[Hash]map[Link]LinkOpts)
		for hash, links := range info {
			l.Links[hash] = make(map[Link]LinkOpts)
			switch untypedLinks := links.(type) {
			case map[Link]interface{}:
				l.Type = LinkV2
				for link, linkopts := range untypedLinks {
					l.Links[hash][link] = make(LinkOpts)
					if linkopts, ok := linkopts.(map[string]interface{}); ok {
						for opt, optval := range linkopts {
							l.Links[hash][link][opt], ok = optval.(string)
							if !ok {
								return fmt.Errorf("LinkV2: linkopt parse failed: %T", opt)
							}
						}
					} else {
						return fmt.Errorf("LinkV2: link parse failed: %T", link)
					}
				}
			case []interface{}:
				l.Type = LinkV1
				for _, untypedLink := range untypedLinks {
					link, ok := untypedLink.(Link)
					if !ok {
						return fmt.Errorf("LinkV1: link parse failed: %T", link)
					}
					l.Links[hash][link] = LinkOpts{}
				}
			case string:
				l.Type = LinkV1
				l.Links[hash][untypedLinks] = LinkOpts{}
			default:
				return fmt.Errorf("LinkV2: links parse failed: %T", links)
			}
		}
	case map[Hash]map[Link]LinkOpts:
		l.Links = info
		return nil
	default:
		return fmt.Errorf("LinkV2: parse failed: %T", info)
	}
	return nil
}

func (l *Info) SetOpts(generalopts LinkOpts) error {
	for _, links := range l.Links {
		for _, linkopts := range links {
			for optkey, optval := range linkopts {
				// look for setterfunc in setters
				// if not found we send back 400
				f, ok := setters[optkey]
				if !ok {
					return fmt.Errorf("unknown option: %v", optkey)
				}
				err := f(linkopts, optval)
				if err != nil {
					return err
				}
			}
			for optkey, optval := range generalopts {
				// set options, applicable for all the links
				// source_id, for example
				f, ok := setters[optkey]
				if !ok {
					return fmt.Errorf("unknown general option: %v", optkey)
				}
				err := f(linkopts, optval)
				if err != nil {
					return err
				}
			}
		}
	}
	return nil
}

func (l *Info) Update(newLinks Info) error {
	skip := true
	for hash, currentLinks := range l.Links {
		if len(currentLinks) >= 25 {
			return ErrExceededLinksLimit
		}
		if len(newLinks.Links[hash]) == 0 {
			return ErrHashHasNoLinks
		}
		if len(newLinks.Links[hash]) > 1 {
			return ErrTooManyLinksPerHash
		}
		for link, linkopts := range newLinks.Links[hash] {
			if _, ok := currentLinks[link]; !ok {
				currentLinks[link] = linkopts
				skip = false
			}
		}
	}
	if skip {
		return ErrSkipUpdate
	}
	return nil
}

func (l *Info) Delete(SourceID string) (obliterate bool) {
	// obliteration will remove resource from db completely if no source_id left in links
	obliterate = true
	for _, links := range l.Links {
		for link, linkOpts := range links {
			if sourceID, ok := linkOpts[OptSourceID]; ok {
				if sourceID == SourceID {
					delete(links, link)
				} else {
					obliterate = false
				}
			}
		}
	}
	return
}

func (l *Info) Pack() ([]byte, error) {
	return msgpack.Marshal(l.Links)
}

// Two formats of links are currently present in Skyboned

// # {md5: [link1, link2, ...]}  # v1
// # {md5: {link1: linkopts, link2: linkopts, ...} #v3 (v1 extended)
// produces a Links interface, where either LinkV1 or LinkV2 is stored

var (
	ErrHashHasNoLinks      = errors.New("incoming request is flawed: hash has no link")
	ErrExceededLinksLimit  = errors.New("hash has peaked links amount")
	ErrTooManyLinksPerHash = errors.New("incoming request is flawed: hash has more than one link")
	ErrV1CantDelete        = errors.New("LinkV1 does not support source_id deletion")
	ErrV1CantSetOpts       = errors.New("LinkV1 does not support link options")
	ErrLinkParse           = errors.New("link parse failed")
	ErrWrongFormat         = errors.New("format is wrong")
	ErrSkipUpdate          = errors.New("no new links in update - skip")
)
