package skyboned

import (
	"context"
	"os"
	"time"

	yazap "a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/metrics/solomon"
	"a.yandex-team.ru/library/go/yandex/solomon/reporters/pusher"
	"a.yandex-team.ru/library/go/yandex/solomon/reporters/pusher/httppusher"
	"a.yandex-team.ru/library/go/yandex/tvm"

	"go.uber.org/zap"
)

var (
	Timers               map[string]metrics.Timer
	Counters             map[string]metrics.CounterVec
	Gauges               map[string]metrics.GaugeVec
	defaultTimingBuckets = metrics.NewDurationBuckets(
		20*time.Millisecond,
		100*time.Millisecond,
		250*time.Millisecond,
		500*time.Millisecond,
		1000*time.Millisecond,
		5000*time.Millisecond,
		10000*time.Millisecond,
	)
)

func SetupMetrics() (registry *solomon.Registry) {
	// Create registry with options applied
	opts := solomon.NewRegistryOpts()
	opts.UseNameTag = true
	hname, err := os.Hostname()
	if err != nil {
		zap.S().Error(err)
	} else {
		opts.AddTags(map[string]string{
			"host": hname,
		})
	}
	registry = solomon.NewRegistry(opts)

	Timers = make(map[string]metrics.Timer, 3)
	Counters = make(map[string]metrics.CounterVec, 2)
	Gauges = make(map[string]metrics.GaugeVec, 1)

	// Histograms for measuring timings for stuff
	//Timers["database_request_sec"] = registry.DurationHistogram("database_request_sec", defaultTimingBuckets)
	Timers["add_resource_time"] = registry.DurationHistogram("add_resource_time", metrics.NewDurationBuckets(
		500*time.Millisecond,
		1000*time.Millisecond,
		2000*time.Millisecond,
		4000*time.Millisecond,
		6000*time.Millisecond,
		8000*time.Millisecond,
		10000*time.Millisecond,
		15000*time.Millisecond,
		20000*time.Millisecond,
	))
	Timers["remove_resource_time"] = registry.DurationHistogram("remove_resource_time", metrics.NewDurationBuckets(
		20*time.Millisecond,
		100*time.Millisecond,
		250*time.Millisecond,
		500*time.Millisecond,
		1000*time.Millisecond,
		5000*time.Millisecond,
		10000*time.Millisecond,
	))

	// Counters for {add, remove}_resource codes
	Counters["add_resource_status"] = registry.CounterVec("add_resource_status", []string{"status"})
	Counters["remove_resource_status"] = registry.CounterVec("remove_resource_status", []string{"status"})

	//Gauge for announce_overdue
	Gauges["announce_overdue"] = registry.GaugeVec("announce_overdue", []string{"tracker_address"})
	Gauges["resource_count"] = registry.GaugeVec("resource_count", []string{"owner"})

	return
}
func SetupPusher(tvm tvm.Client, registry *solomon.Registry) {
	l := yazap.NewWithCore(zap.L().Core(), zap.AddCaller())
	push, err := httppusher.NewPusher(
		httppusher.SetCluster("main"),
		httppusher.SetProject("skyboned"),
		httppusher.SetService("main"),
		httppusher.WithTVMTokenProvider(tvm, pusher.TVMIDProduction),
		httppusher.WithLogger(l),
	)
	if err != nil {
		zap.S().Error(err)
	}

	if registry != nil {
		go func() {
			err := push.Start(context.Background(), registry, time.Second*5)
			if err != nil {
				zap.S().Error(err)
			}
		}()
		zap.S().Info("pusher started")
	} else {
		zap.S().Info("skipped pusher due to nil metric registry")
	}

}
