package skyboned

import (
	"context"
	"flag"
	"net/http"
	"strings"

	pb "a.yandex-team.ru/infra/skyboned/go/skyboned_rpc"
	"go.uber.org/zap"

	grpc_rt "github.com/grpc-ecosystem/grpc-gateway/runtime"
	"golang.org/x/net/http2"
	"golang.org/x/net/http2/h2c"
	"google.golang.org/grpc"
)

var (
	HTTPPort       int
	HTTPSPort      int
	MaxConnections int
	NoWait         bool
)

func ServerFlags() {
	flag.IntVar(&HTTPPort, "http-port", 80, "")
	flag.IntVar(&HTTPSPort, "https-port", 443, "")
	flag.IntVar(&MaxConnections, "sem-cap", 1024, "")
	flag.BoolVar(&NoWait, "no-wait", false, "skyboned server will not wait for resource to be announced")
}

// Request duplex for grpc/json over http requesting
func httpGrpcRouter(grpcServer *grpc.Server, httpHandler *grpc_rt.ServeMux) http.Handler {
	return h2c.NewHandler(http.HandlerFunc(func(w http.ResponseWriter, r *http.Request) {
		if r.ProtoMajor == 2 && strings.Contains(r.Header.Get("Content-Type"), "application/grpc") {
			grpcServer.ServeHTTP(w, r)
		} else {
			httpHandler.ServeHTTP(w, r)
		}
	}), &http2.Server{})
}

func SetupServer() (*http.Server, error) {
	zap.S().Info("starting server...")

	gserv := grpc.NewServer() // grpc.Server for gRPC requests
	mux := grpc_rt.NewServeMux(
		grpc_rt.WithIncomingHeaderMatcher(func(header string) (string, bool) {
			switch header {
			case "X-Ya-Service-Ticket":
				{
					return header, true
				}
			default:
				{
					return grpc_rt.DefaultHeaderMatcher(header)
				}
			}
		}),
	) // ServeMux from grpc_runtime for HTTP connections

	zap.S().Info("init skyboned...")
	skbnd, err := SetupSkybonedServer()
	if err != nil {
		zap.S().Fatal(err)
	}
	zap.S().Info("skyboned: ready")
	pb.RegisterSkybonedServer(gserv, skbnd)
	err = pb.RegisterSkybonedHandlerServer(context.Background(), mux, skbnd)
	if err != nil {
		zap.S().Fatal(err)
	}
	serv := &http.Server{
		Handler: httpGrpcRouter(gserv, mux),
	}
	return serv, nil
}
