from __future__ import division

import contextlib
import errno
import gevent.event
import gevent.pool
import gevent.queue
import gevent.socket
import msgpack
import random
import time

from collections import deque, defaultdict, OrderedDict

from infra.skybit.src.component import Component

"""
Announcer.scheduler -- schedules announce requests
Announcer.announce_table
"""

ANNOUNCE_SEEDING = 0
ANNOUNCE_DOWNLOADING = 1
ANNOUNCE_STOPPED = 2

ANNOUNCE_STATE_SEEDING = ANNOUNCE_SEEDING
ANNOUNCE_STATE_DOWNLOADING = ANNOUNCE_DOWNLOADING
ANNOUNCE_STATE_STOPPED = ANNOUNCE_STOPPED

RESULT_OK = 0
RESULT_ERR = 1
RESULT_TIMEOUT = -1


class ResizeablePool(gevent.pool.Pool):
    def increase(self, size):
        self.size += size
        for _ in range(0, size):
            self._semaphore.release()

    def decrease(self, size):
        self.size -= size
        self._semaphore.counter -= size


class AnnounceResponseSeeding(object):
    __slots__ = ('dbid', 'interval', )

    def __init__(self, interval):
        self.interval = interval
        self.dbid = None


class AnnounceResponseLeeching(object):
    __slots__ = ('dbid', 'interval', 'seeders', 'leechers', 'peers')

    def __init__(self, interval, seeders, leechers, peers):
        self.interval = interval
        self.seeders = seeders
        self.leechers = leechers
        self.peers = peers
        self.dbid = None


class AnnounceResponseStopped(object):
    __slots__ = ('dbid', )

    def __init__(self):
        self.dbid = None


class BaseClient(Component):
    MAX_PACKETS_IN_FLIGHT = 512
    MIN_PACKETS_IN_FLIGHT = 2

    codename = 'base'

    def __init__(self, logname, log_prefix, send, recv, uid, ips, data_port, parent=None):
        super(BaseClient, self).__init__(
            logname=logname,
            log_msg_prefix=log_prefix,
            parent=parent
        )

        self.uid = uid
        self.hexuid = bytes.fromhex(uid)
        self._db_id = None

        self.send = send
        self.recv = recv

        self.ips = ips
        self.data_port = data_port

        self.connect_deadline = 0

        self.new_job_event = gevent.event.Event()
        self.jobs = OrderedDict()
        self.pool = ResizeablePool(size=self.MIN_PACKETS_IN_FLIGHT)

        # pending transactions
        # map: {tid: ev}
        self.transactions = OrderedDict()

        self.fails_in_a_row = 0
        self.success_in_a_row = 0

        self.packet_stats = [0, 0, 0]  # send, received, lost

        self.links = defaultdict(list)

        self._connect_lock = gevent.lock.Semaphore(1)
        self._connect_grn = None

    def get_state(self):
        return {
            'connect_deadline': self.connect_deadline
        }

    def set_state(self, state):
        connect_deadline = state.get('connect_deadline', None)
        if connect_deadline:
            self.log.info('Switch connect deadline %d => %d (from state)', self.connect_deadline, connect_deadline)
            self.connect_deadline = connect_deadline

    def generate_tid(self):
        return int(random.random() * 2 ** 32 - 1)

    def set_db_id(self, dbid):
        self._db_id = dbid

    def get_db_id(self):
        return self._db_id

    def add_link(self, infohash, meth):
        self.links[infohash].append(meth)

    def del_link(self, infohash, meth):
        if infohash not in self.links:
            return

        try:
            self.links[infohash].remove(meth)
        except ValueError:
            pass

        if not self.links[infohash]:
            del self.links[infohash]

    def _get_connect_packet(self, tid):
        raise NotImplementedError()

    def _analyze_packet(self, data):
        raise NotImplementedError()

    def _analyze_connect_response(self, data):
        raise NotImplementedError()

    def _get_announce_packet(self, tid, infohash, state, network):
        raise NotImplementedError()

    def _analyze_announce_response(self, data):
        raise NotImplementedError()

    def _send_reconnect(self, log, reason):
        connect_fails_in_a_row = 0

        while True:
            if reason:
                log.info('Attempting to reconnect to tracker (reason: %s)', reason)

            tid = self.generate_tid()
            ev = self.transactions[tid] = gevent.event.AsyncResult()

            try:
                req = self._get_connect_packet(tid)

                self.packet_stats[0] += 1
                self.send(tid, req)
                timeout = min(120, max(1, connect_fails_in_a_row + 1) * 3)
                data = ev.wait(timeout=timeout)

                if data is None:
                    self._packet_fail()
                    connect_fails_in_a_row += 1
                    log.debug('Reconnect timeout (%ds)', timeout)
                    self.send(tid, -1)
                else:
                    assert isinstance(data, (list, tuple)), 'invalid data type %r' % (data, )
                    assert len(data) == 2, 'invalid data length %r' % (data, )

                    restype, data = data[0], data[1]
                    if restype == RESULT_ERR:
                        self._packet_fail()
                        connect_fails_in_a_row += 1
                        log.debug('Reconnect failed (%s)', data[0])
                        gevent.sleep(1)
                        self.send(tid, -1)
                    elif restype == RESULT_OK:
                        self._packet_ok()
                        self.connect_deadline = self._analyze_connect_response(data)
                        return

            except Exception:
                import traceback
                self.log.warning('Unhandled error in _send_reconnect: %s', traceback.format_exc())
                gevent.sleep(1)  # avoid busy loops
                return

            finally:
                self.transactions.pop(tid)

    def _send_announce(self, infohash, action, network):
        tid = self.generate_tid()

        try:
            ev = self.transactions[tid] = gevent.event.AsyncResult()

            req = self._get_announce_packet(tid, infohash, action, network)

            self.packet_stats[0] += 1
            self.send(tid, req)
            result = ev.wait(timeout=10)

            if result is not None:
                restype, data = result
            else:
                restype, data = None, None

            if restype == RESULT_OK and data is not None:
                self._packet_ok()
                response = self._analyze_announce_response(data)
                response.dbid = self._db_id
                return response
            else:
                self._packet_fail()
                if data:
                    if data == ('Unknown peer', ):
                        if self.connect_deadline > 0:
                            self.log.warning('Tracker dont know us - resetting connect deadline')
                            self.connect_deadline = 0
                        else:
                            self.log.error('We have connect_deadline=0, but tracker still dont know us')
                    else:
                        if data is not None:
                            self.log.warning('Got error from tracker: %s', data[0])
                        else:
                            self.log.warning('Announce timed out (10s)')

                    if data is not None:
                        gevent.sleep(1)

                return None

        except Exception:
            import traceback
            self.log.warning('Unhandled exception in _send_announce: %s', traceback.format_exc())
            gevent.sleep(1)  # avoid busy loops
            return None

        finally:
            self.transactions.pop(tid)

    def announce(self, infohash, state, network, timeout):
        """
        Announce and block until it completes or timeout occurs

        """

        if self._db_id is None:
            raise Exception('db id not set!')

        with gevent.Timeout(timeout) as tout:
            try:
                while True:
                    with self._connect_lock:
                        if self._connect_grn and not self._connect_grn.ready():
                            # Already connecting
                            self._connect_grn.join()

                        if self.connect_deadline == 0 or time.time() >= self.connect_deadline:
                            reconnect = True
                            reconnect_reason = None
                        elif self.fails_in_a_row > 128:
                            reconnect = True
                            reconnect_reason = '%d packets failed in a row' % (
                                self.fails_in_a_row,
                            )
                        else:
                            reconnect = False
                            reconnect_reason = None

                        if reconnect:
                            self._connect_grn = self.pool.spawn(self._send_reconnect, self.log, reconnect_reason)
                            self._connect_grn.join()
                            self._connect_grn = None

                            # Go thru next loop
                            continue

                    grn = self.pool.spawn(
                        self._send_announce, infohash, state,
                        network
                    )
                    try:
                        ret = grn.get()
                        if not ret:
                            continue
                        return ret
                    except Exception as ex:
                        grn.kill()
                        if ex == tout:
                            raise
                        self.log.debug('Announce failed: %s', ex)
                        continue

            except gevent.Timeout as ex:
                if ex != tout:
                    raise
                return None

    @Component.green_loop(logname='recv')
    def receiver(self, log):
        data = self.recv()

        if isinstance(data, (list, tuple)) and len(data) == 2:
            tid, restype, payload = data[0], data[1][0], data[1][1]
        else:
            ret = self._analyze_packet(data)
            if not ret:
                return 0

            tid, restype, payload = ret

        if tid == 0 and restype == RESULT_ERR:
            log.warning('Got error from tracker with tid=0: %s', payload[0])
            return 0

        try:
            ev = self.transactions[tid]
        except KeyError:
            return 0

        ev.set((restype, payload))

    def _packet_ok(self):
        self.fails_in_a_row = 0
        self.success_in_a_row += 1

        self.packet_stats[1] += 1

        if self.success_in_a_row >= self.MIN_PACKETS_IN_FLIGHT:
            if self.pool.size < self.MAX_PACKETS_IN_FLIGHT:
                self.pool.increase(1)

    def _packet_fail(self):
        self.fails_in_a_row += 1
        self.success_in_a_row = 0

        self.packet_stats[2] += 1

        if self.pool.size > self.MIN_PACKETS_IN_FLIGHT:
            decrease_by = 2
            self.pool.decrease(decrease_by)

    def _packet_stats_reset(self):
        if self.pool.size > self.MIN_PACKETS_IN_FLIGHT:
            self.pool.decrease(self.pool.size - self.MIN_PACKETS_IN_FLIGHT)

    def __repr__(self):
        return '<Client %s %s>' % (self.codename, hash(self), )


class SkyboneCoordClient(BaseClient):
    codename = 'coord'
    initial_cid = 4079332072  # protocol v3

    ACTION_CONNECT = 0
    ACTION_ANNOUNCE = 10
    ACTION_ANNOUNCE_DFS = 11
    ACTION_STARTUP = 20
    ACTION_SHUTDOWN = 30

    def __init__(self, log_prefix, send, recv, uid, ips, data_port, dfs=False, parent=None):
        self.dfs = dfs

        super(SkyboneCoordClient, self).__init__(
            logname='sc',
            log_prefix=log_prefix,
            send=send, recv=recv, uid=uid, ips=ips, data_port=data_port, parent=parent
        )

    def _get_connect_packet(self, tid):
        ext = {'peer_types'}

        if self.dfs:
            ext.add('dfs')

        return msgpack.dumps((
            self.initial_cid,
            self.ACTION_CONNECT,     # action (connect)
            tid,                     # transaction id
            self.hexuid,             # uid (raw bytes)
            [
                self.ips['bb'], self.ips['bb6'],
                self.ips['fb'], self.ips['fb6']
            ],
            self.data_port,
            list(ext),
        ),
        use_bin_type=False
        )

    def _get_announce_packet(self, tid, infohash, state, network):
        return msgpack.dumps((
            self.initial_cid,
            self.ACTION_ANNOUNCE,
            tid,
            self.hexuid,
            infohash,
            {ANNOUNCE_SEEDING: 2, ANNOUNCE_STOPPED: 3, ANNOUNCE_DOWNLOADING: 1}[state],
            {
                'auto': 0,
                'bb': 1,
                'fb': 2,
            }[network]
        ),
        use_bin_type=False
        )

    def _analyze_packet(self, data):
        data = msgpack.loads(data)
        tid, restype, data = data[0], data[1], data[2:]

        return tid, restype, data

    def _analyze_connect_response(self, data):
        # Connect packet will set us deadline in seconds, but we return deadline as
        # a timestamp
        return time.time() + data[0]

    def _analyze_announce_response(self, data):
        # seeding: (interval, )
        # leeching: (interval, n_seeders, n_leechers, (peers, ))
        # stopping: ()

        if len(data) >= 1:
            # seeding or leeching
            interval = data[0]
        else:
            # stopped
            return AnnounceResponseStopped()

        if len(data) == 1:
            # seeding
            return AnnounceResponseSeeding(interval=interval)

        if len(data) == 4:
            return AnnounceResponseLeeching(
                interval=interval,
                seeders=data[1], leechers=data[2],
                peers=data[3]
            )

        self.log.warning('Unable to analyze announce response: %r', data)
        return None


class TrackerClient(Component):
    def __init__(self, uid, ips, data_port, dfs=False, parent=None):
        super(TrackerClient, self).__init__(logname='', parent=parent)

        self.uid = uid
        self.dfs = dfs
        self.active_kinds = []
        self.send_queue = gevent.queue.Queue()
        self.ips = ips
        self.data_port = data_port

        # map: {addr: [worker, peers]}
        self.clients = {}

        # map: {peer: (queue, worker)}
        self.queue_by_peer = {}

    def add(self, kind, address, peers):
        possible_kinds = [c.codename for c in (SkyboneCoordClient, )]
        assert kind in possible_kinds

        if kind not in self.active_kinds:
            self.active_kinds.append(kind)

        cls = {
            'coord': SkyboneCoordClient,
        }[kind]

        receive_queue = gevent.queue.Queue()
        worker = cls(
            address,
            lambda tid, data: self._send(address, tid, data),
            receive_queue.get,
            uid=self.uid,
            ips=self.ips,
            data_port=self.data_port,
            dfs=self.dfs,
            parent=self
        )

        self.clients[address] = [worker, deque(peers)]
        for peer in peers:
            self.queue_by_peer[peer] = receive_queue

    def get_state(self):
        state = {}
        for address, (worker, queue) in self.clients.items():
            state[address] = worker.get_state()
        return state

    def set_state(self, state):
        for address, (worker, queue) in self.clients.items():
            if address in state:
                worker.set_state(state[address])

    @contextlib.contextmanager
    def linked(self, infohash, meth):
        try:
            clients = self.add_link(infohash, meth)
            yield
        finally:
            self.del_link(infohash, meth, clients)

    def add_link(self, infohash, meth):
        clients = []
        for client, _ in self.clients.values():
            client.add_link(infohash, meth)
            clients.append(client)
        return clients

    def del_link(self, infohash, meth, clients=None):
        if not clients:
            clients = (_[0] for _ in self.clients.values())
        for client in clients:
            client.del_link(infohash, meth)

    def _send(self, address, tid, data):
        peers = self.clients[address][1]
        if data == -1:
            peers.rotate()
            return

        return self.send_queue.put((tid, data, peers[0]))

    def status(self):
        status = {
            'packets': {
                'sent': 0, 'recv': 0, 'lost': 0
            }
        }

        for client, _ in self.clients.values():
            status['packets']['sent'] += client.packet_stats[0]
            status['packets']['recv'] += client.packet_stats[1]
            status['packets']['lost'] += client.packet_stats[2]

        return status


class Announcer(Component):
    MAX_ANNOUNCE_DB_INTERVAL = 30  # max reannounce interval
    MIN_ANNOUNCE_DB_INTERVAL = 5
    ANNOUNCE_TABLE_CHECK_INTERVAL = 600

    def __init__(
        self, uid, db, dbnew, dbt, trackers, ips, port, data_port,
        db_scheduler=True, dfs=False, parent=None
    ):
        super(Announcer, self).__init__(logname='ann', parent=parent)

        self.uid = uid
        self.db = db
        self.dbnew = dbnew
        self.dbt = dbt

        self.sock = None
        self.port = port
        self.data_port = data_port
        self.scheduler_loops = []
        self.db_scheduler = db_scheduler

        for key, hosts in trackers.items():
            assert key in ('coord', )

        # map: {kind: {hostname: [((ip, port), worker)]}}
        # map: {peer: worker}
        # map: {hostname: [peer, peer]}
        self.client = TrackerClient(self.uid, ips=ips, data_port=self.data_port, dfs=dfs, parent=self)

        # Create a bunch of our tracker workers
        for kind, tiers in trackers.items():
            for hostname, ips in tiers.items():
                address = '%s:%d' % (hostname, ips[0][1])

                for idx, pair in enumerate(ips):
                    if isinstance(pair, list):
                        ips[idx] = tuple(pair)

                self.log.info('Adding tracker client %r at %s:%s', kind, address, ips)
                self.client.add(kind, address, ips)

        self._next_announce_table_check_ts = int(time.time()) + self.ANNOUNCE_TABLE_CHECK_INTERVAL

    def start(self):
        if self.db_scheduler:
            existent_trackers = {}
            for tracker_id, tracker_hostname in self.db.query('SELECT id, address FROM announce_tracker'):
                existent_trackers[tracker_hostname] = tracker_id

            good_trackers = {}
            new_trackers = []
            for address, (worker, peers) in self.client.clients.items():
                if address not in existent_trackers:
                    new_trackers.append(address)
                else:
                    good_trackers[address] = existent_trackers[address]

            if new_trackers:
                for address in new_trackers:
                    good_trackers[address] = self.db.query_one_col(
                        'INSERT INTO announce_tracker (address) VALUES (%s) RETURNING id',
                        (address, )
                    )

            assert len(good_trackers) == 1, 'Only 1 tracker supported as of now'

            self.tracker_id = good_trackers[address]

            for address, (worker, _) in self.client.clients.items():
                worker.set_db_id(good_trackers[address])

                if '.search.yandex.net' in address:
                    short_name = address.replace('.search.yandex.net', '')
                elif '.yandex.ru':
                    short_name = address.replace('.yandex.ru', '')
                else:
                    short_name = address
                loop = self.add_loop(
                    self.scheduler,
                    logname='scheduler][%s][%s' % (worker.codename, short_name),
                    tracker_id=worker.get_db_id(), address=address
                )
                self.scheduler_loops.append(loop)
                loop = self.add_loop(
                    self.scheduler,
                    logname='scheduler_new][%s][%s' % (worker.codename, short_name),
                    tracker_id=worker.get_db_id(), address=address, only_new=True
                )
                self.scheduler_loops.append(loop)

        self.sock = gevent.socket.socket(gevent.socket.AF_INET6, gevent.socket.SOCK_DGRAM)
        self.sock.setsockopt(gevent.socket.IPPROTO_IPV6, gevent.socket.IPV6_V6ONLY, 1)
        self.sock.bind(('', self.port))

        return super(Announcer, self).start()

    def status(self, use_db):
        client_status = self.client.status()

        return {
            'pending': (
                self.db.query_one_col(
                    'SELECT COUNT(*) FROM announce WHERE timestamp < ?',
                    [time.time()],
                    log=False
                ) if use_db else None
            ),
            'stats': client_status,
        }

    class Announce(object):
        EV_FAIL = 1
        EV_OK = 2

        def __init__(self, infohash, clients, action, network, timeout):
            self.events = gevent.queue.Queue()
            self.group = gevent.pool.Group()
            self.jobs = 0
            for client, (clientobj, _) in clients.items():
                self.group.spawn(
                    self._announce,
                    clientobj, infohash, action, network, timeout
                )
                self.jobs += 1

        def _announce(self, client, infohash, action, network, timeout):
            reply = client.announce(infohash, action, network, timeout)
            if not reply:
                self.events.put((self.EV_FAIL, client, None))
            else:
                self.events.put((self.EV_OK, client, reply))

        def wait(self):
            while self.jobs > 0:
                yield self.events.get()
                self.jobs -= 1

        def kill(self):
            self.group.kill()

    @contextlib.contextmanager
    def announce(self, infohash, action, network, timeout=None, clients=None):
        if not clients:
            clients = self.client.clients
        else:
            clients_dict = self.client.clients.copy()
            for client, (clientobj, _) in clients_dict.items():
                if clientobj not in clients:
                    clients_dict.pop(client)
            clients = clients_dict

        ann = self.Announce(infohash, clients, action, network, timeout)

        try:
            yield ann
        finally:
            ann.kill()

    def announce_or_fail(self, infohash, count=2, timeout=300):
        assert 0, 'not working atm'

        try:
            with self.announce(infohash, ANNOUNCE_SEEDING, 'auto', timeout=timeout) as ann:
                registered_on = 0
                for ev, client, data in ann.wait():
                    if not data:
                        self.log.warning('Tracker: %r timed out', client)
                    else:
                        registered_on += 1

                    if registered_on >= min(count, len(self.client.clients)):
                        return True
        finally:
            self._renew_db_announce_records(infohash, top=True)
            [sched.wakeup() for sched in self.scheduler_loops]

    def _renew_db_announce_records(self, infohash, tracker_id, top=False):
        now = int(time.time())

        with self.dbt:
            self.dbt.execute('DELETE FROM announce WHERE resource_id = %s AND tracker_id = %s', (
                infohash, tracker_id
            ))

            # Note what we sent next scheduled timestamp to now - 1 second.
            # So, scheduler will not be required to fullfill current second frame
            # to announce that. Speedup for sharing :).
            # If top asked, that means move this announce to top of queue
            # In that case we set next announce timestamp to zero
            # This allos share jobs to complete even if we have our announce queue
            # overflowed.
            if top:
                ts = 0
            else:
                ts = now - 1

            self.dbt.execute(
                'DELETE FROM announce_op WHERE resource = %s AND tracker_id = %s', (infohash, tracker_id)
            )

            self.dbt.execute(
                'INSERT INTO announce (resource_id, schedule_ts, tracker_id) VALUES (%s, %s, %s)', (
                    infohash, ts, tracker_id
                )
            )

        return infohash

    def _resync_announce_table(self):
        now = int(time.time())

        assert len(self.client.clients) == 1

        tracker_id = next(iter(self.client.clients.values()))[0].get_db_id()

        if self._next_announce_table_check_ts <= now:
            self._next_announce_table_check_ts = now + self.ANNOUNCE_TABLE_CHECK_INTERVAL

            log = self.log.getChild('sync')

            with self.dbt, self.dbt.deblock.lock('resync announces'):
                resources_count = self.db.query_one_col('SELECT COUNT(*) FROM resource')
                announce_count = self.db.query_one_col(
                    'SELECT COUNT(*) FROM announce WHERE tracker_id = %s',
                    (tracker_id, )
                )

                if resources_count is None:
                    resources_count = 0

                if announce_count is None:
                    announce_count = 0

                required_count = resources_count

                if required_count != announce_count:
                    log.info(
                        'Announce count (%d) dont match resources (%d), resyncing...',
                        announce_count, required_count
                    )
                    with self.dbt:
                        infohashes_to_renew = set()

                        for tracker, _ in self.client.clients.values():
                            infohashes_to_renew.update(self.db.query_col(
                                'SELECT r.id FROM resource r '
                                'LEFT JOIN announce a ON a.resource_id = r.id AND a.tracker_id = %s '
                                'WHERE a.resource_id IS NULL',
                                [tracker.get_db_id()]
                            ))

                        infohashes_to_renew_count = len(infohashes_to_renew)

                        print_ts = 0
                        for idx, infohash in enumerate(infohashes_to_renew):
                            self._renew_db_announce_records(infohash, tracker_id)
                            now = int(time.time() // 10)
                            if print_ts != now:
                                done = idx + 1
                                log.debug(
                                    'Resyncing announces: done %0.2f%%: %d of %d',
                                    done / infohashes_to_renew_count * 100,
                                    done, infohashes_to_renew_count
                                )
                                print_ts = now

                    log.info('Resync complete')

    def scheduler(self, log, tracker_id, address, only_new=False):
        db = self.dbnew if only_new else self.db

        now = int(time.time())

        if not only_new:
            self._resync_announce_table()

        # Smartass scheduler
        # 1) we grab packets we should send in next 60 seconds frame
        # 2) we split them per second basis
        # 3) in each second we sleep equally between scheduling packets
        #
        # E.g.: we have 600 packets to send in each second for next 60 seconds (assuming they are
        # scheduled equally in db per second). After that each second we should send 10 packets, so we sleep
        # 0.1s between each schedule. If we unable to schedule all packets within required second -- we schedule
        # them all at second one.

        query_by = 10000
        query_by_stops = 500
        status = 'ok'
        stats = {'announces_ok': 0, 'announces_ok_saved': 0, 'stops_ok': 0, 'announces_failed': 0, 'stops_failed': 0}

        db.execute(
            'DELETE FROM announce_op WHERE tracker_id = %s AND op = %s AND deadline <= %s',
            (
                tracker_id, 'remove', now
            ),
        )

        announce_pool = gevent.pool.Pool(size=1000)

        jobs = None

        def _has_new_announces():
            found = False
            for idx in range(len(db.notifies) - 1, -1, -1):
                if db.notifies[idx].channel == 'announce_new':
                    found = True
                    db.notifies.pop(idx)
            return found

        try:
            if only_new:
                jobs = (
                    db.query(
                        'SELECT '
                        '   resource_id, schedule_ts, %s '
                        'FROM announce '
                        'WHERE tracker_id = %s AND schedule_ts = 0 '
                        'LIMIT %s',
                        [
                            ANNOUNCE_SEEDING,
                            tracker_id,
                            query_by,
                        ]
                    )
                )
            else:
                db.execute('LISTEN announce_new')
                _has_new_announces()  # just clean notifies if any

                jobs = (
                    db.query(
                        'SELECT resource, 0, %s '
                        'FROM announce_op '
                        'WHERE tracker_id = %s AND op = %s AND deadline > %s '
                        'ORDER BY deadline ASC '
                        'LIMIT %s',
                        [
                            ANNOUNCE_STOPPED,
                            tracker_id,
                            'remove',
                            now,
                            query_by_stops,
                        ]
                    )
                    + db.query(
                        'SELECT '
                        '   resource_id, schedule_ts, %s '
                        'FROM announce '
                        'WHERE tracker_id = %s AND schedule_ts < %s '
                        'ORDER BY schedule_ts ASC '
                        'LIMIT %s',
                        [
                            ANNOUNCE_SEEDING,
                            tracker_id,
                            now + 60,
                            query_by,
                        ]
                    )
                )

            # pps = len(jobs) // 60.
            pps = 10000  # set max pps to 500 directly (30krps per minute)

            sent = [int(time.time()), 0]
            gevent.sleep(int(time.time()) + 1 - time.time())

            db_update = []
            db_delete = []

            def _db_ops():
                if db_update:
                    upd_q = []
                    for infohash, schedule_ts in db_update:
                        # ::bpchar cast for byte-patting char, so index can be used
                        # ::char(40) also can be used here
                        upd_q.append('(\'%s\'::bpchar, %d)' % (infohash, schedule_ts))

                    upd_q = ', '.join(upd_q)
                    updates_cnt = len(db_update)
                    db_update[:] = []

                    query = (
                        'UPDATE announce AS a SET '
                        '   schedule_ts = c.schedule_ts '
                        'FROM (values ' + upd_q + ') AS c(infohash, schedule_ts) '
                        'WHERE c.infohash = a.resource_id AND a.tracker_id = %s'
                    )

                    db.execute(query, [tracker_id])
                    log.info('  updated %d announces in db', updates_cnt)

                if db_delete:
                    db_delete_copy = db_delete[:]
                    db_delete[:] = []
                    db.execute(
                        'DELETE FROM announce_op WHERE tracker_id = %s AND op = %s AND resource IN (%S)',
                        [tracker_id, 'remove', db_delete_copy]
                    )
                    log.info('  deleted %d announces in db', len(db_delete_copy))

            for infohash, timestamp, action in jobs:
                def _announce(tracker_id=tracker_id, address=address, infohash=infohash, action=action):
                    reply = self.client.clients[address][0].announce(
                        infohash, action, 'auto', timeout=60
                    )
                    if not reply:
                        if action == ANNOUNCE_SEEDING:
                            stats['announces_failed'] += 1
                        elif action == ANNOUNCE_STOPPED:
                            stats['stops_failed'] += 1
                        return

                    if action == ANNOUNCE_SEEDING:
                        stats['announces_ok'] += 1
                        # db.execute(
                        #     'UPDATE announce SET schedule_ts = %s WHERE resource_id = %s AND tracker_id = %s',
                        #     [int(time.time()) + reply.interval, infohash, tracker_id],
                        #     log=False
                        # )
                        db_update.append((infohash, int(time.time()) + reply.interval))
                    elif action == ANNOUNCE_STOPPED:
                        # stop
                        stats['stops_ok'] += 1
                        # db.execute(
                        #     'DELETE FROM announce_op WHERE tracker_id = %s AND op = %s AND resource = %s',
                        #     [tracker_id, 'remove', infohash],
                        #     log=False
                        # )
                        db_delete.append(infohash)

                if infohash != 'fake':
                    announce_pool.spawn(_announce)

                if sent[0] != int(time.time()):
                    sent[:] = (int(time.time()), 1)
                else:
                    sent[1] += 1
                    if sent[1] >= pps:
                        sleep = sent[0] + 1 - time.time()
                        deadline = time.time() + sleep
                        while time.time() < deadline:
                            to_sleep = deadline - time.time()
                            to_sleep = min(0.3, to_sleep)
                            to_sleep = max(to_sleep, 0)

                            if to_sleep > 0:
                                _db_ops()
                                gevent.sleep(to_sleep)  # poll db for notifies every 0.3s

                            db.poll()
                            if _has_new_announces():
                                raise self.WakeUp

            announce_pool.join()

        except self.WakeUp:
            announce_pool.join(timeout=2)  # allow pending announced to complete
            announce_pool.kill()
            _db_ops()
            status = 'wakeup'
            wakeup = True
        else:
            _db_ops()
            wakeup = False

        if stats['stops_failed'] or stats['announces_failed']:
            next_check = self.MIN_ANNOUNCE_DB_INTERVAL
        elif wakeup:
            next_check = 0
        elif len(jobs) >= query_by:
            next_check = 0
        else:
            pending_stops = db.query_one_col(
                'SELECT COUNT(*) FROM announce_op WHERE tracker_id = %s AND op = %s',
                (tracker_id, 'remove')
            )

            if pending_stops:
                next_check = self.MIN_ANNOUNCE_DB_INTERVAL
            else:
                next_ts = db.query_one_col(
                    'SELECT min(schedule_ts) '
                    'FROM announce '
                    'WHERE tracker_id = %s AND schedule_ts > %s',
                    [tracker_id, now], log=False
                )
                if next_ts is not None:
                    next_check = min(self.MAX_ANNOUNCE_DB_INTERVAL, max(next_ts - now, self.MIN_ANNOUNCE_DB_INTERVAL))
                else:
                    next_check = self.MAX_ANNOUNCE_DB_INTERVAL

        if jobs:
            log.info(
                'Announced %d seeding + %d stopping (%d + %d failed), '
                'next check in %ds (status: %s, total %d jobs, %d left)',
                stats['announces_ok'], stats['stops_ok'],
                stats['announces_failed'], stats['stops_failed'],
                next_check, status, len(jobs),
                len(jobs) - (
                    stats['announces_ok'] + stats['stops_ok'] + stats['announces_failed'] + stats['stops_failed']
                )
            )

        if only_new:
            return 0

        if next_check > 0:
            # Sleep here, not outside of shceduler loop
            # this allows us to wait on db and listen new announce events
            deadline = now + next_check

            while time.time() < deadline:
                db.wait(timeout=1)
                if _has_new_announces():
                    log.info('  aborted wait, got new announces')
                    return 0

        return next_check

    @Component.green_loop(logname='recv')
    def receiver(self):
        data, peer = self.sock.recvfrom(2048)
        self.client.queue_by_peer[peer[:2]].put(data)
        return 0

    @Component.green_loop(logname='send')
    def sender(self):
        tid, data, peer = self.client.send_queue.get()
        try:
            self.sock.sendto(data, peer)
        except gevent.socket.error as ex:
            if ex.errno == errno.ENETUNREACH:
                # we will have ENETUNREACH errno if we are attemping to send
                # ipv4 packet on machine which has no ipv4 interfaces and vise versa.
                gevent.sleep(1)
                self.client.queue_by_peer[peer[:2]].put((tid, (-1, str(ex))))
            else:
                raise
        return 0

    def get_state(self):
        return {
            'client': self.client.get_state()
        }

    def set_state(self, state):
        if 'client' in state:
            self.client.set_state(state['client'])
