import msgpack

from infra.skyboned.src.utils import human_ts, human_size, human_dict


class Resource:
    def __init__(self, db, db_ro, dbt, id, head=None, info=None, mode=None, owner=None, ts_added=None, ts_used=None):
        self.db = db
        self.db_ro = db_ro
        self.dbt = dbt

        self.id = id
        self.type = 'rbtorrent1'
        self.head = head
        self.info = info
        self.mode = mode
        self.owner = owner
        self.ts_added = ts_added
        self.ts_used = ts_used

        self._stored = False

    def __str__(self):
        files = 0
        size = 0
        for f in self.head['structure'].values():
            if f['type'] == 'file':
                files += 1
                size += f['size']

        return """resource: %s
type: %s
files: %s
size: %s
head: %s
links: %s
mode: %s
owner: %s
ts_added: %s
ts_used: %s""" % (self.id,
                  self.type,
                  files,
                  human_size(size),
                  human_dict(self.head) if files < 10 else '...',
                  human_dict(self.info) if files < 10 else '...',
                  self.mode,
                  self.owner,
                  human_ts(self.ts_added),
                  human_ts(self.ts_used))

    def load_dbrow(self, row):
        assert self.id == row[0]
        self.type = row[1]
        self.head = msgpack.unpackb(row[2])
        self.info = msgpack.unpackb(row[3])
        self.mode = row[4]
        self.owner = row[5]
        self.ts_added = row[6]
        self.ts_used = row[7]

    def dbrow(self):
        return [
            self.id,
            self.type,
            msgpack.packb(self.head, use_bin_type=True),
            msgpack.packb(self.info, use_bin_type=True),
            self.mode,
            self.owner,
            self.ts_added,
            self.ts_used
        ]

    def load(self):
        assert self.id is not None

        row = self.db_ro.query_one(
            'SELECT id, type, data, info, mode, owner, ts_added, ts_used FROM resource WHERE id = %s',
            (self.id, )
        )

        if not row:
            return False

        self.load_dbrow(row)
        self._stored = True

        return True

    def save(self):
        assert self.mode == 'plain'

        self.db.execute(
            'INSERT INTO resource (id, type, data, info, mode, owner, ts_added, ts_used) VALUES (%S)',
            (
                self.dbrow(),
            )
        )

        self._stored = True
