import grequests
import datetime
import json
from contextlib import contextmanager
import time


def human_ts(ts):
    return datetime.datetime.fromtimestamp(ts).strftime('%Y-%m-%d %H:%M:%S')


def human_secs(secs, months=False):
    if months and secs > 86400 * 60:
        months = int(secs // (86400 * 30))
        weeks = (secs % (86400 * 30)) // (86400 * 7)

        if months >= 12 or weeks == 0:
            return '%dM' % (months, )

        return '%dM%dw' % (months, weeks)

    elif secs > 86400 * 14:
        weeks = int(secs // (86400 * 7))
        days = (secs % (86400 * 7)) // 86400

        if weeks >= 4 or days == 0:
            return '%dw' % (weeks, )

        return '%dw%dd' % (weeks, days)

    elif secs > 86400:
        days = int(secs // 86400)
        hrs = (secs % 86400) // 3600

        if hrs == 0:
            return '%dd' % (days, )

        return '%dd%dh' % (days, hrs)

    elif secs > 3600:
        hrs = int(secs // 3600)
        mins = (secs % 3600) // 60

        return '%dh%dm' % (hrs, mins)

    elif secs > 60:
        return '%dm' % (secs // 60)

    else:
        return '%ds' % (secs, )


def human_size(b):
    kb = 1024
    mb = kb * 1024
    gb = mb * 1024
    tb = gb * 1024
    pb = tb * 1024

    if b >= pb:
        return '{:.2f}Pb'.format(b / pb)
    elif b >= tb:
        return '{:.2f}Tb'.format(b / tb)
    elif b >= gb:
        return '{:.2f}Gb'.format(b / gb)
    elif b >= mb:
        return '{:.2f}Mb'.format(b / mb)
    elif b >= kb:
        return '{:.2f}Kb'.format(b / kb)
    else:
        return '{:d}b'.format(b)


def human_dict(info):
    return json.dumps(info, indent=4, default=lambda o: o.hex())


def resource_info_fix_encoding(dct):
    new_dct = {}
    for key, value in dct.items():
        if isinstance(value, dict):
            value = resource_info_fix_encoding(value)
            new_dct[key.decode('ascii')] = value
        else:
            if isinstance(value, bytes) and key not in (b'pieces', b'md5sum'):
                value = value.decode('ascii')

            new_dct[key.decode('ascii')] = value

    return new_dct


@contextmanager
def timer():
    class TimerResult(object):
        __slots__ = 'spent',

    ts = time.perf_counter()
    result = TimerResult()

    try:
        yield result
    finally:
        result.spent = time.perf_counter() - ts


class TvmClient(object):
    def __init__(self, port, token):
        self._port = port
        self._token = token

    def _exception_handler(self, request, exception):
        raise exception

    def check_service_ticket(self, ticket):
        """
        Returns TVM id of ticket owner.
        """
        req = grequests.get(
            'http://localhost:{}/tvm/checksrv'.format(self._port),
            headers={
                'Authorization': self._token,
                'X-Ya-Service-Ticket': ticket
            },
            timeout=1
        )
        response = grequests.map([req], exception_handler=self._exception_handler)[0]
        response.raise_for_status()

        return response.json()['src']
