HTTP_SOURCE_ID = '0'


class Validator:
    def __init__(self):
        self.default_checks = {
            'links': True
        }

    def validate_head(self, head):
        # validate that head contains at least one file (can be either a sized file or a 0-size touch)
        if not head:
            return "Malformed request: head is missing", 400
        files = head.get('structure', None)
        if not files:
            return "Malformed head: structure is missing", 400
        valid = False
        try:
            for finfo in files.values():
                fileresource = finfo.get('resource', None)
                if not fileresource:
                    return "Malformed head: file has no resource", 400
                filetype = fileresource.get('type', None)
                if not filetype:
                    return "Malformed head: file has no type", 400

                if filetype == "torrent":
                    self.default_checks['links'] = True
                    return "", None
                if filetype == "touch":
                    self.default_checks['links'] = False
                    valid = True
        except Exception:
            return "Malformed head", 400

        if valid:
            return "", None
        else:
            return "Resource should contain at least one non-zero file or 0-sized touch file", 400

    def validate_http_ranges(self, link, header):
        # validates "Range: bytes=start,end" http-header
        # header variable should contain only the "bytes=start-end" part
        if header.startswith("bytes="):
            try:
                start, end = header[6:].split('-')
                start, end = int(start), int(end)
                if start <= end:
                    return "", None
            except Exception:
                return 'Malformed HTTP Range request %s for url %s' % (header, link), 400
        return 'Invalid HTTP Range request %s for url %s' % (header, link), 400

    def validate_linkopts(self, links):
        # validates link options from request
        supported_opts = ("Range",)
        for link, opts in links.items():
            for opt in opts.keys():
                if opt not in supported_opts:
                    return "Unsupported link option '%s' for link '%s'" % (opt, link), 400

        return "", None

    # takes in hashes part of incoming request and validates it for emptiness
    def validate_empty_links(self, incoming_info):
        if not self.default_checks['links']:
            return "", None

        # check if each md5 has a link
        for md5, links in incoming_info.items():
            if not links:
                return 'No links for md5: %s' % md5, 400

        return "", None

    def check_new_info(self, incoming_info, current_info, source_id):
        # checks if it is possible to update current resource with new info from the request
        # returns ("", None) if it cannot define the outcome of addition
        # returns ("Error text", 400) if request is somehow malformed
        # returns ("", 200) if oldstyle format is being updated with oldstyle format without source_id (LEGACY)

        # if we will not update anything
        if not self.default_checks['links']:
            return "", 200

        # check if info is not empty (sanity check)
        if not incoming_info:
            return 'No links and m5 hashes in request', 400

        # check if md5 pieces haven't changed
        if current_info.keys() != incoming_info.keys():
            return 'Request must contain exactly the same resource files', 400

        # evaluate which formats we are dealing with
        incoming_format_extended = isinstance(list(incoming_info.values())[0], dict)
        current_format_extended = isinstance(list(current_info.values())[0], dict)

        errortext, status = self.validate_empty_links(incoming_info)
        if status == 400:
            return errortext, status

        if incoming_format_extended:
            # incoming extended format does not update current oldstyle format
            if not current_format_extended:
                return 'Current resource has outdated format', 400

            # we do not accept extended format without source_id
            if not source_id:
                return 'No source_id supplied in request', 400

            for md5, links in current_info.items():
                for opts in links.values():
                    if opts[HTTP_SOURCE_ID] == source_id:
                        return 'Current resource already contains your source_id', 400

            return "", None

        elif current_format_extended:
            # updating extended format with legacy format is impossible without source_id
            if not source_id:
                return "", 200  # WARNING: temporary fix for SPI-35730 - allow sandbox to avoid retying 400
                # return 'No source_id supplied in request', 400

            for md5, links in current_info.items():
                for opts in links.values():
                    if opts[HTTP_SOURCE_ID] == source_id:
                        return 'Current resource already contains your source_id', 400

            return "", None

        else:
            # if source_id was provided, but current resource is outdated - 400
            # else we want to update announce table and return 200 - legacy behaviour
            if source_id:
                return 'Current resource has outdated format', 400

            return "", 200
