import msgpack
import logging
import platform
import requests
import time
import flask
import gevent
import gevent.lock
import gevent.pywsgi
import yaml

from flask import g

from collections import OrderedDict

from ..deblock import Deblock

from infra.skyboned.src.model import Resource
from infra.skyboned.src.utils import resource_info_fix_encoding, timer
from infra.skyboned.src.validate import Validator
from infra.skybit.src.component import Component
from infra.skybit.src.bencode import bdecode

from library.python.monlib.metric_registry import MetricRegistry
from library.python.monlib.encoder import dumps


CONTENT_TYPE_SPACK = 'application/x-solomon-spack'
CONTENT_TYPE_JSON = 'application/json'

HTTP_SOURCE_ID = '0'
HTTP_RANGES = '1'

REQUEST_TIME_MS = 1000


class Web(Component):
    def __init__(self, db, db_ro, dbt, tracked_db, tvm_client, hostname, sem_cap, parent=None):
        super(Web, self).__init__(parent=parent, logname='web')

        self.db = db
        self.db_ro = db_ro
        self.dbt = dbt
        self.tracked_db = tracked_db
        self.tvm_client = tvm_client
        self.http_app = flask.Flask(__name__)
        self.log = logging.getLogger('web')
        self.hostname = hostname

        self._ssl = False
        self._ssl_key = self._ssl_cert = None

        # Setup basic http
        self.http_app.add_url_rule(
            '/', 'index', self.web_index
        )
        self.http_app.add_url_rule(
            '/ping', 'ping', self.web_ping
        )

        self.http_app.add_url_rule(
            '/add_resource', 'add_resource', self.web_add_resource, methods=['POST']
        )

        self.http_app.add_url_rule(
            '/remove_resource', 'remove_resource', self.web_remove_resource, methods=['POST']
        )

        self.http_app.add_url_rule(
            '/metrics/solomon', 'metrics_solomon',
            self.web_metrics_solomon
        )

        self.http_app.after_request(self._web_after_request)

        yaml.add_representer(
            OrderedDict,
            lambda self, data: self.represent_mapping('tag:yaml.org,2002:map', data.items())
        )

        self._deblocker = Deblock(logger=self.log.getChild('deblock'), keepalive=None)

        self.log.info("Semaphore capacity is %d" % sem_cap)
        self._sem = {
            'add_resource': gevent.lock.Semaphore(sem_cap),
            'del_resource': gevent.lock.Semaphore(sem_cap),
        }

        self._metrics = MetricRegistry()
        self._db_counters_ts = 0
        self._db_tracking_ts = 0
        self._db_counters_update_interval = 300
        self._db_tracking_interval = 60

    def _get_service_ticket_owner(self, ticket):
        tvm_id = self.tvm_client.check_service_ticket(ticket)
        res = self.db_ro.query_one('SELECT id, name FROM owner WHERE tvm_id = %s LIMIT 1', (tvm_id, ))
        if res:
            owner, name = res
        else:
            name = str(tvm_id)
            owner = self.db.execute('INSERT INTO owner (name, tvm_id) VALUES (%s, %s) '
                                    'ON CONFLICT DO NOTHING '
                                    'RETURNING id',
                                    (name, tvm_id)).fetchone()[0]
        return owner, name

    def _web_after_request(self, response):
        response.headers['X-Forwarded-Host'] = self.hostname

        if flask.request.path in ('/ping', 'favicon.ico'):
            return response

        status = response.status_code
        self.http_app.logger.debug(
            '%s %s %s %s%s%s',
            flask.request.remote_addr, flask.request.method,
            flask.request.full_path, status,
            ' ' + response.get_data(as_text=True)[:100] if status >= 500 else '',
            ' ' + g.uid if hasattr(g, 'uid') else ''
        )
        return response

    def enable_ssl(self, ssl_key, ssl_cert):
        self._ssl = True
        self._ssl_key = ssl_key
        self._ssl_cert = ssl_cert

    def listen(self, http_port, https_port):
        self.log.info('Listening HTTP on *:%d', http_port)
        addr = ('', http_port)
        # self.http_server = gevent.pywsgi.WSGIServer(addr, self.http_app, log=None)
        s = gevent.socket.socket(gevent.socket.AF_INET6, gevent.socket.SOCK_STREAM)
        s.setsockopt(gevent.socket.SOL_SOCKET, gevent.socket.SO_REUSEPORT, 1)
        s.setsockopt(gevent.socket.SOL_SOCKET, gevent.socket.SO_REUSEADDR, 1)
        s.bind(addr)
        s.listen(128)
        self.http_server = gevent.pywsgi.WSGIServer(s, self.http_app, log=None)

        if self._ssl:
            self.log.info('Listening HTTPS on *:%s', https_port)
            self.http_server_ssl = gevent.pywsgi.WSGIServer(
                ('', self.https_port), self.http_app,
                keyfile=self._ssl_key, certfile=self._ssl_cert,
                log=None
            )

    def start(self):
        gevent.spawn(self.http_server.serve_forever)

        if self._ssl:
            gevent.spawn(self.http_server_ssl.serve_forever)

    def _header(self):
        response = ['<body>']
        response.append('<style type="text/css">')
        response.append('body { font-family: Verdana; font-size: 0.7em; background: #222; color: white }')
        response.append('a { color: orange; font-weight: bold }')
        response.append('a:hover { color: yellow }')
        response.append('hr { border-color: #735f39 }')
        response.append('</style>')
        response.append('<pre>')

        response.append('Hi, world! (skyboned v0.2)')
        response.append('Current node: %s' % (platform.node(), ))

        # response.append('')
        # response.append(
        #     '<a href="/">uploads</a> | '
        #     '<a href="/evoq">evoq</a> | '
        #     '<a href="/stats">stats</a>'
        #     '<hr/>'
        # )

        response.append('<hr/>')

        return '\n'.join(response)

    def web_index(self):
        response = [self._header()]

        response.append('Web UI working')

        response.append('</body>')

        return '\n'.join(response)

    def web_ping(self):
        return ''

    def web_hostname(self):
        return platform.node()

    def _web_metric(self, name, owner, status, success=None):
        self._metrics.rate({'name': name, 'owner': owner, 'status': str(status), 'success': 'yes' if success else 'no'}).inc()

    def _countable_metric(self, name, owner, value):
        self._metrics.gauge({'name': name, 'owner': owner}).set(value)

    def web_add_resource(self):
        uid = flask.request.json['uid']
        if uid.startswith('rbtorrent:'):
            uid = uid.split(':', 1)[1]
        g.uid = uid

        owner = 0
        owner_name = 'unknown'

        tvm_ticket = flask.request.headers.get('X-Ya-Service-Ticket', None)
        if not tvm_ticket:
            self._web_metric('api_resource_add', owner_name, 403)
            return 'Forbidden: TVM Service Ticket required', 403

        try:
            owner, owner_name = self._get_service_ticket_owner(tvm_ticket)
        except requests.RequestException as ex:
            if isinstance(ex, requests.HTTPError) and ex.response.status_code == 403:
                self._web_metric('api_resource_add', owner_name, 403)
                return 'Forbidden: invalid TVM Service Ticket', 403
            else:
                self._web_metric('api_resource_add', owner_name, 503)
                return 'Failed to check TVM Service Ticket', 503
        except Exception as ex:
            self._web_metric('api_resource_add', owner_name, 500)
            self.log.error('Unexpected exception while checking TVM ticket: %r', ex)
            return 'Failed to check TVM Service Ticket', 500

        if not self._sem['add_resource'].acquire(blocking=False):
            self._web_metric('api_resource_add', owner_name, 503)
            return 'Too much add_resource queries', 503

        try:
            now = int(time.time())

            request_validator = Validator()

            info = flask.request.json['info']
            if not isinstance(info, dict):
                flask.abort('Malformed links')

            mode = flask.request.json['mode']
            if mode != 'plain':
                flask.abort('Only plain mode supported as of now')

            head = bytes.fromhex(flask.request.json['head'])
            head = bdecode(head)
            head = resource_info_fix_encoding(head)
            errortext, status = request_validator.validate_head(head)
            if status == 400:
                self._web_metric('api_resource_add', owner_name, 400)
                return errortext, status

            source_id = flask.request.json['source_id'] if 'source_id' in flask.request.json else None

            # early skip of existing resource, source_id logic doesn't work
            exist = self.db_ro.query_one_col('SELECT id FROM resource WHERE id = %s', (uid, ))
            if exist == uid:
                self._web_metric('api_resource_add', owner_name, 200, True)
                return flask.jsonify({'success': True})

            with self.dbt:
                self.dbt.execute('DELETE FROM announce_op WHERE resource = %s', (uid, ))

                with timer() as t:
                    current_info = self.db_ro.query_one_col('SELECT info FROM resource WHERE id = %s', (uid, ))
                self._countable_metric('api_db_resource_select_ms', owner_name, t.spent * REQUEST_TIME_MS)

                if not current_info:
                    # no current resource in db, clean insert
                    errortext, status = request_validator.validate_empty_links(info)
                    if status:
                        self._web_metric('api_resource_add', owner_name, 400)
                        return errortext, status

                    for md5, link in info.items():

                        if isinstance(link, dict):
                            # links format is extended, therefore source_id must be supplied
                            if not source_id:
                                self._web_metric('api_resource_add', owner_name, 400)
                                return 'No source_id supplied in request', 400

                            errortext, status = request_validator.validate_linkopts(link)
                            if status == 400:
                                self._web_metric('api_resource_add', owner_name, 400)
                                return errortext, status

                            for lnk in link.keys():
                                info[md5][lnk][HTTP_SOURCE_ID] = source_id
                                ranges = info[md5][lnk].get('Range', None)
                                if ranges is not None:
                                    # ranges are supplied as "Range" : "bytes=100-300"
                                    # "Range" is replaced with "1"
                                    errortext, status = request_validator.validate_http_ranges(lnk, ranges)
                                    if status == 400:
                                        self._web_metric('api_resource_add', owner_name, 400)
                                        return errortext, status
                                    info[md5][lnk][HTTP_RANGES] = ranges
                                    del info[md5][lnk]['Range']
                        elif source_id:
                            # oldstyle links format is extended only if source_id is supplied
                            info[md5] = {}
                            if isinstance(link, str):
                                link = [link]
                            for lnk in link:
                                info[md5][lnk] = {HTTP_SOURCE_ID : source_id}

                    # Note, we are using dbt as db in Resource here
                    resource = Resource(self.dbt, self.dbt, self.dbt, uid, head, info, mode, owner, now, 0)

                    with timer() as t:
                        resource.save()
                    self._countable_metric('api_db_resource_insert_ms', owner_name, t.spent * REQUEST_TIME_MS)
                else:
                    self._web_metric('api_resource_add', owner_name, 200, True)
                    return flask.jsonify({'success': True})

                    current_info = msgpack.unpackb(current_info)

                    # new info validation
                    errortext, status = request_validator.check_new_info(info, current_info, source_id)
                    if status == 200:
                        self._web_metric('api_resource_add', owner_name, 200, True)
                        return flask.jsonify({'success': True})

                    if status == 400:
                        self._web_metric('api_resource_add', owner_name, 400)
                        return errortext, status

                    if status is None:
                        # add incoming lnks to current_links
                        # we return 400 if any link from the incoming is already in current
                        for md5, link in info.items():
                            current_links = current_info[md5]

                            # check if incoming links are in extended format
                            links = link
                            extension = False
                            if isinstance(link, dict):
                                errortext, status = request_validator.validate_linkopts(link)
                                if status == 400:
                                    self._web_metric('api_resource_add', owner_name, 400)
                                    return errortext, status
                                links = link.keys()
                                extension = True
                            elif isinstance(link, str):
                                link = [link]

                            for lnk in links:
                                if lnk in current_links:
                                    self._web_metric('api_resource_add', owner_name, 400)
                                    return 'Requst must contain only new links', 400
                                current_links[lnk] = {}
                                current_links[lnk][HTTP_SOURCE_ID] = source_id
                                if extension:
                                    ranges = link[lnk].get('Range', None)
                                    if ranges is not None:
                                        # ranges are supplied as "Range" : "bytes=100-300"
                                        # "Range" is replaced with "1"
                                        errortext, status = request_validator.validate_http_ranges(lnk, ranges)
                                        if status == 400:
                                            self._web_metric('api_resource_add', owner_name, 400)
                                            return errortext, status
                                        current_links[lnk][HTTP_RANGES] = ranges

                            # TODO: update ts_used later
                            with timer() as t:
                                self.db.execute(
                                    'UPDATE resource SET info = %s WHERE id = %s', (msgpack.packb(current_info, use_bin_type=True), uid)
                                )
                            self._countable_metric('api_db_resource_update_ms', owner_name, t.spent * REQUEST_TIME_MS)

                    elif status == 400:
                        self._web_metric('api_resource_add', owner_name, 400)
                        return errortext, status
                    # legacy case with 200 falls through to update announce table

                # Insert missing announce records if any
                with timer() as t:
                    self.dbt.execute(
                        'INSERT INTO announce (resource_id, schedule_ts, tracker_id) '
                        'SELECT %s, 0, id as tracker_id '
                        '   FROM announce_tracker '
                        '   ON CONFLICT DO NOTHING',
                        (uid, )
                    )
                self._countable_metric('api_db_announce_insert_ms', owner_name, t.spent * REQUEST_TIME_MS)

                self.dbt.execute('NOTIFY announce_new')

            # wait at least one announce
            deadline = now + 300

            while time.time() < deadline:
                gevent.sleep(0.3)
                with timer() as t:
                    result = self.db_ro.query_one_col(
                        'SELECT COUNT(*) FROM announce '
                        'WHERE resource_id = %s AND schedule_ts != 0',
                        (uid, )
                    )
                self._countable_metric('api_db_announce_select_ms', owner_name, t.spent * REQUEST_TIME_MS)
                if result:
                    break
            else:
                self.log.error('Unable to wait success announce')
                self._web_metric('api_resource_add', owner_name, 200, False)
                return flask.jsonify({'success': False, 'error': 'Timeout waiting tracker announce'})

            self._web_metric('api_resource_add', owner_name, 200, True)
            return flask.jsonify({'success': True})

        except BaseException:
            self._web_metric('api_resource_add', owner_name, 500)
            raise

        finally:
            self._sem['add_resource'].release()

    def web_remove_resource(self):
        owner = 0
        owner_name = 'unknown'

        tvm_ticket = flask.request.headers.get('X-Ya-Service-Ticket', None)
        if not tvm_ticket:
            self._web_metric('api_resource_remove', owner_name, 403)
            return 'Forbidden: TVM Service Ticket required', 403

        try:
            owner, owner_name = self._get_service_ticket_owner(tvm_ticket)
        except requests.RequestException as ex:
            if isinstance(ex, requests.HTTPError) and ex.response.status_code == 403:
                self._web_metric('api_resource_remove', owner_name, 403)
                return 'Forbidden: invalid TVM Service Ticket', 403
            else:
                self._web_metric('api_resource_remove', owner_name, 503)
                return 'Failed to check TVM Service Ticket', 503
        except Exception as ex:
            self._web_metric('api_resource_remove', owner_name, 500)
            self.log.error('Unexpected exception while checking TVM ticket: %r', ex)
            return 'Failed to check TVM Service Ticket', 500

        if not self._sem['del_resource'].acquire(blocking=False):
            self._web_metric('api_resource_remove', owner_name, 503)
            return 'Too much remove_resource queries', 503

        try:
            uid = flask.request.json['uid']
            # source_id = flask.request.json['source_id'] if 'source_id' in flask.request.json else None # Temporary exclusion of source_id logic from runtime

            if uid.startswith('rbtorrent:'):
                uid = uid.split(':', 1)[1]
            g.uid = uid

            resource = Resource(self.db, self.db_ro, self.dbt, uid)

            with timer() as t:
                result = resource.load()
            self._countable_metric('api_db_resource_select_ms', owner_name, t.spent * REQUEST_TIME_MS)

            if not result:
                # resource doesn't exist
                self._web_metric('api_resource_remove', owner_name, 200, True)
                return flask.jsonify({'success': True})

            # hack to allow sandbox (id=1, tvm_id=2002826) and sandbox-testing (id=3, tvm_id=2011100) delete resources from each other
            if owner in (1, 3) and resource.owner in (1, 3):
                pass
            elif resource.owner != owner:
                self._web_metric('api_resource_remove', owner_name, 403)
                return 'Forbidden: TVM Service Ticket doesn\'t match resource owner', 403

            """
            # Temporary exclusion of source_id logic from runtime
            if not source_id:
                resource.info = {}
            else:
                links_to_remove = []

                for md5, links in resource.info.items():
                    if not isinstance(links, dict):
                        resource.info = {}
                        break

                    for link, info in links.items():
                        if info[HTTP_SOURCE_ID] == source_id:
                            # Do not remove elements during iteration
                            links_to_remove.append((md5, link))

                for md5, link in links_to_remove:
                    del resource.info[md5][link]
                    if not resource.info[md5]:
                        resource.info = {}
                        break

            if resource.info:
                with timer() as t:
                    self.db.execute(
                        'UPDATE resource SET info = %s WHERE id = %s', (msgpack.packb(resource.info, use_bin_type=True), uid)
                    )
                self._countable_metric('api_db_resource_update_ms', owner_name, t.spent * REQUEST_TIME_MS)
            else:
            """
            with self.dbt:
                self.dbt.execute('DELETE FROM resource WHERE id = %s', (uid, ))
                self.dbt.execute(
                    'INSERT INTO announce_op (resource, tracker_id, deadline, op) '
                    'SELECT %s, id as tracker_id, %s, %s'
                    '   FROM announce_tracker '
                    '   ON CONFLICT DO NOTHING',
                    (uid, int(time.time() + 86400*2), 'remove')
                )
                # Do not notify announce_new here, bc we dont want announce loop to announce "remove" requests asap

            self._web_metric('api_resource_remove', owner_name, 200, True)
            return flask.jsonify({'success': True})

        except BaseException:
            self._web_metric('api_resource_remove', owner_name, 500)
            raise

        finally:
            self._sem['del_resource'].release()

    def web_metrics_solomon(self):
        now = time.time()

        if self.tracked_db and now - self._db_tracking_ts > self._db_tracking_interval:
            self._db_tracking_ts = now
            tracked_db_name = self.tracked_db.host.split('.')[0]
            announce_cnt = self.tracked_db.query_one_col('SELECT COUNT(*) FROM announce')
            self._metrics.gauge({'name': 'db_count_announce', 'db_name': tracked_db_name}).set(announce_cnt)
            announce_op_cnt = self.tracked_db.query_one_col('SELECT COUNT(*) FROM announce_op')
            self._metrics.gauge({'name': 'db_count_announce_op', 'db_name': tracked_db_name}).set(announce_op_cnt)
            resource_cnt = self.tracked_db.query_one_col('SELECT COUNT(*) FROM resource')
            self._metrics.gauge({'name': 'db_count_resource', 'db_name': tracked_db_name}).set(resource_cnt)

        if self._db_counters_update_interval and now - self._db_counters_ts > self._db_counters_update_interval:
            self._db_counters_ts = now

            resource_cnt_by_owner = {}

            resource_cnt_ts = self.db_ro.query_one_col('SELECT resource_cnt_ts FROM owner WHERE id = 1')
            if now - resource_cnt_ts <= self._db_counters_update_interval:
                for owner, resource_cnt in self.db_ro.query('SELECT name, resource_cnt FROM owner'):
                    resource_cnt_by_owner[owner] = resource_cnt
            else:
                for owner, resource_cnt in self.db_ro.query(
                    'SELECT o.name, COUNT(r.id) '
                    'FROM resource r RIGHT JOIN owner o ON r.owner = o.id '
                    'GROUP BY o.id'
                ):
                    resource_cnt_by_owner[owner] = resource_cnt
                with self.dbt:
                    for owner, resource_cnt in resource_cnt_by_owner.items():
                        self.dbt.execute('UPDATE owner SET resource_cnt = %s WHERE name = %s', (resource_cnt, owner, ))
                    self.dbt.execute('UPDATE owner SET resource_cnt_ts = %s WHERE id = 1', (now, ))

            for owner, resource_cnt in resource_cnt_by_owner.items():
                self._metrics.gauge({'name': 'resource', 'owner': owner}).set(resource_cnt)

            for tracker_address, overdue_cnt in self.db_ro.query(
                'SELECT at.address, COUNT(a.resource_id) '
                'FROM announce a JOIN announce_tracker at ON at.id = a.tracker_id '
                'WHERE a.schedule_ts < %s '
                'GROUP BY at.id',
                (time.time() - 60, )  # offset by 60 sec to reduce rattle
            ):
                if ':' in tracker_address:
                    # cut port number
                    tracker_address = tracker_address.rsplit(':', 1)[0]

                self._metrics.gauge({'name': 'announce_overdue', 'tracker_address': tracker_address}).set(overdue_cnt)

        if flask.request.headers['accept'].find(CONTENT_TYPE_SPACK) >= 0:
            return flask.Response(dumps(self._metrics), mimetype=CONTENT_TYPE_SPACK)

        return flask.Response(dumps(self._metrics, format='json'), mimetype=CONTENT_TYPE_JSON)
