#!/usr/bin/env python3

import argparse
import hashlib
import json
import os

import infra.skyboned.api


def try_parse_json(s):
    try:
        return json.loads(s)
    except ValueError:
        return None


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument('action', choices=['add', 'del'])
    parser.add_argument('target')
    parser.add_argument('--link')
    parser.add_argument('--server')
    parser.add_argument('--source-id')
    parser.add_argument('--tvm-ticket')
    parser.add_argument('--tvm-id')
    parser.add_argument('--tvm-secret')
    return parser.parse_args()


def calc_hashes(filename):
    md5_hash = hashlib.md5()
    sha1_hashes = []
    size = 0
    with open(filename, 'rb') as f:
        for piece in iter(lambda: f.read(4 * 1024 * 1024), b''):
            md5_hash.update(piece)
            sha1_hashes.append(hashlib.sha1(piece).digest())
            size += len(piece)
    return (md5_hash.hexdigest(), sha1_hashes, size)


def add_resource(args):
    j = try_parse_json(args.target)
    if j is not None:
        items = j['items']
        hashes = {md5_hash: bytes.fromhex(sha1_hashes) for md5_hash, sha1_hashes in j['hashes'].items()}
        links = j['links']
        print(infra.skyboned.api.skyboned_add_resource(items, hashes, links, args.server, args.tvm_ticket, args.source_id)[0])
        return

    filename = args.target
    basename = os.path.basename(filename)

    md5_hash, sha1_hashes, size = calc_hashes(filename)
    items = {
        basename: {
            'type': 'file',
            'md5': md5_hash,
            'executable': False,
            'size': size
        }
    }
    hashes = {
        md5_hash: b''.join(sha1_hashes)
    }
    links = {
        md5_hash: args.link or 'http://localhost/skyboned/files/' + basename
    }
    result = {'items': items, 'hashes': {md5_hash: sha1_hashes.hex() for md5_hash, sha1_hashes in hashes.items()}, 'links': links}
    #print(json.dumps(result, separators=(',', ':')))
    if args.server:
        print(infra.skyboned.api.skyboned_add_resource(items, hashes, links, (args.server,), args.tvm_ticket, args.source_id)[0])
    else:
        print(infra.skyboned.api.skyboned_generate_resource(items, hashes)[0])


def remove_resource(args):
    uid = args.target

    infra.skyboned.api.skyboned_remove_resource(uid, (args.server,), args.tvm_ticket, args.source_id)


def main():
    args = parse_args()

    if not args.tvm_ticket:
        import tvmauth
        tvm_cache_path = '/tmp/tvm_cache/'
        if not os.path.exists(tvm_cache_path):
            os.makedirs(tvm_cache_path)
        tvm_client = tvmauth.TvmClient(
            tvmauth.TvmApiClientSettings(
                self_tvm_id=int(args.tvm_id),
                self_secret=args.tvm_secret,
                dsts={'skyboned': 2021848, "datasync": 2000060},
                disk_cache_dir=tvm_cache_path,
            )
        )
        args.tvm_ticket = tvm_client.get_service_ticket_for('skyboned')

    if args.action == 'add':
        add_resource(args)
    else:
        remove_resource(args)


if __name__ == '__main__':
    main()
