import os
import yaml
import signal
import time
import errno
from ..framework.component import Component


CGROUPS_CHECK_PERIOD = 1800


class CgroupsController(Component):
    def __init__(self, parent=None, workdir='', registry=None, log=None, lock=None):
        logname = 'cgroups'
        super(CgroupsController, self).__init__(logname=logname, parent=parent)

        self.cgroup_config_filepath = os.path.join(workdir, 'skycore.cgroups')
        self.registry = registry
        self.process_lock = lock
        self.cgroup_root_path = None

        self.valid_cgroups = []
        self.known_cgroups = set()
        if log:
            self.log = log.getChild(logname)

        self.registry.subscribe(self._cgroups_changed, [('sys', 'cgroups')], config_only=True)

    def state(self):
        with self.process_lock:
            return {
                'cgroup_root_path': self.cgroup_root_path,
                'valid_cgroups': self.valid_cgroups,
                'known_cgroups': list(self.known_cgroups)
            }

    @Component.green_loop(logname='checker')
    def _check_cgroups_loop(self, log):
        # check skycore cgroups
        start = time.time()

        with self.process_lock:
            # reset cgroups
            self.valid_cgroups = []
            self.known_cgroups = set()
            log.debug('Checking skycore cgroups ...')
            self._update_skycore_cgroups(log)

            log.debug('Checking service cgroups ...')
            # check other cgroups
            config = self.registry.query(['sys', 'cgroups'], deepcopy=True)
            self._update_cgroups(config['config'], log=log, details=False)

        log.debug('Done {}.'.format(time.time() - start))
        return CGROUPS_CHECK_PERIOD

    def _cgroups_changed(self, _, config):
        self.log.debug('CGROUPS config changed')

        with self.process_lock:
            # remove config hash from the config
            config.pop('__config_hash', None)
            # reset cgroups
            self.valid_cgroups = []
            self.known_cgroups = set()
            self._update_cgroups(config)

        return True

    def _update_skycore_cgroups(self, log):
        try:
            with open(self.cgroup_config_filepath, 'rb') as f:
                configuration = yaml.load(f, Loader=getattr(yaml, 'CSafeLoader', yaml.SafeLoader))
                cgroup_path = configuration.pop('__cgroup_path')
        except Exception as ex:
            log.warning('Failed to load config from %r', self.cgroup_config_filepath)
            log.warning(str(ex))
        else:
            for controller, config in configuration.items():
                log.debug('  Controller: %s', controller)
                controller_path = os.path.join(cgroup_path, controller)
                skip = False
                if not os.path.exists(controller_path) or not os.path.isdir(controller_path):
                    skip = True
                    log.debug('    not exists, skipping')

                for root_cgroup, cfg in config.items():
                    self._update_known_cgroups(
                        root_cgroup,
                        cfg,
                    )

                    if not skip:
                        self._update_cgroups_tree(
                            log,
                            os.path.join(controller_path, root_cgroup),
                            cfg,
                            no_drop=True)

    def _update_cgroups(self, cgroups_config, log=None, details=True):
        log = log or self.log

        if details:
            log.debug('Raw cgroups config:')
            for line in yaml.dump(
                cgroups_config,
                default_flow_style=0,
                Dumper=getattr(yaml, 'CSafeDumper', yaml.SafeDumper)
            ).split('\n'):
                log.debug(line)

        config_version = cgroups_config['options']['config_version']
        assert config_version == 1

        self.cgroup_root_path = cgroups_config['options']['path']
        if not os.path.exists(self.cgroup_root_path) or not os.path.isdir(self.cgroup_root_path):
            # there is no cgroups, but it's OK
            return True

        cgroups_config.pop('options')
        config = self._convert_cgroups_config_to_machine_form(cgroups_config)

        if details:
            log.debug('Converted cgroups config:')
            for line in yaml.dump(
                config,
                default_flow_style=0,
                Dumper=getattr(yaml, 'CSafeDumper', yaml.SafeDumper)
            ).split('\n'):
                log.debug(line)

        log.debug('Apply cgroups:')

        for controller, controller_config in config.items():
            log.debug('  Controller: %s', controller)
            controller_path = os.path.join(self.cgroup_root_path, controller)

            skip = False
            if not os.path.exists(controller_path) or not os.path.isdir(controller_path):
                skip = True
                log.debug('    not exists, skipping')

            for root_cgroup, cgroup_config in sorted(controller_config.items()):
                self._update_known_cgroups(
                    root_cgroup,
                    cgroup_config,
                )
                if not skip:
                    self._update_cgroups_tree(
                        log,
                        os.path.join(controller_path, root_cgroup),
                        cgroup_config,
                        log_indent=2
                    )

        log.debug('Success!')

        return True

    def _mapping_deep_update(self, mapping, mapping2):
        """ Deeply update one dict with another. """

        for key, value in mapping2.iteritems():
            if isinstance(value, dict):
                mapping[key] = self._mapping_deep_update(mapping.get(key, {}), value)
            else:
                mapping[key] = value
        return mapping

    def _convert_cgroups_config_to_machine_form(self, config, root='', paths=set()):
        machine_config = {}

        if root:
            paths.add(root)

        for item, value in config.items():
            if item.startswith('cgroup:'):
                item_root = os.path.join(root, item.split(':', 1)[1])
                value = self._convert_cgroups_config_to_machine_form(value, item_root, paths)

                for ctrl, ctrl_paths in value.items():
                    for path, cfg in ctrl_paths.items():
                        current_path_cfg = machine_config.setdefault(ctrl, {})

                        for path_part in path.split('/'):
                            current_path_cfg = current_path_cfg.setdefault(path_part, {})

                        self._mapping_deep_update(current_path_cfg, cfg)
            else:
                machine_config.setdefault(item, {})[root] = value

        if root == '':
            for ctrl, paths_cfg in machine_config.items():
                for path in sorted(paths):
                    stack = paths_cfg
                    for path_part in path.split('/'):
                        stack = stack.setdefault(path_part, {})

        return machine_config

    def _create_cgroup(self, path):
        os.mkdir(path)

    def _update_cgroups_tree(self, log, root, config, level=0, log_indent=0, no_drop=False):
        log_indent_prefix = '  ' * (log_indent + level)

        for tryout in ([0, 1]) if level == 0 else [1]:
            try:
                log.debug(log_indent_prefix + root)

                if not os.path.exists(root):
                    log.debug(log_indent_prefix + 'creating')
                    try:
                        self._create_cgroup(root)
                        # successfully created cgroup
                        self.valid_cgroups.append(root)
                    except OSError as ex:
                        if ex.errno == errno.EROFS:
                            log.warning(
                                log_indent_prefix + 'failed to create "%s": read-only file system', root
                            )
                            break
                        else:
                            raise
                else:
                    self.valid_cgroups.append(root)

                for key, value in sorted(config.items(), key=lambda kv: [isinstance(kv[1], dict), kv]):
                    fpath = os.path.join(root, key)

                    if isinstance(value, dict):
                        self._update_cgroups_tree(
                            log,
                            os.path.join(root, key),
                            value,
                            level=level + 1,
                            log_indent=log_indent,
                            no_drop=no_drop
                        )
                        continue

                    try:
                        with open(fpath, mode='wb') as fp:
                            value = str(value)
                            log.debug(log_indent_prefix + 'set %s to %s', fpath, value)
                            fp.write(value)
                    except IOError as ex:
                        log.error(log_indent_prefix + 'failed to set %s: %s', fpath, ex)

            except BaseException as ex:
                if tryout == 0 and not no_drop:
                    log.debug(log_indent_prefix + 'failed: %s, drop and retry', ex)
                    self._drop_cgroup_tree(log, root, log_indent_prefix)
                else:
                    raise

            else:
                break

    def _update_known_cgroups(self, root, config):
        self.known_cgroups.add(root)

        for key, value in sorted(config.items(), key=lambda kv: [isinstance(kv[1], dict), kv]):
            if isinstance(value, dict):
                self._update_known_cgroups(os.path.join(root, key), value)

    def _drop_cgroup_tree(self, log, path, log_prefix):
        log.debug(log_prefix + 'dropping tree at %s', path)

        if not os.path.exists(path):
            log.debug(log_prefix + 'tree nonexistent at %s', path)
            return

        for sub in os.listdir(path):
            subpath = os.path.join(path, sub)
            if os.path.isdir(subpath):
                self._drop_cgroup_tree(log, subpath, log_prefix + '  ')

        tasks = os.path.join(path, 'tasks')

        deadline = time.time() + 30

        while True:
            for sig in (signal.SIGSTOP, signal.SIGKILL, signal.SIGCONT):
                for pid in open(tasks, mode='rb').read().split():
                    if not pid:
                        continue
                    log.debug(log_prefix + 'killing pid %r (signal %r)', pid, sig)
                    try:
                        os.kill(int(pid), sig)
                    except OSError as ex:
                        log.warning(log_prefix + 'error with pid %r (signal %r): %s', pid, sig, ex)

            pids = open(tasks, mode='rb').read()
            log.debug(log_prefix + 'left %d pids', len(pids.split()))

            if time.time() > deadline or not pids:
                if not pids:
                    log.debug(log_prefix + 'rmdir %s', path)
                    os.rmdir(path)
                break

            time.sleep(1)

    def get_valid_cgroups(self, name):
        result = []

        with self.process_lock:
            # it's OK if cgroups do not exist
            if name and os.path.exists(self.cgroup_root_path) and os.path.isdir(self.cgroup_root_path):
                for controller in os.listdir(self.cgroup_root_path):
                    dest_dir = os.path.join(self.cgroup_root_path, controller, name)
                    if dest_dir in self.valid_cgroups:
                        dest = os.path.join(dest_dir, 'tasks')
                        if os.path.exists(dest) and os.path.isfile(dest):
                            result.append(dest_dir)

                if not result and name not in self.known_cgroups:
                    raise Exception("cgroup %r not found in any controller nor in cgroup config" % (name,))

        return result
