import os
import sys
import time
import copy

from collections import defaultdict

import six
import yaml
import gevent

try:
    from gevent.coros import RLock
except ImportError:
    from gevent.lock import RLock

from ..framework.utils import Path, detect_hostname
from ..framework.component import Component
from ..genisys import download_config, send_report


CONFIG_UPDATE_PERIOD = 300
CONFIG_APPLY_TIMEOUT = 120  # 2 minutes because we can spend time downloading large archives
STATUS_RESEND_PERIOD = (15 * 60)


class ConfigSubscription(object):
    def __init__(self, cfg, paths, callback, config_only=False):
        super(ConfigSubscription, self).__init__()
        self._section_hashes = {}
        self._paths = paths
        self._callback = callback
        self._config_only = config_only
        self._notifying = False
        self._cfg = cfg

    @property
    def callback(self):
        return self._callback

    def add_section(self, section_name, section_hash):
        self._section_hashes[section_name] = section_hash

    def notify(self, section_name, section_hash):
        if self._section_hashes[section_name] and self._section_hashes[section_name] == section_hash:
            return None

        # update section config hash
        self._section_hashes[section_name] = section_hash

        if not self._notifying:
            self._notifying = True
            return gevent.spawn(self.do_notification)

        return None

    def do_notification(self):
        new_config = {}
        for path in self._paths:
            try:
                value = self._cfg.query(path)
            except LookupError:
                # missing section, nothing to return yet
                continue

            if self._config_only:
                # extract config section only, no metadata required
                config = value['config']
                if config:
                    config['__config_hash'] = value['config_hash']
                self._callback(path, config)
            else:
                # prepare dict for all subscribed sections
                p = new_config
                for section in path:
                    if 'subsections' not in p:
                        p['subsections'] = {}
                    if section not in p['subsections']:
                        p['subsections'][section] = {}

                    if section == path[-1]:
                        p['subsections'][section] = value
                    else:
                        p = p['subsections'][section]

        if new_config and not self._config_only:
            self._callback(None, new_config)
        self._notifying = False


class ConfigUpdater(Component):
    def __init__(self,
                 deblock,
                 parent=None,
                 config_dir='',
                 filename='',
                 hostname=None,
                 log=None,
                 paused=False,
                 ):
        logname = 'cfgupd '
        super(ConfigUpdater, self).__init__(logname=logname, parent=parent)

        if log:
            self.log = log.getChild(logname)

        self.deblock = deblock
        self.send_report_lock = RLock()
        self.config_dir = Path(config_dir)
        self.filename = filename
        self.hostname = hostname
        self.paused = paused
        self.overrides = []
        self._check_overrides(self.log)
        self.data = self._apply_overrides_deblocked(self._load_config_deblocked())
        self.config_hash = self.data.get('config_hash', None) if self.data else None
        self.snapshot_revision = self.data.get('snapshot_revision', None) if self.data else None
        self.last_modified = None
        self.subscriptions = defaultdict(list)
        self.last_report_time = 0
        self.last_report = True
        self.config = {}

        self._update_own_config()
        self.log.info('Config directory: %r', self.config_dir.strpath)

    def subscribe(self, callback, paths, config_only=False):
        """
        :param Callable callback: function(new_config)
        :param list paths: sections to subscribe
        :param bool config_only: provide meta info in callback or not
        """

        subscription = ConfigSubscription(self, paths, callback, config_only)

        for path in paths:
            section_name = '.'.join(path)
            try:
                cfg = self.query(path)
            except LookupError:
                cfg = None
            subscription.add_section(section_name, cfg.get('config_hash', None) if cfg else None)
            self.subscriptions[section_name].append(subscription)
            self.log.info(
                'Subscription for %s [%s] added (%r)',
                section_name,
                'config only' if config_only else 'with metainfo',
                callback
            )

        subscription.do_notification()

    def unsubscribe(self, callback):
        self.log.info('Subscription (%r) removed', callback)
        for subscriptions in self.subscriptions.itervalues():
            for s in subscriptions:
                if s.callback == callback:
                    subscriptions.remove(s)
                    break

    def state(self):
        return {
            'paused': self.paused,
            'config_hash': self.config_hash,
            'last_modified': self.last_modified,
            'snapshot_revision': self.snapshot_revision,
            'config': self.config,
            'subscriptions': [
                '{} subscription(s) for {}'.format(len(value), key) for key, value in self.subscriptions.items()
            ],
            'overrides': [
                '{}:{}'.format(file_name, mtime) for (file_name, mtime) in self.overrides
            ] if self.overrides else None
        }

    def query(self, path, data=None, deepcopy=True):
        if data is None:
            self.log.debug('Query configuration for %s', path)
            if not self.data:
                raise LookupError("Config is not ready")
            data = self.data

        if data:
            for p in path:
                data = data['subsections'][p]

        return copy.deepcopy(data) if deepcopy else data

    @Component.green_loop(logname='updater')
    def _check_update_loop(self, log):
        try:
            self.update_config(log)
        except Exception as e:
            log.exception("config update failed: %s" % (e,), exc_info=sys.exc_info())

        return self.config.get('update_period', CONFIG_UPDATE_PERIOD)

    def update_config(self, log=None, forced=False):
        log = log or self.log

        if self.paused and not forced:
            # config update is paused -> do not request new config from genisys
            log.warning('Configuration update is paused!')
            report_hostname = ""
            last_modified = None
            data = None
        else:
            # detect hostname
            hostname = (self.hostname
                        or os.getenv('SKYCORE_CONFIG_HOSTNAME')
                        or self._load_hostname_deblocked(log)
                        or detect_hostname())
            report_hostname = (self.hostname
                               or os.getenv('SKYCORE_CONFIG_REPORT_HOSTNAME')
                               or detect_hostname()
                               )
            log.debug('Updating configuration for %s', hostname)

            (last_modified, data) = self.deblock.apply(
                download_config,
                hostname, log, self.last_modified, self.config_hash
            )

        # check for overrides
        new_overrides = self._check_overrides(log)

        if data:
            self.config_hash = data['config_hash']
            self.snapshot_revision = data.get('snapshot_revision', None)
            self.last_modified = last_modified

            success = True
            exc_text = ''
            try:
                self.data = self._apply_overrides_deblocked(data, log)
                self._apply_config(self.data, log)
            except Exception as e:
                log.exception("Config apply failed: %s" % (e,), exc_info=sys.exc_info())
                success = False
                exc_text = str(e)
            else:
                self._save_config_deblocked(data, log)  # we store vanilla config, not overriden!
                log.debug('New config loaded, applied and stored:'
                          '\nconfig_hash: %s'
                          '\nlast_modified: %r'
                          '\nsnapshot_revision: %s',
                          self.config_hash,
                          self.last_modified if self.last_modified else 'n/a',
                          self.snapshot_revision)

            self.send_report(log, success, exc_text, report_hostname, True)
        elif new_overrides:
            # config is not changed, but overrides are
            try:
                # first load vanilla config
                data = self._load_config_deblocked(log)
                self.data = self._apply_overrides_deblocked(data, log)
                self._apply_config(self.data, log)
            except Exception as e:
                log.exception("Overrides apply failed: %s" % (e,), exc_info=sys.exc_info())

        if not self.data:
            # failed to download config -> try to load last config from the file
            self.data = self._apply_overrides_deblocked(
                self._load_config_deblocked(log),
                log
            )

        self._update_own_config(log)

    def send_report(self, log, success, description, hostname=None, forced=False):
        with self.send_report_lock:
            cur_time = time.time()
            if forced or \
                    cur_time - self.last_report_time >= self.config.get('resend_period', STATUS_RESEND_PERIOD) or \
                    self.last_report != success:
                hostname = hostname or self.hostname or detect_hostname()
                log.info("Send report (%d): %s", cur_time - self.last_report_time, 'success' if success else 'error')
                self.deblock.apply(send_report, hostname, log, self.snapshot_revision, success, description)

                self.last_report_time = cur_time
                self.last_report = success

    def _save_config(self, data, log=None):
        log = log or self.log

        # store new config to the file and update simlink
        actpath = self.config_dir.join(self.filename)
        newpath = self.config_dir.join('new.yaml')
        path = self.config_dir.join('%s.yaml' % self.config_hash)

        log.debug('Storing new configuration at %r', path.strpath)

        with open(path.strpath, 'wb') as f:
            yaml.dump(
                data,
                f,
                default_flow_style=False,
                Dumper=getattr(yaml, 'CSafeDumper', yaml.SafeDumper),
            )
            f.flush()
            os.fsync(f.fileno())

        if newpath.check(file=1):
            log.warning('Removing conflicting file at %r', newpath.strpath)
            newpath.remove()

        newpath.mksymlinkto(path.relto(newpath.dirpath()))
        newpath.move(actpath)

        # remove old config file(s)
        for p in self.config_dir.listdir():
            if p.ext == '.yaml' and p != actpath and p != path:
                log.debug('Removing previous configuration: %r', p.strpath)
                p.remove()

    def _save_config_deblocked(self, data, log=None):
        return self.deblock.apply(self._save_config, data, log=log)

    def _load_config(self, log=None):
        log = log or self.log

        file_path = self.config_dir.join(self.filename)

        # load data from the file
        log.debug('Loading config from %r', file_path)
        try:
            return yaml.load(file_path.open('rb'), Loader=getattr(yaml, 'CSafeLoader', yaml.SafeLoader))
        except Exception as ex:
            log.warning('Failed to load config from %r', file_path)
            log.warning(str(ex))

        return None

    def _load_config_deblocked(self, log=None):
        return self.deblock.apply(self._load_config, log=log)

    def _apply_overrides(self, cfg, log=None):
        log = log or self.log

        for path, ts in self.overrides:
            log.debug('Loading config override from %r', path)
            try:
                data = yaml.load(open(path, 'rb'), Loader=getattr(yaml, 'CSafeLoader', yaml.SafeLoader))
            except Exception:
                log.exception('Failed to load config from %r' % (path,))
                data = None

            if not data or not isinstance(data, dict):
                log.debug("No config data in file %r", path)
                continue

            log.debug("Applying config override from %r", path)
            cfg = self._merge_config(cfg, data, ts)

        return cfg

    def _apply_overrides_deblocked(self, cfg, log=None):
        return self.deblock.apply(self._apply_overrides, cfg, log=log)

    def _merge_config(self, cfg, data, ts):
        if not isinstance(data, dict):
            return data
        elif not isinstance(cfg, dict):
            return copy.deepcopy(data)

        out = copy.deepcopy(cfg)
        for k, v in six.iteritems(data):
            baseval = out.get(k)
            if isinstance(baseval, dict):
                out[k] = self._merge_config(baseval, v, ts)
                # update config_hash with override time stamp
                if k == 'config' and out.get('config_hash', False):
                    out['config_hash'] += '_{}'.format(ts)
            else:
                out[k] = copy.deepcopy(v)

        return out

    def _apply_config(self, data, log=None):
        log = log or self.log

        log.debug('Applying new config...')
        results = []
        self._iterate_subsections(None, data, results)
        if results:
            try:
                log.debug('Waiting for %d subscribers ...' % len(results))
                gevent.joinall(results, timeout=self.config.get('apply_timeout', CONFIG_APPLY_TIMEOUT))
                for res in results:
                    res.get()  # trigger an exception if any occurred
                log.debug('New config applied')
            except gevent.Timeout:
                log.critical('Failed to apply new config - timed out')
                raise Exception("failed to apply new config - timed out")
        else:
            log.debug('There are no subscribers for these changes')

    def _iterate_subsections(self, parent_section_name, data, results, log=None):
        log = log or self.log

        if 'subsections' not in data or not data['subsections']:
            return

        for key, value in data['subsections'].iteritems():
            section_name = (parent_section_name + '.' if parent_section_name else '') + key
            subscriptions = self.subscriptions.get(section_name, None)
            if subscriptions:
                log.debug('Found subscriptions for %s' % section_name)
                config_hash = value.get('config_hash', None)
                if config_hash:
                    for s in subscriptions:
                        res = s.notify(section_name, config_hash)
                        if res is not None:
                            log.debug('Applying config changes in %s' % section_name)
                            results.append(res)

            self._iterate_subsections(section_name, value, results, log)

    def _update_own_config(self, log=None):
        log = log or self.log
        try:
            skycore_config = self.data['subsections']['skynet']['subsections']['skycore']['subsections']['config']
            self.config = skycore_config['config']['config_updater']
        except Exception:
            # missing config -> use hard-coded default
            log.warning('Failed to find own config, use default values')
            self.config = {
                'update_period': CONFIG_UPDATE_PERIOD,
                'apply_timeout': CONFIG_APPLY_TIMEOUT,
                'status_resend_period': STATUS_RESEND_PERIOD
            }

    def _load_hostname(self, log):
        hostname_file = self.config_dir.join('config.hostname')
        if not hostname_file.check(exists=1):
            return None

        try:
            return hostname_file.open('r').read().strip()
        except Exception:
            log.exception("Failed to read hostname file %r", hostname_file.strpath)

    def _load_hostname_deblocked(self, log):
        return self.deblock.apply(self._load_hostname, log)

    def _check_overrides(self, log):
        log = log or self.log
        overrides = self.config_dir.join('overrides.d')
        if not overrides.check(exists=1, dir=1):
            if self.overrides:
                self.overrides = []
                log.info('All overrides have been removed')
            return False

        changed = False
        new_overrides = [
            (path.strpath, os.stat(path.strpath).st_mtime)
            for path in sorted(overrides.listdir())
            if path.ext == '.yaml'
        ]

        if len(new_overrides) != len(self.overrides):
            changed = True
        else:
            for ov1, ov2 in zip(self.overrides, new_overrides):
                if ov1[0] != ov2[0] or ov1[1] != ov2[1]:
                    changed = True
                    break

        if changed:
            log.info('Overrides have been changed')
            self.overrides = new_overrides

        return changed

    def __str__(self):
        return "%s (%s, config_hash %s, last modified %s)" % (
            self.__class__.__name__,
            "paused" if self.paused else "active",
            self.config_hash,
            self.last_modified,
        )
